/*
 * Copyright (C) 2014-2025 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#pragma once

#include <QDialog>

class AcntId; /* Forward declaration. */
class MessageDbSet; /* Forward declaration. */
class QString; /* Forward declaration. */

namespace Ui {
	class DlgChangeDirectory;
}

/*!
 * @brief Provides interface for database file location change.
 */
class DlgChangeDirectory : public QDialog {
	Q_OBJECT

private:
	/*!
	 * @brief Constructor.
	 *
	 * @param[in] acntId Account identifier (current database location directory).
	 * @param[in] parent Parent widget.
	 */
	explicit DlgChangeDirectory(const AcntId &acntId,
	    QWidget *parent = Q_NULLPTR);

public:
	/*!
	 * @brief Destructor.
	 */
	virtual
	~DlgChangeDirectory(void);

	/*!
	 * @brief Asks the user for the new database location and relocates it.
	 *
	 * @param[in] acntId Account identifier.
	 * @param[in] dbSet Affected database set.
	 * @param[in] parent Parent widget.
	 * @return True on success.
	 */
	static
	bool changeDataDirectory(const AcntId &acntId, MessageDbSet *dbSet,
	    QWidget *parent = Q_NULLPTR);

private Q_SLOTS:
	/*!
	 * @brief Use default target directory.
	 */
	void useDefaultDirectory(void);

	/*!
	 * @brief Choose new data directory.
	 */
	void chooseNewDirectory(void);

	/*!
	 * @brief Generate a warning dialogue to confirm deletion of current
	 *     data if the connected radio button checked.
	 *
	 * @param[in] checked Check status of the connected radio button.
	 */
	void eraseSrcWarning(bool checked);

	/*
	 * @brief Update window content according to input field content.
	 */
	void updateContent(void);

private:
	/*!
	 * @brief Identifies chosen action.
	 *
	 * @note The actions consist of imaginary components:
	 *     OPEN_TGT -- 0x00
	 *     ERASE_TGT -- 0x01
	 *     COPYTO_SRC_TO_TGT -- 0x02
	 *     ERASE_SRC -- 0x04
	 *
	 *     Standalone 2 and 6 don't make sense.
	 *
	 *     7 - ACT_MOVE = ERASE_TGT | COPYTO_SRC_TO_TGT | ERASE_SRC | OPEN_TGT
	 *     3 - ACT_COPY = ERASE_TGT | COPYTO_SRC_TO_TGT | OPEN_TGT
	 *     5 - ACT_NEW_ERASE_OLD = ERASE_TGT | ERASE_SRC | OPEN_TGT
	 *     1 - ACT_NEW_LEAVE_OLD = ERASE_TGT | OPEN_TGT
	 *     4 - ACT_USE_ERASE_OLD = ERASE_SRC | OPEN_TGT
	 *     0 - ACT_USE_LEAVE_OLD = OPEN_TGT
	 */
	enum Action {
		ACT_NONE, /*!< Convenience value. */
		ACT_MOVE, /*!< Move database location, delete in original location. */
		ACT_COPY, /*!< Copy database to new location, leave in original location. */
		ACT_NEW_ERASE_OLD, /*!< Start new database in new location, delete in original location. */
		ACT_NEW_LEAVE_OLD, /*!< Start new database in new location, leave in original location. */
		ACT_USE_ERASE_OLD, /*!< Use existing database in new location, delete in original location. */
		ACT_USE_LEAVE_OLD /*!< Use existing database in new location, leave in original location. */
	};

	/*!
	 * @brief Generates a dialogue asking the user for new location.
	 *
	 * @param[in] acntId Account identifier.
	 * @param[out] newDir Chosen new directory.
	 * @param[out] action Chosen relocation action.
	 * @param[in] parent Parent widget.
	 * @return True if a new directory and relocation action has been
	 *     selected.
	 */
	static
	bool chooseAction(const AcntId &acntId, QString &newDir,
	    enum Action &action, QWidget *parent = Q_NULLPTR);

	/*!
	 * @brief Performs a database relocation.
	 *
	 * @param[in] acntId Account identifier.
	 * @param[in] dbSet Affected database set.
	 * @param[in] oldDir Old location.
	 * @param[in] newDir New location.
	 * @param[in] action Type or relocation operation.
	 * @param[in] parent Parent widget.
	 * @return True on success.
	 */
	static
	bool relocateDatabase(const AcntId &acntId, MessageDbSet *dbSet,
	    const QString &oldDir, const QString &newDir, enum Action action,
	    QWidget *parent = Q_NULLPTR);

	Ui::DlgChangeDirectory *m_ui; /*!< UI generated from UI file. */

	const AcntId &m_acntId; /*!< Current account identifier. */
};
