/*
libspecbleach - A spectral processing library

Copyright 2022 Luciano Dato <lucianodato@gmail.com>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "denoise_mixer.h"
#include "../post_estimation/spectral_whitening.h"
#include <stdlib.h>
#include <string.h>

struct DenoiseMixer {
  float* residual_spectrum;
  float* denoised_spectrum;

  uint32_t fft_size;
  uint32_t real_spectrum_size;
  uint32_t sample_rate;
  uint32_t hop;
};

DenoiseMixer* denoise_mixer_initialize(uint32_t fft_size, uint32_t sample_rate,
                                       uint32_t hop) {
  DenoiseMixer* self = (DenoiseMixer*)calloc(1U, sizeof(DenoiseMixer));
  if (!self) {
    return NULL;
  }

  self->fft_size = fft_size;
  self->real_spectrum_size = self->fft_size / 2U + 1U;
  self->sample_rate = sample_rate;
  self->hop = hop;

  self->residual_spectrum = (float*)calloc((self->fft_size), sizeof(float));
  self->denoised_spectrum = (float*)calloc((self->fft_size), sizeof(float));

  if (!self->residual_spectrum || !self->denoised_spectrum) {
    denoise_mixer_free(self);
    return NULL;
  }

  return self;
}

void denoise_mixer_free(DenoiseMixer* self) {
  free(self->residual_spectrum);
  free(self->denoised_spectrum);

  free(self);
}

bool denoise_mixer_run(DenoiseMixer* self, float* fft_spectrum,
                       const float* gain_spectrum,
                       DenoiseMixerParameters parameters) {

  if (!fft_spectrum || !gain_spectrum) {
    return false;
  }

  // Get denoised spectrum - Apply to both real and complex parts
  for (uint32_t k = 0U; k < self->fft_size; k++) {
    self->denoised_spectrum[k] = fft_spectrum[k] * gain_spectrum[k];
  }

  // Get residual spectrum - Apply to both real and complex parts
  for (uint32_t k = 0U; k < self->fft_size; k++) {
    self->residual_spectrum[k] = fft_spectrum[k] - self->denoised_spectrum[k];
  }

  // Mix denoised and residual - Now a simple toggle
  if (parameters.residual_listen) {
    for (uint32_t k = 0U; k < self->fft_size; k++) {
      fft_spectrum[k] = self->residual_spectrum[k];
    }
  } else {
    for (uint32_t k = 0U; k < self->fft_size; k++) {
      fft_spectrum[k] = self->denoised_spectrum[k];
    }
  }

  return true;
}
