// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.attn_fwd.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_attn_fwd__A0__F525

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"PERSISTENT_TYPE": 0, "GRID_CU_MULTIP": 2, "BLOCK_M": 16, "BLOCK_N": 16, "PRE_LOAD_V": false})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 0, "GRID_CU_MULTIP": 2, "BLOCK_M": 16, "BLOCK_N": 16, "PRE_LOAD_V": false})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 0, "GRID_CU_MULTIP": 2, "BLOCK_M": 16, "BLOCK_N": 16, "PRE_LOAD_V": false})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 0, "GRID_CU_MULTIP": 2, "BLOCK_M": 16, "BLOCK_N": 16, "PRE_LOAD_V": false})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 0, "GRID_CU_MULTIP": 2, "BLOCK_M": 32, "BLOCK_N": 32, "PRE_LOAD_V": false})xyzw",
  R"xyzw({"PERSISTENT_TYPE": 0, "GRID_CU_MULTIP": 2, "BLOCK_M": 64, "BLOCK_N": 64, "PRE_LOAD_V": false})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
    int8_t  PERSISTENT_TYPE;
    int8_t  GRID_CU_MULTIP;
    bool    PRE_LOAD_V;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16, .PERSISTENT_TYPE = 0, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = false },
    { .BLOCK_M = 16, .BLOCK_N = 16, .PERSISTENT_TYPE = 0, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = false },
    { .BLOCK_M = 16, .BLOCK_N = 16, .PERSISTENT_TYPE = 0, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = false },
    { .BLOCK_M = 16, .BLOCK_N = 16, .PERSISTENT_TYPE = 0, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = false },
    { .BLOCK_M = 32, .BLOCK_N = 32, .PERSISTENT_TYPE = 0, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = false },
    { .BLOCK_M = 64, .BLOCK_N = 64, .PERSISTENT_TYPE = 0, .GRID_CU_MULTIP = 2, .PRE_LOAD_V = false }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/attn_fwd/FONLY__＊fp32@16_192_T_T_0_1___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/attn_fwd/FONLY__＊fp32@16_192_T_T_0_1___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp32@16_192_T_T_0_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x893bab41u, 0xdf705c5fu, 302, 83 }, // 893bab41df705c5f = b2sum -l 64 <<< amd-gfx950/flash/attn_fwd/FONLY__＊fp32@16_192_T_T_0_1___gfx950__P__0_2_16_16_False__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x2663d670u, 0xf4c1163au, 302, 16 }, // 2663d670f4c1163a = b2sum -l 64 <<< amd-gfx950/flash/attn_fwd/FONLY__＊fp32@16_192_T_T_0_1___gfx950__P__0_2_16_16_False__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xc3d4ecceu, 0x749d1b2du, 302, 116 }, // c3d4ecce749d1b2d = b2sum -l 64 <<< amd-gfx950/flash/attn_fwd/FONLY__＊fp32@16_192_T_T_0_1___gfx950__P__0_2_16_16_False__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x1efd292fu, 0x1488c70eu, 302, 167 }, // 1efd292f1488c70e = b2sum -l 64 <<< amd-gfx950/flash/attn_fwd/FONLY__＊fp32@16_192_T_T_0_1___gfx950__P__0_2_16_16_False__CO__wave3_warp4_stg1--Arch_gfx950
    { 0xb961f552u, 0x2ee73a33u, 318, 83 }, // b961f5522ee73a33 = b2sum -l 64 <<< amd-gfx950/flash/attn_fwd/FONLY__＊fp32@16_192_T_T_0_1___gfx950__P__0_2_32_32_False__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xf178bf99u, 0x424c5ca5u, 100, 202 }, // f178bf99424c5ca5 = b2sum -l 64 <<< amd-gfx950/flash/attn_fwd/FONLY__＊fp32@16_192_T_T_0_1___gfx950__P__0_2_64_64_False__CO__wave1_warp4_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::attn_fwd_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               attn_fwd_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx950_mod0
{{2,2,2,2,2,2,2,2,3,3},
 {2,2,2,2,2,2,2,2,3,3},
 {0,0,0,0,0,0,0,0,0,0},
 {1,1,1,1,1,1,4,0,5,5},
 {1,1,1,1,1,1,4,4,5,5},
 {1,1,1,1,1,4,4,4,5,5},
 {1,1,1,1,1,4,4,1,5,3},
 {1,1,1,1,1,1,4,1,0,0},
 {1,1,1,1,1,1,1,5,5,5},
 {1,1,1,1,1,1,4,5,5,5}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::AttnFwdContext;

void CURRENT_ENTRY_PUBLIC(AttnFwdContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_attn_fwd__A0__F525 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 10;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.PERSISTENT_TYPE = perf.PERSISTENT_TYPE;
    context.GRID_CU_MULTIP = perf.GRID_CU_MULTIP;
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
    context.PRE_LOAD_V = perf.PRE_LOAD_V;
        return ;
    }
#endif
    auto kernel_index = attn_fwd__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 10;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.PERSISTENT_TYPE = perf.PERSISTENT_TYPE;
    context.GRID_CU_MULTIP = perf.GRID_CU_MULTIP;
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
    context.PRE_LOAD_V = perf.PRE_LOAD_V;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp32:16"
// B = "*fp32:16"
// A = 0
// Sm_scale = "fp32"
// L = "*fp32:16"
// Q_descale = 0
// stride_az = 0
// Num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// BLOCK_DMODEL = 192
// Head_dim = "i32"
// PADDED_HEAD = True
// ENABLE_DROPOUT = True
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// RETURN_ENCODED_SOFTMAX = False
// encoded_softmax = 0
// CAUSAL_TYPE = 0
// Window_left = 0
// BIAS_TYPE = 1
// USE_ALIBI = False
// INT8 = False
// persistent_atomic_counter = 0
// Num_CU = "i32"

// SELECT * FROM FLASH$attn_fwd WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.float32' AND inputs$BLOCK_DMODEL = 192 AND inputs$PADDED_HEAD = False AND inputs$ENABLE_DROPOUT = True AND inputs$CAUSAL_TYPE = 0 AND inputs$BIAS_TYPE = 1

// vim: set fileencoding=utf-8

