//! Module defining all known symbols required by the rest of rust-analyzer.
#![allow(non_upper_case_globals)]

use std::hash::{BuildHasher, BuildHasherDefault};

use dashmap::{DashMap, SharedValue};
use rustc_hash::FxHasher;

use crate::{Symbol, symbol::TaggedArcPtr};

macro_rules! define_symbols {
    (@WITH_NAME: $($alias:ident = $value:literal,)* @PLAIN: $($name:ident,)*) => {
        // The strings should be in `static`s so that symbol equality holds.
        $(
            pub const $name: Symbol = {
                static SYMBOL_STR: &str = stringify!($name);
                Symbol { repr: TaggedArcPtr::non_arc(&SYMBOL_STR) }
            };
        )*
        $(
            pub const $alias: Symbol = {
                static SYMBOL_STR: &str = $value;
                Symbol { repr: TaggedArcPtr::non_arc(&SYMBOL_STR) }
            };
        )*


        pub(super) fn prefill() -> DashMap<Symbol, (), BuildHasherDefault<FxHasher>> {
            let mut dashmap_ = <DashMap<Symbol, (), BuildHasherDefault<FxHasher>>>::with_hasher(BuildHasherDefault::default());

            let hasher_ = dashmap_.hasher().clone();
            let hash_one = |it_: &str| hasher_.hash_one(it_);
            {
                $(
                    let s = stringify!($name);
                    let hash_ = hash_one(s);
                    let shard_idx_ = dashmap_.determine_shard(hash_ as usize);
                    dashmap_.shards_mut()[shard_idx_].get_mut().insert(hash_, ($name, SharedValue::new(())), |(x, _)| hash_one(x.as_str()));
                )*
                $(
                    let s = $value;
                    let hash_ = hash_one(s);
                    let shard_idx_ = dashmap_.determine_shard(hash_ as usize);
                    dashmap_.shards_mut()[shard_idx_].get_mut().insert(hash_, ($alias, SharedValue::new(())), |(x, _)| hash_one(x.as_str()));
                )*
            }
            dashmap_
        }
    };
}
define_symbols! {
    @WITH_NAME:

    dotdotdot = "...",
    INTEGER_0 = "0",
    INTEGER_1 = "1",
    INTEGER_2 = "2",
    INTEGER_3 = "3",
    INTEGER_4 = "4",
    INTEGER_5 = "5",
    INTEGER_6 = "6",
    INTEGER_7 = "7",
    INTEGER_8 = "8",
    INTEGER_9 = "9",
    INTEGER_10 = "10",
    INTEGER_11 = "11",
    INTEGER_12 = "12",
    INTEGER_13 = "13",
    INTEGER_14 = "14",
    INTEGER_15 = "15",
    __empty = "",
    unsafe_ = "unsafe",
    in_ = "in",
    super_ = "super",
    self_ = "self",
    Self_ = "Self",
    tick_static = "'static",
    tick_underscore = "'_",
    dollar_crate = "$crate",
    MISSING_NAME = "[missing name]",
    fn_ = "fn",
    crate_ = "crate",
    underscore = "_",
    true_ = "true",
    false_ = "false",
    let_ = "let",
    const_ = "const",
    kw_impl = "impl",
    proc_dash_macro = "proc-macro",
    aapcs_dash_unwind = "aapcs-unwind",
    avr_dash_interrupt = "avr-interrupt",
    avr_dash_non_dash_blocking_dash_interrupt = "avr-non-blocking-interrupt",
    C_dash_cmse_dash_nonsecure_dash_call = "C-cmse-nonsecure-call",
    C_dash_cmse_dash_nonsecure_dash_entry = "C-cmse-nonsecure-entry",
    C_dash_unwind = "C-unwind",
    cdecl_dash_unwind = "cdecl-unwind",
    fastcall_dash_unwind = "fastcall-unwind",
    msp430_dash_interrupt = "msp430-interrupt",
    ptx_dash_kernel = "ptx-kernel",
    riscv_dash_interrupt_dash_m = "riscv-interrupt-m",
    riscv_dash_interrupt_dash_s = "riscv-interrupt-s",
    rust_dash_call = "rust-call",
    rust_dash_cold = "rust-cold",
    rust_dash_intrinsic = "rust-intrinsic",
    stdcall_dash_unwind = "stdcall-unwind",
    system_dash_unwind = "system-unwind",
    sysv64_dash_unwind = "sysv64-unwind",
    thiscall_dash_unwind = "thiscall-unwind",
    vectorcall_dash_unwind = "vectorcall-unwind",
    win64_dash_unwind = "win64-unwind",
    x86_dash_interrupt = "x86-interrupt",

    @PLAIN:
    __ra_fixup,
    aapcs,
    add_assign,
    add,
    alias,
    align_offset,
    align,
    all,
    alloc_layout,
    alloc,
    allow_internal_unsafe,
    allow,
    any,
    as_str,
    asm,
    assert,
    attr,
    attributes,
    begin_panic,
    bench,
    bitand_assign,
    bitand,
    bitor_assign,
    bitor,
    bitxor_assign,
    bitxor,
    bool,
    bootstrap,
    box_free,
    Box,
    boxed,
    branch,
    Break,
    c_void,
    C,
    call_mut,
    call_once,
    async_call_once,
    async_call_mut,
    async_call,
    call,
    cdecl,
    Center,
    cfg_accessible,
    cfg_attr,
    cfg_eval,
    cfg,
    cfg_select,
    char,
    clone,
    trivial_clone,
    Clone,
    coerce_unsized,
    column,
    completion,
    compile_error,
    concat_bytes,
    concat,
    const_format_args,
    const_panic_fmt,
    const_param_ty,
    Context,
    Continue,
    convert,
    copy,
    Copy,
    core_panic,
    core,
    coroutine_state,
    coroutine,
    coroutine_return,
    coroutine_yield,
    count,
    crate_type,
    CStr,
    debug_assertions,
    Debug,
    default,
    Default,
    deprecated,
    deref_mut,
    deref_target,
    deref,
    derive_const,
    derive,
    discriminant_kind,
    discriminant_type,
    dispatch_from_dyn,destruct,
    div_assign,
    div,
    doc,
    drop_in_place,
    drop,
    dyn_metadata,
    efiapi,
    eh_catch_typeinfo,
    eh_personality,
    env,
    eq,
    Eq,
    Err,
    exchange_malloc,
    exhaustive_patterns,
    export_name,
    f128,
    f16,
    f32,
    f64,
    fastcall,
    feature,
    file,
    filter_map,
    fmt,
    fn_mut,
    fn_once_output,
    fn_once,
    async_fn_once,
    async_fn_once_output,
    async_fn_mut,
    async_fn,
    call_ref_future,
    call_once_future,
    fn_ptr_addr,
    fn_ptr_trait,
    format_alignment,
    format_args_nl,
    format_args,
    format_argument,
    format_arguments,
    format_count,
    format_placeholder,
    format_unsafe_arg,
    format,
    freeze,
    from,
    From,
    FromStr,
    from_str,
    from_output,
    from_residual,
    from_usize,
    from_yeet,
    fundamental,
    future_trait,
    future,
    future_output,
    Future,
    ge,
    generic_associated_type_extended,
    get_context,
    global_allocator,
    global_asm,
    gt,
    Hash,
    hidden,
    html_root_url,
    i128,
    i16,
    i32,
    i64,
    i8,
    ignore,
    Implied,
    include_bytes,
    include_str,
    include,
    index_mut,
    index,
    Index,
    into,
    Into,
    into_future,
    into_iter,
    IntoFuture,
    IntoIter,
    IntoIterator,
    is_empty,
    Is,
    isize,
    Item,
    iter_mut,
    iter,
    Iterator,
    iterator,
    keyword,
    lang,
    lang_items,
    le,
    Left,
    len,
    line,
    llvm_asm,
    local_inner_macros,
    log_syntax,
    lt,
    macro_export,
    macro_rules,
    macro_use,
    main,
    manually_drop,
    may_dangle,
    maybe_uninit,
    metadata_type,
    min_exhaustive_patterns,
    miri,
    missing,
    module_path,
    mul_assign,
    mul,
    naked_asm,
    ne,
    neg,
    Neg,
    new_binary,
    new_debug,
    new_display,
    new_lower_exp,
    new_lower_hex,
    new_octal,
    new_pointer,
    new_unchecked,
    new_upper_exp,
    new_upper_hex,
    new_v1_formatted,
    new,
    next,
    no_core,
    no_mangle,
    no_std,
    non_exhaustive,
    none,
    None,
    not,
    Not,
    notable_trait,
    Ok,
    opaque,
    ops,
    option_env,
    option,
    Option,
    Ord,
    Ordering,
    Output,
    CallRefFuture,
    CallOnceFuture,
    owned_box,
    packed,
    panic_2015,
    panic_2021,
    panic_bounds_check,
    panic_cannot_unwind,
    panic_display,
    panic_fmt,
    panic_impl,
    panic_info,
    panic_location,
    panic_misaligned_pointer_dereference,
    panic_nounwind,
    panic_null_pointer_dereference,
    panic,
    Param,
    parse,
    partial_ord,
    PartialEq,
    PartialOrd,
    CoercePointee,
    path,
    Pending,
    phantom_data,
    pieces,
    pin,
    pointee_trait,
    pointer_like,
    poll,
    Poll,
    prelude_import,
    prelude,
    proc_macro_attribute,
    proc_macro_derive,
    proc_macro,
    quote,
    range_inclusive_new,
    Range,
    RangeFrom,
    RangeFull,
    RangeInclusive,
    RangeTo,
    RangeToInclusive,
    Ready,
    receiver,
    receiver_target,
    recursion_limit,
    register_attr,
    register_tool,
    rem_assign,
    rem,
    repr,
    result,
    Result,
    ResumeTy,
    Right,
    rust_2015,
    rust_2018,
    rust_2021,
    rust_2024,
    rust_analyzer,
    Rust,
    rustc_allocator_zeroed,
    rustc_allocator,
    rustc_allow_incoherent_impl,
    rustc_builtin_macro,
    rustc_coherence_is_core,
    rustc_coinductive,
    rustc_const_panic_str,
    rustc_deallocator,
    rustc_deprecated_safe_2024,
    rustc_has_incoherent_inherent_impls,
    rustc_intrinsic_must_be_overridden,
    rustc_intrinsic,
    rustc_layout_scalar_valid_range_end,
    rustc_layout_scalar_valid_range_start,
    rustc_legacy_const_generics,
    rustc_macro_transparency,
    rustc_paren_sugar,
    rustc_reallocator,
    rustc_reservation_impl,
    rustc_safe_intrinsic,
    rustc_skip_array_during_method_dispatch,
    rustc_skip_during_method_dispatch,
    rustc_force_inline,
    semitransparent,
    shl_assign,
    shl,
    shr_assign,
    shr,
    simd,
    sized,
    meta_sized,
    pointee_sized,
    skip,
    slice_len_fn,
    Some,
    start,
    std_panic,
    std,
    stdcall,
    str,
    string,
    String,
    stringify,
    structural_peq,
    structural_teq,
    sub_assign,
    sub,
    sync,
    system,
    sysv64,
    Target,
    target_feature,
    enable,
    termination,
    test_case,
    test,
    then,
    thiscall,
    to_string,
    trace_macros,
    transmute_opts,
    transmute_trait,
    transparent,
    try_into,
    Try,
    TryFrom,
    try_from,
    tuple_trait,
    u128,
    u16,
    u32,
    u64,
    u8,
    unadjusted,
    unknown,
    Unknown,
    unpin,
    unreachable_2015,
    unreachable_2021,
    unreachable,
    unsafe_cell,
    unsafe_pinned,
    unsize,
    unstable,
    usize,
    v1,
    va_list,
    vectorcall,
    wasm,
    win64,
    args,
    array,
    boxed_slice,
    completions,
    ignore_flyimport,
    ignore_flyimport_methods,
    ignore_methods,
    position,
    flags,
    precision,
    width,
    never_type_fallback,
    specialization,
    min_specialization,
    arbitrary_self_types,
    arbitrary_self_types_pointers,
    supertrait_item_shadowing,
    hash,
    partial_cmp,
    cmp,
    CoerceUnsized,
    DispatchFromDyn,
    define_opaque,
    marker,
}
