﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/es/ElasticsearchServiceRequest.h>
#include <aws/es/ElasticsearchService_EXPORTS.h>
#include <aws/es/model/AdvancedSecurityOptionsInput.h>
#include <aws/es/model/AutoTuneOptions.h>
#include <aws/es/model/CognitoOptions.h>
#include <aws/es/model/DomainEndpointOptions.h>
#include <aws/es/model/EBSOptions.h>
#include <aws/es/model/ElasticsearchClusterConfig.h>
#include <aws/es/model/EncryptionAtRestOptions.h>
#include <aws/es/model/LogPublishingOption.h>
#include <aws/es/model/LogType.h>
#include <aws/es/model/NodeToNodeEncryptionOptions.h>
#include <aws/es/model/SnapshotOptions.h>
#include <aws/es/model/VPCOptions.h>

#include <utility>

namespace Aws {
namespace ElasticsearchService {
namespace Model {

/**
 * <p>Container for the parameters to the
 * <code><a>UpdateElasticsearchDomain</a></code> operation. Specifies the type and
 * number of instances in the domain cluster.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/es-2015-01-01/UpdateElasticsearchDomainConfigRequest">AWS
 * API Reference</a></p>
 */
class UpdateElasticsearchDomainConfigRequest : public ElasticsearchServiceRequest {
 public:
  AWS_ELASTICSEARCHSERVICE_API UpdateElasticsearchDomainConfigRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateElasticsearchDomainConfig"; }

  AWS_ELASTICSEARCHSERVICE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the Elasticsearch domain that you are updating. </p>
   */
  inline const Aws::String& GetDomainName() const { return m_domainName; }
  inline bool DomainNameHasBeenSet() const { return m_domainNameHasBeenSet; }
  template <typename DomainNameT = Aws::String>
  void SetDomainName(DomainNameT&& value) {
    m_domainNameHasBeenSet = true;
    m_domainName = std::forward<DomainNameT>(value);
  }
  template <typename DomainNameT = Aws::String>
  UpdateElasticsearchDomainConfigRequest& WithDomainName(DomainNameT&& value) {
    SetDomainName(std::forward<DomainNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type and number of instances to instantiate for the domain cluster.</p>
   */
  inline const ElasticsearchClusterConfig& GetElasticsearchClusterConfig() const { return m_elasticsearchClusterConfig; }
  inline bool ElasticsearchClusterConfigHasBeenSet() const { return m_elasticsearchClusterConfigHasBeenSet; }
  template <typename ElasticsearchClusterConfigT = ElasticsearchClusterConfig>
  void SetElasticsearchClusterConfig(ElasticsearchClusterConfigT&& value) {
    m_elasticsearchClusterConfigHasBeenSet = true;
    m_elasticsearchClusterConfig = std::forward<ElasticsearchClusterConfigT>(value);
  }
  template <typename ElasticsearchClusterConfigT = ElasticsearchClusterConfig>
  UpdateElasticsearchDomainConfigRequest& WithElasticsearchClusterConfig(ElasticsearchClusterConfigT&& value) {
    SetElasticsearchClusterConfig(std::forward<ElasticsearchClusterConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify the type and size of the EBS volume that you want to use. </p>
   */
  inline const EBSOptions& GetEBSOptions() const { return m_eBSOptions; }
  inline bool EBSOptionsHasBeenSet() const { return m_eBSOptionsHasBeenSet; }
  template <typename EBSOptionsT = EBSOptions>
  void SetEBSOptions(EBSOptionsT&& value) {
    m_eBSOptionsHasBeenSet = true;
    m_eBSOptions = std::forward<EBSOptionsT>(value);
  }
  template <typename EBSOptionsT = EBSOptions>
  UpdateElasticsearchDomainConfigRequest& WithEBSOptions(EBSOptionsT&& value) {
    SetEBSOptions(std::forward<EBSOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Option to set the time, in UTC format, for the daily automated snapshot.
   * Default value is <code>0</code> hours. </p>
   */
  inline const SnapshotOptions& GetSnapshotOptions() const { return m_snapshotOptions; }
  inline bool SnapshotOptionsHasBeenSet() const { return m_snapshotOptionsHasBeenSet; }
  template <typename SnapshotOptionsT = SnapshotOptions>
  void SetSnapshotOptions(SnapshotOptionsT&& value) {
    m_snapshotOptionsHasBeenSet = true;
    m_snapshotOptions = std::forward<SnapshotOptionsT>(value);
  }
  template <typename SnapshotOptionsT = SnapshotOptions>
  UpdateElasticsearchDomainConfigRequest& WithSnapshotOptions(SnapshotOptionsT&& value) {
    SetSnapshotOptions(std::forward<SnapshotOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Options to specify the subnets and security groups for VPC endpoint. For more
   * information, see <a
   * href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-vpc.html#es-creating-vpc"
   * target="_blank">Creating a VPC</a> in <i>VPC Endpoints for Amazon Elasticsearch
   * Service Domains</i></p>
   */
  inline const VPCOptions& GetVPCOptions() const { return m_vPCOptions; }
  inline bool VPCOptionsHasBeenSet() const { return m_vPCOptionsHasBeenSet; }
  template <typename VPCOptionsT = VPCOptions>
  void SetVPCOptions(VPCOptionsT&& value) {
    m_vPCOptionsHasBeenSet = true;
    m_vPCOptions = std::forward<VPCOptionsT>(value);
  }
  template <typename VPCOptionsT = VPCOptions>
  UpdateElasticsearchDomainConfigRequest& WithVPCOptions(VPCOptionsT&& value) {
    SetVPCOptions(std::forward<VPCOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Options to specify the Cognito user and identity pools for Kibana
   * authentication. For more information, see <a
   * href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-cognito-auth.html"
   * target="_blank">Amazon Cognito Authentication for Kibana</a>.</p>
   */
  inline const CognitoOptions& GetCognitoOptions() const { return m_cognitoOptions; }
  inline bool CognitoOptionsHasBeenSet() const { return m_cognitoOptionsHasBeenSet; }
  template <typename CognitoOptionsT = CognitoOptions>
  void SetCognitoOptions(CognitoOptionsT&& value) {
    m_cognitoOptionsHasBeenSet = true;
    m_cognitoOptions = std::forward<CognitoOptionsT>(value);
  }
  template <typename CognitoOptionsT = CognitoOptions>
  UpdateElasticsearchDomainConfigRequest& WithCognitoOptions(CognitoOptionsT&& value) {
    SetCognitoOptions(std::forward<CognitoOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Modifies the advanced option to allow references to indices in an HTTP
   * request body. Must be <code>false</code> when configuring access to individual
   * sub-resources. By default, the value is <code>true</code>. See <a
   * href="http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomain-configure-advanced-options"
   * target="_blank">Configuration Advanced Options</a> for more information.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetAdvancedOptions() const { return m_advancedOptions; }
  inline bool AdvancedOptionsHasBeenSet() const { return m_advancedOptionsHasBeenSet; }
  template <typename AdvancedOptionsT = Aws::Map<Aws::String, Aws::String>>
  void SetAdvancedOptions(AdvancedOptionsT&& value) {
    m_advancedOptionsHasBeenSet = true;
    m_advancedOptions = std::forward<AdvancedOptionsT>(value);
  }
  template <typename AdvancedOptionsT = Aws::Map<Aws::String, Aws::String>>
  UpdateElasticsearchDomainConfigRequest& WithAdvancedOptions(AdvancedOptionsT&& value) {
    SetAdvancedOptions(std::forward<AdvancedOptionsT>(value));
    return *this;
  }
  template <typename AdvancedOptionsKeyT = Aws::String, typename AdvancedOptionsValueT = Aws::String>
  UpdateElasticsearchDomainConfigRequest& AddAdvancedOptions(AdvancedOptionsKeyT&& key, AdvancedOptionsValueT&& value) {
    m_advancedOptionsHasBeenSet = true;
    m_advancedOptions.emplace(std::forward<AdvancedOptionsKeyT>(key), std::forward<AdvancedOptionsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>IAM access policy as a JSON-formatted string.</p>
   */
  inline const Aws::String& GetAccessPolicies() const { return m_accessPolicies; }
  inline bool AccessPoliciesHasBeenSet() const { return m_accessPoliciesHasBeenSet; }
  template <typename AccessPoliciesT = Aws::String>
  void SetAccessPolicies(AccessPoliciesT&& value) {
    m_accessPoliciesHasBeenSet = true;
    m_accessPolicies = std::forward<AccessPoliciesT>(value);
  }
  template <typename AccessPoliciesT = Aws::String>
  UpdateElasticsearchDomainConfigRequest& WithAccessPolicies(AccessPoliciesT&& value) {
    SetAccessPolicies(std::forward<AccessPoliciesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Map of <code>LogType</code> and <code>LogPublishingOption</code>, each
   * containing options to publish a given type of Elasticsearch log.</p>
   */
  inline const Aws::Map<LogType, LogPublishingOption>& GetLogPublishingOptions() const { return m_logPublishingOptions; }
  inline bool LogPublishingOptionsHasBeenSet() const { return m_logPublishingOptionsHasBeenSet; }
  template <typename LogPublishingOptionsT = Aws::Map<LogType, LogPublishingOption>>
  void SetLogPublishingOptions(LogPublishingOptionsT&& value) {
    m_logPublishingOptionsHasBeenSet = true;
    m_logPublishingOptions = std::forward<LogPublishingOptionsT>(value);
  }
  template <typename LogPublishingOptionsT = Aws::Map<LogType, LogPublishingOption>>
  UpdateElasticsearchDomainConfigRequest& WithLogPublishingOptions(LogPublishingOptionsT&& value) {
    SetLogPublishingOptions(std::forward<LogPublishingOptionsT>(value));
    return *this;
  }
  inline UpdateElasticsearchDomainConfigRequest& AddLogPublishingOptions(LogType key, LogPublishingOption value) {
    m_logPublishingOptionsHasBeenSet = true;
    m_logPublishingOptions.emplace(key, value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Options to specify configuration that will be applied to the domain
   * endpoint.</p>
   */
  inline const DomainEndpointOptions& GetDomainEndpointOptions() const { return m_domainEndpointOptions; }
  inline bool DomainEndpointOptionsHasBeenSet() const { return m_domainEndpointOptionsHasBeenSet; }
  template <typename DomainEndpointOptionsT = DomainEndpointOptions>
  void SetDomainEndpointOptions(DomainEndpointOptionsT&& value) {
    m_domainEndpointOptionsHasBeenSet = true;
    m_domainEndpointOptions = std::forward<DomainEndpointOptionsT>(value);
  }
  template <typename DomainEndpointOptionsT = DomainEndpointOptions>
  UpdateElasticsearchDomainConfigRequest& WithDomainEndpointOptions(DomainEndpointOptionsT&& value) {
    SetDomainEndpointOptions(std::forward<DomainEndpointOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies advanced security options.</p>
   */
  inline const AdvancedSecurityOptionsInput& GetAdvancedSecurityOptions() const { return m_advancedSecurityOptions; }
  inline bool AdvancedSecurityOptionsHasBeenSet() const { return m_advancedSecurityOptionsHasBeenSet; }
  template <typename AdvancedSecurityOptionsT = AdvancedSecurityOptionsInput>
  void SetAdvancedSecurityOptions(AdvancedSecurityOptionsT&& value) {
    m_advancedSecurityOptionsHasBeenSet = true;
    m_advancedSecurityOptions = std::forward<AdvancedSecurityOptionsT>(value);
  }
  template <typename AdvancedSecurityOptionsT = AdvancedSecurityOptionsInput>
  UpdateElasticsearchDomainConfigRequest& WithAdvancedSecurityOptions(AdvancedSecurityOptionsT&& value) {
    SetAdvancedSecurityOptions(std::forward<AdvancedSecurityOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the NodeToNodeEncryptionOptions.</p>
   */
  inline const NodeToNodeEncryptionOptions& GetNodeToNodeEncryptionOptions() const { return m_nodeToNodeEncryptionOptions; }
  inline bool NodeToNodeEncryptionOptionsHasBeenSet() const { return m_nodeToNodeEncryptionOptionsHasBeenSet; }
  template <typename NodeToNodeEncryptionOptionsT = NodeToNodeEncryptionOptions>
  void SetNodeToNodeEncryptionOptions(NodeToNodeEncryptionOptionsT&& value) {
    m_nodeToNodeEncryptionOptionsHasBeenSet = true;
    m_nodeToNodeEncryptionOptions = std::forward<NodeToNodeEncryptionOptionsT>(value);
  }
  template <typename NodeToNodeEncryptionOptionsT = NodeToNodeEncryptionOptions>
  UpdateElasticsearchDomainConfigRequest& WithNodeToNodeEncryptionOptions(NodeToNodeEncryptionOptionsT&& value) {
    SetNodeToNodeEncryptionOptions(std::forward<NodeToNodeEncryptionOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the Encryption At Rest Options.</p>
   */
  inline const EncryptionAtRestOptions& GetEncryptionAtRestOptions() const { return m_encryptionAtRestOptions; }
  inline bool EncryptionAtRestOptionsHasBeenSet() const { return m_encryptionAtRestOptionsHasBeenSet; }
  template <typename EncryptionAtRestOptionsT = EncryptionAtRestOptions>
  void SetEncryptionAtRestOptions(EncryptionAtRestOptionsT&& value) {
    m_encryptionAtRestOptionsHasBeenSet = true;
    m_encryptionAtRestOptions = std::forward<EncryptionAtRestOptionsT>(value);
  }
  template <typename EncryptionAtRestOptionsT = EncryptionAtRestOptions>
  UpdateElasticsearchDomainConfigRequest& WithEncryptionAtRestOptions(EncryptionAtRestOptionsT&& value) {
    SetEncryptionAtRestOptions(std::forward<EncryptionAtRestOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies Auto-Tune options.</p>
   */
  inline const AutoTuneOptions& GetAutoTuneOptions() const { return m_autoTuneOptions; }
  inline bool AutoTuneOptionsHasBeenSet() const { return m_autoTuneOptionsHasBeenSet; }
  template <typename AutoTuneOptionsT = AutoTuneOptions>
  void SetAutoTuneOptions(AutoTuneOptionsT&& value) {
    m_autoTuneOptionsHasBeenSet = true;
    m_autoTuneOptions = std::forward<AutoTuneOptionsT>(value);
  }
  template <typename AutoTuneOptionsT = AutoTuneOptions>
  UpdateElasticsearchDomainConfigRequest& WithAutoTuneOptions(AutoTuneOptionsT&& value) {
    SetAutoTuneOptions(std::forward<AutoTuneOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> This flag, when set to True, specifies whether the
   * <code>UpdateElasticsearchDomain</code> request should return the results of
   * validation checks without actually applying the change. This flag, when set to
   * True, specifies the deployment mechanism through which the update shall be
   * applied on the domain. This will not actually perform the Update. </p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline UpdateElasticsearchDomainConfigRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_domainName;
  bool m_domainNameHasBeenSet = false;

  ElasticsearchClusterConfig m_elasticsearchClusterConfig;
  bool m_elasticsearchClusterConfigHasBeenSet = false;

  EBSOptions m_eBSOptions;
  bool m_eBSOptionsHasBeenSet = false;

  SnapshotOptions m_snapshotOptions;
  bool m_snapshotOptionsHasBeenSet = false;

  VPCOptions m_vPCOptions;
  bool m_vPCOptionsHasBeenSet = false;

  CognitoOptions m_cognitoOptions;
  bool m_cognitoOptionsHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_advancedOptions;
  bool m_advancedOptionsHasBeenSet = false;

  Aws::String m_accessPolicies;
  bool m_accessPoliciesHasBeenSet = false;

  Aws::Map<LogType, LogPublishingOption> m_logPublishingOptions;
  bool m_logPublishingOptionsHasBeenSet = false;

  DomainEndpointOptions m_domainEndpointOptions;
  bool m_domainEndpointOptionsHasBeenSet = false;

  AdvancedSecurityOptionsInput m_advancedSecurityOptions;
  bool m_advancedSecurityOptionsHasBeenSet = false;

  NodeToNodeEncryptionOptions m_nodeToNodeEncryptionOptions;
  bool m_nodeToNodeEncryptionOptionsHasBeenSet = false;

  EncryptionAtRestOptions m_encryptionAtRestOptions;
  bool m_encryptionAtRestOptionsHasBeenSet = false;

  AutoTuneOptions m_autoTuneOptions;
  bool m_autoTuneOptionsHasBeenSet = false;

  bool m_dryRun{false};
  bool m_dryRunHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticsearchService
}  // namespace Aws
