﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/dax/DAX_EXPORTS.h>
#include <aws/dax/model/SourceType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace DAX {
namespace Model {

/**
 * <p>Represents a single occurrence of something interesting within the system.
 * Some examples of events are creating a DAX cluster, adding or removing a node,
 * or rebooting a node.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/Event">AWS API
 * Reference</a></p>
 */
class Event {
 public:
  AWS_DAX_API Event() = default;
  AWS_DAX_API Event(Aws::Utils::Json::JsonView jsonValue);
  AWS_DAX_API Event& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_DAX_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The source of the event. For example, if the event occurred at the node
   * level, the source would be the node ID.</p>
   */
  inline const Aws::String& GetSourceName() const { return m_sourceName; }
  inline bool SourceNameHasBeenSet() const { return m_sourceNameHasBeenSet; }
  template <typename SourceNameT = Aws::String>
  void SetSourceName(SourceNameT&& value) {
    m_sourceNameHasBeenSet = true;
    m_sourceName = std::forward<SourceNameT>(value);
  }
  template <typename SourceNameT = Aws::String>
  Event& WithSourceName(SourceNameT&& value) {
    SetSourceName(std::forward<SourceNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the origin of this event - a cluster, a parameter group, a node ID,
   * etc.</p>
   */
  inline SourceType GetSourceType() const { return m_sourceType; }
  inline bool SourceTypeHasBeenSet() const { return m_sourceTypeHasBeenSet; }
  inline void SetSourceType(SourceType value) {
    m_sourceTypeHasBeenSet = true;
    m_sourceType = value;
  }
  inline Event& WithSourceType(SourceType value) {
    SetSourceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A user-defined message associated with the event.</p>
   */
  inline const Aws::String& GetMessage() const { return m_message; }
  inline bool MessageHasBeenSet() const { return m_messageHasBeenSet; }
  template <typename MessageT = Aws::String>
  void SetMessage(MessageT&& value) {
    m_messageHasBeenSet = true;
    m_message = std::forward<MessageT>(value);
  }
  template <typename MessageT = Aws::String>
  Event& WithMessage(MessageT&& value) {
    SetMessage(std::forward<MessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time when the event occurred.</p>
   */
  inline const Aws::Utils::DateTime& GetDate() const { return m_date; }
  inline bool DateHasBeenSet() const { return m_dateHasBeenSet; }
  template <typename DateT = Aws::Utils::DateTime>
  void SetDate(DateT&& value) {
    m_dateHasBeenSet = true;
    m_date = std::forward<DateT>(value);
  }
  template <typename DateT = Aws::Utils::DateTime>
  Event& WithDate(DateT&& value) {
    SetDate(std::forward<DateT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_sourceName;

  SourceType m_sourceType{SourceType::NOT_SET};

  Aws::String m_message;

  Aws::Utils::DateTime m_date{};
  bool m_sourceNameHasBeenSet = false;
  bool m_sourceTypeHasBeenSet = false;
  bool m_messageHasBeenSet = false;
  bool m_dateHasBeenSet = false;
};

}  // namespace Model
}  // namespace DAX
}  // namespace Aws
