﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/omics/OmicsRequest.h>
#include <aws/omics/Omics_EXPORTS.h>
#include <aws/omics/model/StartReadSetImportJobSourceItem.h>

#include <utility>

namespace Aws {
namespace Omics {
namespace Model {

/**
 */
class StartReadSetImportJobRequest : public OmicsRequest {
 public:
  AWS_OMICS_API StartReadSetImportJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartReadSetImportJob"; }

  AWS_OMICS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The read set's sequence store ID.</p>
   */
  inline const Aws::String& GetSequenceStoreId() const { return m_sequenceStoreId; }
  inline bool SequenceStoreIdHasBeenSet() const { return m_sequenceStoreIdHasBeenSet; }
  template <typename SequenceStoreIdT = Aws::String>
  void SetSequenceStoreId(SequenceStoreIdT&& value) {
    m_sequenceStoreIdHasBeenSet = true;
    m_sequenceStoreId = std::forward<SequenceStoreIdT>(value);
  }
  template <typename SequenceStoreIdT = Aws::String>
  StartReadSetImportJobRequest& WithSequenceStoreId(SequenceStoreIdT&& value) {
    SetSequenceStoreId(std::forward<SequenceStoreIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A service role for the job.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  StartReadSetImportJobRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>To ensure that jobs don't run multiple times, specify a unique token for each
   * job.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  StartReadSetImportJobRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The job's source files.</p>
   */
  inline const Aws::Vector<StartReadSetImportJobSourceItem>& GetSources() const { return m_sources; }
  inline bool SourcesHasBeenSet() const { return m_sourcesHasBeenSet; }
  template <typename SourcesT = Aws::Vector<StartReadSetImportJobSourceItem>>
  void SetSources(SourcesT&& value) {
    m_sourcesHasBeenSet = true;
    m_sources = std::forward<SourcesT>(value);
  }
  template <typename SourcesT = Aws::Vector<StartReadSetImportJobSourceItem>>
  StartReadSetImportJobRequest& WithSources(SourcesT&& value) {
    SetSources(std::forward<SourcesT>(value));
    return *this;
  }
  template <typename SourcesT = StartReadSetImportJobSourceItem>
  StartReadSetImportJobRequest& AddSources(SourcesT&& value) {
    m_sourcesHasBeenSet = true;
    m_sources.emplace_back(std::forward<SourcesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_sequenceStoreId;

  Aws::String m_roleArn;

  Aws::String m_clientToken;

  Aws::Vector<StartReadSetImportJobSourceItem> m_sources;
  bool m_sequenceStoreIdHasBeenSet = false;
  bool m_roleArnHasBeenSet = false;
  bool m_clientTokenHasBeenSet = false;
  bool m_sourcesHasBeenSet = false;
};

}  // namespace Model
}  // namespace Omics
}  // namespace Aws
