﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticloadbalancingv2/ElasticLoadBalancingv2Request.h>
#include <aws/elasticloadbalancingv2/ElasticLoadBalancingv2_EXPORTS.h>
#include <aws/elasticloadbalancingv2/model/TargetDescription.h>

#include <utility>

namespace Aws {
namespace ElasticLoadBalancingv2 {
namespace Model {

/**
 */
class DeregisterTargetsRequest : public ElasticLoadBalancingv2Request {
 public:
  AWS_ELASTICLOADBALANCINGV2_API DeregisterTargetsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeregisterTargets"; }

  AWS_ELASTICLOADBALANCINGV2_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICLOADBALANCINGV2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the target group.</p>
   */
  inline const Aws::String& GetTargetGroupArn() const { return m_targetGroupArn; }
  inline bool TargetGroupArnHasBeenSet() const { return m_targetGroupArnHasBeenSet; }
  template <typename TargetGroupArnT = Aws::String>
  void SetTargetGroupArn(TargetGroupArnT&& value) {
    m_targetGroupArnHasBeenSet = true;
    m_targetGroupArn = std::forward<TargetGroupArnT>(value);
  }
  template <typename TargetGroupArnT = Aws::String>
  DeregisterTargetsRequest& WithTargetGroupArn(TargetGroupArnT&& value) {
    SetTargetGroupArn(std::forward<TargetGroupArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The targets. If you specified a port override when you registered a target,
   * you must specify both the target ID and the port when you deregister it.</p>
   */
  inline const Aws::Vector<TargetDescription>& GetTargets() const { return m_targets; }
  inline bool TargetsHasBeenSet() const { return m_targetsHasBeenSet; }
  template <typename TargetsT = Aws::Vector<TargetDescription>>
  void SetTargets(TargetsT&& value) {
    m_targetsHasBeenSet = true;
    m_targets = std::forward<TargetsT>(value);
  }
  template <typename TargetsT = Aws::Vector<TargetDescription>>
  DeregisterTargetsRequest& WithTargets(TargetsT&& value) {
    SetTargets(std::forward<TargetsT>(value));
    return *this;
  }
  template <typename TargetsT = TargetDescription>
  DeregisterTargetsRequest& AddTargets(TargetsT&& value) {
    m_targetsHasBeenSet = true;
    m_targets.emplace_back(std::forward<TargetsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_targetGroupArn;

  Aws::Vector<TargetDescription> m_targets;
  bool m_targetGroupArnHasBeenSet = false;
  bool m_targetsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticLoadBalancingv2
}  // namespace Aws
