# Copyright 2020 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

require 'google/apis/core/base_service'
require 'google/apis/core/json_representation'
require 'google/apis/core/hashable'
require 'google/apis/errors'

module Google
  module Apis
    module ComputeV1
      # Compute Engine API
      #
      # Creates and runs virtual machines on Google Cloud Platform.
      #
      # @example
      #    require 'google/apis/compute_v1'
      #
      #    Compute = Google::Apis::ComputeV1 # Alias the module
      #    service = Compute::ComputeService.new
      #
      # @see https://cloud.google.com/compute/
      class ComputeService < Google::Apis::Core::BaseService
        DEFAULT_ENDPOINT_TEMPLATE = "https://compute.$UNIVERSE_DOMAIN$/"

        # @return [String]
        #  API key. Your API key identifies your project and provides you with API access,
        #  quota, and reports. Required unless you provide an OAuth 2.0 token.
        attr_accessor :key

        # @return [String]
        #  Available to use for quota purposes for server-side applications. Can be any
        #  arbitrary string assigned to a user, but should not exceed 40 characters.
        attr_accessor :quota_user

        # @return [String]
        #  Legacy name for parameter that has been superseded by `quotaUser`.
        attr_accessor :user_ip

        def initialize
          super(DEFAULT_ENDPOINT_TEMPLATE, 'compute/v1/',
                client_name: 'google-apis-compute_v1',
                client_version: Google::Apis::ComputeV1::GEM_VERSION)
          @batch_path = 'batch/compute/v1'
        end
        
        # Retrieves an aggregated list of accelerator types. To prevent failure, Google
        # recommends that you set the `returnPartialSuccess` parameter to `true`.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Boolean] include_all_scopes
        #   Indicates whether every visible scope for each scope type (zone, region,
        #   global) should be included in the response. For new resource types added after
        #   this field, the flag has no effect as new resource types will always include
        #   every visible scope for each scope type in response. For resource types which
        #   predate this field, if this flag is omitted or false, only scopes of the scope
        #   types where the resource type is expected to be found will be included.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [Fixnum] service_project_number
        #   The Shared VPC service project id or service project number for which
        #   aggregated list request is invoked for subnetworks list-usable api.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::AcceleratorTypeAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::AcceleratorTypeAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_accelerator_type_list(project, filter: nil, include_all_scopes: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, service_project_number: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/aggregated/acceleratorTypes', options)
          command.response_representation = Google::Apis::ComputeV1::AcceleratorTypeAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::AcceleratorTypeAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['includeAllScopes'] = include_all_scopes unless include_all_scopes.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['serviceProjectNumber'] = service_project_number unless service_project_number.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified accelerator type.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] accelerator_type
        #   Name of the accelerator type to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::AcceleratorType] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::AcceleratorType]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_accelerator_type(project, zone, accelerator_type, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/acceleratorTypes/{acceleratorType}', options)
          command.response_representation = Google::Apis::ComputeV1::AcceleratorType::Representation
          command.response_class = Google::Apis::ComputeV1::AcceleratorType
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['acceleratorType'] = accelerator_type unless accelerator_type.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of accelerator types that are available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::AcceleratorTypeList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::AcceleratorTypeList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_accelerator_types(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/acceleratorTypes', options)
          command.response_representation = Google::Apis::ComputeV1::AcceleratorTypeList::Representation
          command.response_class = Google::Apis::ComputeV1::AcceleratorTypeList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of addresses. To prevent failure, Google
        # recommends that you set the `returnPartialSuccess` parameter to `true`.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Boolean] include_all_scopes
        #   Indicates whether every visible scope for each scope type (zone, region,
        #   global) should be included in the response. For new resource types added after
        #   this field, the flag has no effect as new resource types will always include
        #   every visible scope for each scope type in response. For resource types which
        #   predate this field, if this flag is omitted or false, only scopes of the scope
        #   types where the resource type is expected to be found will be included.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [Fixnum] service_project_number
        #   The Shared VPC service project id or service project number for which
        #   aggregated list request is invoked for subnetworks list-usable api.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::AddressAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::AddressAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_addresses(project, filter: nil, include_all_scopes: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, service_project_number: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/aggregated/addresses', options)
          command.response_representation = Google::Apis::ComputeV1::AddressAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::AddressAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['includeAllScopes'] = include_all_scopes unless include_all_scopes.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['serviceProjectNumber'] = service_project_number unless service_project_number.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified address resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] address
        #   Name of the address resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_address(project, region, address, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/regions/{region}/addresses/{address}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['address'] = address unless address.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified address resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] address
        #   Name of the address resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Address] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Address]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_address(project, region, address, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/regions/{region}/addresses/{address}', options)
          command.response_representation = Google::Apis::ComputeV1::Address::Representation
          command.response_class = Google::Apis::ComputeV1::Address
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['address'] = address unless address.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an address resource in the specified project by using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [Google::Apis::ComputeV1::Address] address_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_address(project, region, address_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/regions/{region}/addresses', options)
          command.request_representation = Google::Apis::ComputeV1::Address::Representation
          command.request_object = address_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of addresses contained within the specified region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::AddressList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::AddressList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_addresses(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/regions/{region}/addresses', options)
          command.response_representation = Google::Apis::ComputeV1::AddressList::Representation
          command.response_class = Google::Apis::ComputeV1::AddressList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Moves the specified address resource.
        # @param [String] project
        #   Source project ID which the Address is moved from.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] address
        #   Name of the address resource to move.
        # @param [Google::Apis::ComputeV1::RegionAddressesMoveRequest] region_addresses_move_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def move_address(project, region, address, region_addresses_move_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/regions/{region}/addresses/{address}/move', options)
          command.request_representation = Google::Apis::ComputeV1::RegionAddressesMoveRequest::Representation
          command.request_object = region_addresses_move_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['address'] = address unless address.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on an Address. To learn more about labels, read the Labeling
        # Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The region for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::RegionSetLabelsRequest] region_set_labels_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_address_labels(project, region, resource, region_set_labels_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/regions/{region}/addresses/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeV1::RegionSetLabelsRequest::Representation
          command.request_object = region_set_labels_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of autoscalers. To prevent failure, Google
        # recommends that you set the `returnPartialSuccess` parameter to `true`.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Boolean] include_all_scopes
        #   Indicates whether every visible scope for each scope type (zone, region,
        #   global) should be included in the response. For new resource types added after
        #   this field, the flag has no effect as new resource types will always include
        #   every visible scope for each scope type in response. For resource types which
        #   predate this field, if this flag is omitted or false, only scopes of the scope
        #   types where the resource type is expected to be found will be included.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [Fixnum] service_project_number
        #   The Shared VPC service project id or service project number for which
        #   aggregated list request is invoked for subnetworks list-usable api.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::AutoscalerAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::AutoscalerAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_autoscalers(project, filter: nil, include_all_scopes: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, service_project_number: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/aggregated/autoscalers', options)
          command.response_representation = Google::Apis::ComputeV1::AutoscalerAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::AutoscalerAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['includeAllScopes'] = include_all_scopes unless include_all_scopes.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['serviceProjectNumber'] = service_project_number unless service_project_number.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified autoscaler.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [String] autoscaler
        #   Name of the autoscaler to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_autoscaler(project, zone, autoscaler, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/zones/{zone}/autoscalers/{autoscaler}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified autoscaler resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [String] autoscaler
        #   Name of the autoscaler to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Autoscaler] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Autoscaler]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_autoscaler(project, zone, autoscaler, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/autoscalers/{autoscaler}', options)
          command.response_representation = Google::Apis::ComputeV1::Autoscaler::Representation
          command.response_class = Google::Apis::ComputeV1::Autoscaler
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an autoscaler in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [Google::Apis::ComputeV1::Autoscaler] autoscaler_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_autoscaler(project, zone, autoscaler_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/autoscalers', options)
          command.request_representation = Google::Apis::ComputeV1::Autoscaler::Representation
          command.request_object = autoscaler_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of autoscalers contained within the specified zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::AutoscalerList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::AutoscalerList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_autoscalers(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/autoscalers', options)
          command.response_representation = Google::Apis::ComputeV1::AutoscalerList::Representation
          command.response_class = Google::Apis::ComputeV1::AutoscalerList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an autoscaler in the specified project using the data included in the
        # request. This method supports PATCH semantics and uses the JSON merge patch
        # format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [Google::Apis::ComputeV1::Autoscaler] autoscaler_object
        # @param [String] autoscaler
        #   Name of the autoscaler to patch.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_autoscaler(project, zone, autoscaler_object = nil, autoscaler: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, 'projects/{project}/zones/{zone}/autoscalers', options)
          command.request_representation = Google::Apis::ComputeV1::Autoscaler::Representation
          command.request_object = autoscaler_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an autoscaler in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [Google::Apis::ComputeV1::Autoscaler] autoscaler_object
        # @param [String] autoscaler
        #   Name of the autoscaler to update.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_autoscaler(project, zone, autoscaler_object = nil, autoscaler: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:put, 'projects/{project}/zones/{zone}/autoscalers', options)
          command.request_representation = Google::Apis::ComputeV1::Autoscaler::Representation
          command.request_object = autoscaler_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds a key for validating requests with signed URLs for this backend bucket.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to which the Signed URL Key should be added.
        #   The name should conform to RFC1035.
        # @param [Google::Apis::ComputeV1::SignedUrlKey] signed_url_key_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_backend_bucket_signed_url_key(project, backend_bucket, signed_url_key_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/backendBuckets/{backendBucket}/addSignedUrlKey', options)
          command.request_representation = Google::Apis::ComputeV1::SignedUrlKey::Representation
          command.request_object = signed_url_key_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified BackendBucket resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_backend_bucket(project, backend_bucket, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/global/backendBuckets/{backendBucket}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a key for validating requests with signed URLs for this backend bucket.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to which the Signed URL Key should be added.
        #   The name should conform to RFC1035.
        # @param [String] key_name
        #   The name of the Signed URL Key to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_backend_bucket_signed_url_key(project, backend_bucket, key_name, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/backendBuckets/{backendBucket}/deleteSignedUrlKey', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['keyName'] = key_name unless key_name.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified BackendBucket resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::BackendBucket] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::BackendBucket]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_backend_bucket(project, backend_bucket, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/backendBuckets/{backendBucket}', options)
          command.response_representation = Google::Apis::ComputeV1::BackendBucket::Representation
          command.response_class = Google::Apis::ComputeV1::BackendBucket
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the access control policy for a resource. May be empty if no such policy
        # or resource exists.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Fixnum] options_requested_policy_version
        #   Requested IAM Policy version.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_backend_bucket_iam_policy(project, resource, options_requested_policy_version: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/backendBuckets/{resource}/getIamPolicy', options)
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['optionsRequestedPolicyVersion'] = options_requested_policy_version unless options_requested_policy_version.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a BackendBucket resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::BackendBucket] backend_bucket_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_backend_bucket(project, backend_bucket_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/backendBuckets', options)
          command.request_representation = Google::Apis::ComputeV1::BackendBucket::Representation
          command.request_object = backend_bucket_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of BackendBucket resources available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::BackendBucketList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::BackendBucketList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_backend_buckets(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/backendBuckets', options)
          command.response_representation = Google::Apis::ComputeV1::BackendBucketList::Representation
          command.response_class = Google::Apis::ComputeV1::BackendBucketList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified BackendBucket resource with the data included in the
        # request. This method supports PATCH semantics and uses the JSON merge patch
        # format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to patch.
        # @param [Google::Apis::ComputeV1::BackendBucket] backend_bucket_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_backend_bucket(project, backend_bucket, backend_bucket_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, 'projects/{project}/global/backendBuckets/{backendBucket}', options)
          command.request_representation = Google::Apis::ComputeV1::BackendBucket::Representation
          command.request_object = backend_bucket_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the edge security policy for the specified backend bucket.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to which the security policy should be set.
        #   The name should conform to RFC1035.
        # @param [Google::Apis::ComputeV1::SecurityPolicyReference] security_policy_reference_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_backend_bucket_edge_security_policy(project, backend_bucket, security_policy_reference_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/backendBuckets/{backendBucket}/setEdgeSecurityPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::SecurityPolicyReference::Representation
          command.request_object = security_policy_reference_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the access control policy on the specified resource. Replaces any
        # existing policy.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::GlobalSetPolicyRequest] global_set_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_backend_bucket_iam_policy(project, resource, global_set_policy_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/backendBuckets/{resource}/setIamPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalSetPolicyRequest::Representation
          command.request_object = global_set_policy_request_object
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_backend_bucket_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/backendBuckets/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified BackendBucket resource with the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to update.
        # @param [Google::Apis::ComputeV1::BackendBucket] backend_bucket_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_backend_bucket(project, backend_bucket, backend_bucket_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:put, 'projects/{project}/global/backendBuckets/{backendBucket}', options)
          command.request_representation = Google::Apis::ComputeV1::BackendBucket::Representation
          command.request_object = backend_bucket_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds a key for validating requests with signed URLs for this backend service.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to which the Signed URL Key should be
        #   added. The name should conform to RFC1035.
        # @param [Google::Apis::ComputeV1::SignedUrlKey] signed_url_key_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_backend_service_signed_url_key(project, backend_service, signed_url_key_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/backendServices/{backendService}/addSignedUrlKey', options)
          command.request_representation = Google::Apis::ComputeV1::SignedUrlKey::Representation
          command.request_object = signed_url_key_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of all BackendService resources, regional and global,
        # available to the specified project. To prevent failure, Google recommends that
        # you set the `returnPartialSuccess` parameter to `true`.
        # @param [String] project
        #   Name of the project scoping this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Boolean] include_all_scopes
        #   Indicates whether every visible scope for each scope type (zone, region,
        #   global) should be included in the response. For new resource types added after
        #   this field, the flag has no effect as new resource types will always include
        #   every visible scope for each scope type in response. For resource types which
        #   predate this field, if this flag is omitted or false, only scopes of the scope
        #   types where the resource type is expected to be found will be included.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [Fixnum] service_project_number
        #   The Shared VPC service project id or service project number for which
        #   aggregated list request is invoked for subnetworks list-usable api.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::BackendServiceAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::BackendServiceAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_backend_service_list(project, filter: nil, include_all_scopes: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, service_project_number: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/aggregated/backendServices', options)
          command.response_representation = Google::Apis::ComputeV1::BackendServiceAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::BackendServiceAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['includeAllScopes'] = include_all_scopes unless include_all_scopes.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['serviceProjectNumber'] = service_project_number unless service_project_number.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified BackendService resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_backend_service(project, backend_service, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/global/backendServices/{backendService}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a key for validating requests with signed URLs for this backend
        # service.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to which the Signed URL Key should be
        #   added. The name should conform to RFC1035.
        # @param [String] key_name
        #   The name of the Signed URL Key to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_backend_service_signed_url_key(project, backend_service, key_name, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/backendServices/{backendService}/deleteSignedUrlKey', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['keyName'] = key_name unless key_name.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified BackendService resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::BackendService] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::BackendService]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_backend_service(project, backend_service, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/backendServices/{backendService}', options)
          command.response_representation = Google::Apis::ComputeV1::BackendService::Representation
          command.response_class = Google::Apis::ComputeV1::BackendService
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the most recent health check results for this BackendService. Example
        # request body: ` "group": "/zones/us-east1-b/instanceGroups/lb-backend-example"
        # `
        # @param [String] project
        # @param [String] backend_service
        #   Name of the BackendService resource to which the queried instance belongs.
        # @param [Google::Apis::ComputeV1::ResourceGroupReference] resource_group_reference_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::BackendServiceGroupHealth] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::BackendServiceGroupHealth]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_backend_service_health(project, backend_service, resource_group_reference_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/backendServices/{backendService}/getHealth', options)
          command.request_representation = Google::Apis::ComputeV1::ResourceGroupReference::Representation
          command.request_object = resource_group_reference_object
          command.response_representation = Google::Apis::ComputeV1::BackendServiceGroupHealth::Representation
          command.response_class = Google::Apis::ComputeV1::BackendServiceGroupHealth
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the access control policy for a resource. May be empty if no such policy
        # or resource exists.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Fixnum] options_requested_policy_version
        #   Requested IAM Policy version.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_backend_service_iam_policy(project, resource, options_requested_policy_version: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/backendServices/{resource}/getIamPolicy', options)
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['optionsRequestedPolicyVersion'] = options_requested_policy_version unless options_requested_policy_version.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a BackendService resource in the specified project using the data
        # included in the request. For more information, see Backend services overview .
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::BackendService] backend_service_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_backend_service(project, backend_service_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/backendServices', options)
          command.request_representation = Google::Apis::ComputeV1::BackendService::Representation
          command.request_object = backend_service_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of BackendService resources available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::BackendServiceList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::BackendServiceList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_backend_services(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/backendServices', options)
          command.response_representation = Google::Apis::ComputeV1::BackendServiceList::Representation
          command.response_class = Google::Apis::ComputeV1::BackendServiceList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of all usable backend services in the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::BackendServiceListUsable] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::BackendServiceListUsable]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_backend_service_usable(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/backendServices/listUsable', options)
          command.response_representation = Google::Apis::ComputeV1::BackendServiceListUsable::Representation
          command.response_class = Google::Apis::ComputeV1::BackendServiceListUsable
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Patches the specified BackendService resource with the data included in the
        # request. For more information, see Backend services overview. This method
        # supports PATCH semantics and uses the JSON merge patch format and processing
        # rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to patch.
        # @param [Google::Apis::ComputeV1::BackendService] backend_service_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_backend_service(project, backend_service, backend_service_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, 'projects/{project}/global/backendServices/{backendService}', options)
          command.request_representation = Google::Apis::ComputeV1::BackendService::Representation
          command.request_object = backend_service_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the edge security policy for the specified backend service.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to which the edge security policy should
        #   be set. The name should conform to RFC1035.
        # @param [Google::Apis::ComputeV1::SecurityPolicyReference] security_policy_reference_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_backend_service_edge_security_policy(project, backend_service, security_policy_reference_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/backendServices/{backendService}/setEdgeSecurityPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::SecurityPolicyReference::Representation
          command.request_object = security_policy_reference_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the access control policy on the specified resource. Replaces any
        # existing policy.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::GlobalSetPolicyRequest] global_set_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_backend_service_iam_policy(project, resource, global_set_policy_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/backendServices/{resource}/setIamPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalSetPolicyRequest::Representation
          command.request_object = global_set_policy_request_object
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the Google Cloud Armor security policy for the specified backend service.
        # For more information, see Google Cloud Armor Overview
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to which the security policy should be set.
        #   The name should conform to RFC1035.
        # @param [Google::Apis::ComputeV1::SecurityPolicyReference] security_policy_reference_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_backend_service_security_policy(project, backend_service, security_policy_reference_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/backendServices/{backendService}/setSecurityPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::SecurityPolicyReference::Representation
          command.request_object = security_policy_reference_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_backend_service_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/backendServices/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified BackendService resource with the data included in the
        # request. For more information, see Backend services overview.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to update.
        # @param [Google::Apis::ComputeV1::BackendService] backend_service_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_backend_service(project, backend_service, backend_service_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:put, 'projects/{project}/global/backendServices/{backendService}', options)
          command.request_representation = Google::Apis::ComputeV1::BackendService::Representation
          command.request_object = backend_service_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of disk types. To prevent failure, Google
        # recommends that you set the `returnPartialSuccess` parameter to `true`.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Boolean] include_all_scopes
        #   Indicates whether every visible scope for each scope type (zone, region,
        #   global) should be included in the response. For new resource types added after
        #   this field, the flag has no effect as new resource types will always include
        #   every visible scope for each scope type in response. For resource types which
        #   predate this field, if this flag is omitted or false, only scopes of the scope
        #   types where the resource type is expected to be found will be included.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [Fixnum] service_project_number
        #   The Shared VPC service project id or service project number for which
        #   aggregated list request is invoked for subnetworks list-usable api.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::DiskTypeAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::DiskTypeAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_disk_types(project, filter: nil, include_all_scopes: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, service_project_number: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/aggregated/diskTypes', options)
          command.response_representation = Google::Apis::ComputeV1::DiskTypeAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::DiskTypeAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['includeAllScopes'] = include_all_scopes unless include_all_scopes.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['serviceProjectNumber'] = service_project_number unless service_project_number.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified disk type.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk_type
        #   Name of the disk type to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::DiskType] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::DiskType]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_disk_type(project, zone, disk_type, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/diskTypes/{diskType}', options)
          command.response_representation = Google::Apis::ComputeV1::DiskType::Representation
          command.response_class = Google::Apis::ComputeV1::DiskType
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['diskType'] = disk_type unless disk_type.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of disk types available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::DiskTypeList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::DiskTypeList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_disk_types(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/diskTypes', options)
          command.response_representation = Google::Apis::ComputeV1::DiskTypeList::Representation
          command.response_class = Google::Apis::ComputeV1::DiskTypeList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds existing resource policies to a disk. You can only add one policy which
        # will be applied to this disk for scheduling snapshot creation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   The disk name for this request.
        # @param [Google::Apis::ComputeV1::DisksAddResourcePoliciesRequest] disks_add_resource_policies_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_disk_resource_policies(project, zone, disk, disks_add_resource_policies_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/disks/{disk}/addResourcePolicies', options)
          command.request_representation = Google::Apis::ComputeV1::DisksAddResourcePoliciesRequest::Representation
          command.request_object = disks_add_resource_policies_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of persistent disks. To prevent failure, Google
        # recommends that you set the `returnPartialSuccess` parameter to `true`.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Boolean] include_all_scopes
        #   Indicates whether every visible scope for each scope type (zone, region,
        #   global) should be included in the response. For new resource types added after
        #   this field, the flag has no effect as new resource types will always include
        #   every visible scope for each scope type in response. For resource types which
        #   predate this field, if this flag is omitted or false, only scopes of the scope
        #   types where the resource type is expected to be found will be included.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [Fixnum] service_project_number
        #   The Shared VPC service project id or service project number for which
        #   aggregated list request is invoked for subnetworks list-usable api.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::DiskAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::DiskAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_disk(project, filter: nil, include_all_scopes: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, service_project_number: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/aggregated/disks', options)
          command.response_representation = Google::Apis::ComputeV1::DiskAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::DiskAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['includeAllScopes'] = include_all_scopes unless include_all_scopes.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['serviceProjectNumber'] = service_project_number unless service_project_number.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Bulk create a set of disks.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [Google::Apis::ComputeV1::BulkInsertDiskResource] bulk_insert_disk_resource_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def bulk_disk_insert(project, zone, bulk_insert_disk_resource_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/disks/bulkInsert', options)
          command.request_representation = Google::Apis::ComputeV1::BulkInsertDiskResource::Representation
          command.request_object = bulk_insert_disk_resource_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on many disks at once. To learn more about labels, read the
        # Labeling Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [Google::Apis::ComputeV1::BulkZoneSetLabelsRequest] bulk_zone_set_labels_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def bulk_disk_set_labels(project, zone, bulk_zone_set_labels_request_object = nil, request_id: nil, resource: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/disks/bulkSetLabels', options)
          command.request_representation = Google::Apis::ComputeV1::BulkZoneSetLabelsRequest::Representation
          command.request_object = bulk_zone_set_labels_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a snapshot of a specified persistent disk. For regular snapshot
        # creation, consider using snapshots.insert instead, as that method supports
        # more features, such as creating snapshots in a project different from the
        # source disk project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   Name of the persistent disk to snapshot.
        # @param [Google::Apis::ComputeV1::Snapshot] snapshot_object
        # @param [Boolean] guest_flush
        #   [Input Only] Whether to attempt an application consistent snapshot by
        #   informing the OS to prepare for the snapshot process.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_disk_snapshot(project, zone, disk, snapshot_object = nil, guest_flush: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/disks/{disk}/createSnapshot', options)
          command.request_representation = Google::Apis::ComputeV1::Snapshot::Representation
          command.request_object = snapshot_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['guestFlush'] = guest_flush unless guest_flush.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified persistent disk. Deleting a disk removes its data
        # permanently and is irreversible. However, deleting a disk does not delete any
        # snapshots previously made from the disk. You must separately delete snapshots.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   Name of the persistent disk to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_disk(project, zone, disk, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/zones/{zone}/disks/{disk}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified persistent disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   Name of the persistent disk to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Disk] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Disk]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_disk(project, zone, disk, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/disks/{disk}', options)
          command.response_representation = Google::Apis::ComputeV1::Disk::Representation
          command.response_class = Google::Apis::ComputeV1::Disk
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the access control policy for a resource. May be empty if no such policy
        # or resource exists.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Fixnum] options_requested_policy_version
        #   Requested IAM Policy version.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_disk_iam_policy(project, zone, resource, options_requested_policy_version: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/disks/{resource}/getIamPolicy', options)
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['optionsRequestedPolicyVersion'] = options_requested_policy_version unless options_requested_policy_version.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a persistent disk in the specified project using the data in the
        # request. You can create a disk from a source (sourceImage, sourceSnapshot, or
        # sourceDisk) or create an empty 500 GB data disk by omitting all properties.
        # You can also create a disk that is larger than the default size by specifying
        # the sizeGb property.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [Google::Apis::ComputeV1::Disk] disk_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] source_image
        #   Source image to restore onto a disk. This field is optional.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_disk(project, zone, disk_object = nil, request_id: nil, source_image: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/disks', options)
          command.request_representation = Google::Apis::ComputeV1::Disk::Representation
          command.request_object = disk_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['sourceImage'] = source_image unless source_image.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of persistent disks contained within the specified zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::DiskList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::DiskList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_disks(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/disks', options)
          command.response_representation = Google::Apis::ComputeV1::DiskList::Representation
          command.response_class = Google::Apis::ComputeV1::DiskList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Removes resource policies from a disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   The disk name for this request.
        # @param [Google::Apis::ComputeV1::DisksRemoveResourcePoliciesRequest] disks_remove_resource_policies_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def remove_disk_resource_policies(project, zone, disk, disks_remove_resource_policies_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/disks/{disk}/removeResourcePolicies', options)
          command.request_representation = Google::Apis::ComputeV1::DisksRemoveResourcePoliciesRequest::Representation
          command.request_object = disks_remove_resource_policies_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Resizes the specified persistent disk. You can only increase the size of the
        # disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   The name of the persistent disk.
        # @param [Google::Apis::ComputeV1::DisksResizeRequest] disks_resize_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def resize_disk(project, zone, disk, disks_resize_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/disks/{disk}/resize', options)
          command.request_representation = Google::Apis::ComputeV1::DisksResizeRequest::Representation
          command.request_object = disks_resize_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the access control policy on the specified resource. Replaces any
        # existing policy.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::ZoneSetPolicyRequest] zone_set_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_disk_iam_policy(project, zone, resource, zone_set_policy_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/disks/{resource}/setIamPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::ZoneSetPolicyRequest::Representation
          command.request_object = zone_set_policy_request_object
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on a disk. To learn more about labels, read the Labeling
        # Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::ZoneSetLabelsRequest] zone_set_labels_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_disk_labels(project, zone, resource, zone_set_labels_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/disks/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeV1::ZoneSetLabelsRequest::Representation
          command.request_object = zone_set_labels_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Starts asynchronous replication. Must be invoked on the primary disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   The name of the persistent disk.
        # @param [Google::Apis::ComputeV1::DisksStartAsyncReplicationRequest] disks_start_async_replication_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def start_disk_async_replication(project, zone, disk, disks_start_async_replication_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/disks/{disk}/startAsyncReplication', options)
          command.request_representation = Google::Apis::ComputeV1::DisksStartAsyncReplicationRequest::Representation
          command.request_object = disks_start_async_replication_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Stops asynchronous replication. Can be invoked either on the primary or on the
        # secondary disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   The name of the persistent disk.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def stop_disk_async_replication(project, zone, disk, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/disks/{disk}/stopAsyncReplication', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Stops asynchronous replication for a consistency group of disks. Can be
        # invoked either in the primary or secondary scope.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request. This must be the zone of the primary or
        #   secondary disks in the consistency group.
        # @param [Google::Apis::ComputeV1::DisksStopGroupAsyncReplicationResource] disks_stop_group_async_replication_resource_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def stop_disk_group_async_replication(project, zone, disks_stop_group_async_replication_resource_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/disks/stopGroupAsyncReplication', options)
          command.request_representation = Google::Apis::ComputeV1::DisksStopGroupAsyncReplicationResource::Representation
          command.request_object = disks_stop_group_async_replication_resource_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_disk_iam_permissions(project, zone, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/disks/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified disk with the data included in the request. The update
        # is performed only on selected fields included as part of update-mask. Only the
        # following fields can be modified: user_license.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   The disk name for this request.
        # @param [Google::Apis::ComputeV1::Disk] disk_object
        # @param [Array<String>, String] paths
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] update_mask
        #   update_mask indicates fields to be updated as part of this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_disk(project, zone, disk, disk_object = nil, paths: nil, request_id: nil, update_mask: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, 'projects/{project}/zones/{zone}/disks/{disk}', options)
          command.request_representation = Google::Apis::ComputeV1::Disk::Representation
          command.request_object = disk_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['paths'] = paths unless paths.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['updateMask'] = update_mask unless update_mask.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified externalVpnGateway.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] external_vpn_gateway
        #   Name of the externalVpnGateways to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_external_vpn_gateway(project, external_vpn_gateway, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/global/externalVpnGateways/{externalVpnGateway}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['externalVpnGateway'] = external_vpn_gateway unless external_vpn_gateway.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified externalVpnGateway. Get a list of available
        # externalVpnGateways by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] external_vpn_gateway
        #   Name of the externalVpnGateway to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ExternalVpnGateway] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ExternalVpnGateway]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_external_vpn_gateway(project, external_vpn_gateway, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/externalVpnGateways/{externalVpnGateway}', options)
          command.response_representation = Google::Apis::ComputeV1::ExternalVpnGateway::Representation
          command.response_class = Google::Apis::ComputeV1::ExternalVpnGateway
          command.params['project'] = project unless project.nil?
          command.params['externalVpnGateway'] = external_vpn_gateway unless external_vpn_gateway.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a ExternalVpnGateway in the specified project using the data included
        # in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::ExternalVpnGateway] external_vpn_gateway_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_external_vpn_gateway(project, external_vpn_gateway_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/externalVpnGateways', options)
          command.request_representation = Google::Apis::ComputeV1::ExternalVpnGateway::Representation
          command.request_object = external_vpn_gateway_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of ExternalVpnGateway available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ExternalVpnGatewayList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ExternalVpnGatewayList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_external_vpn_gateways(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/externalVpnGateways', options)
          command.response_representation = Google::Apis::ComputeV1::ExternalVpnGatewayList::Representation
          command.response_class = Google::Apis::ComputeV1::ExternalVpnGatewayList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on an ExternalVpnGateway. To learn more about labels, read the
        # Labeling Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::GlobalSetLabelsRequest] global_set_labels_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_external_vpn_gateway_labels(project, resource, global_set_labels_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/externalVpnGateways/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalSetLabelsRequest::Representation
          command.request_object = global_set_labels_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_external_vpn_gateway_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/externalVpnGateways/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Inserts an association for the specified firewall policy.
        # @param [String] firewall_policy
        #   Name of the firewall policy to update.
        # @param [Google::Apis::ComputeV1::FirewallPolicyAssociation] firewall_policy_association_object
        # @param [Boolean] replace_existing_association
        #   Indicates whether or not to replace it if an association of the attachment
        #   already exists. This is false by default, in which case an error will be
        #   returned if an association already exists.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_firewall_policy_association(firewall_policy, firewall_policy_association_object = nil, replace_existing_association: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'locations/global/firewallPolicies/{firewallPolicy}/addAssociation', options)
          command.request_representation = Google::Apis::ComputeV1::FirewallPolicyAssociation::Representation
          command.request_object = firewall_policy_association_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['firewallPolicy'] = firewall_policy unless firewall_policy.nil?
          command.query['replaceExistingAssociation'] = replace_existing_association unless replace_existing_association.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Inserts a rule into a firewall policy.
        # @param [String] firewall_policy
        #   Name of the firewall policy to update.
        # @param [Google::Apis::ComputeV1::FirewallPolicyRule] firewall_policy_rule_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_firewall_policy_rule(firewall_policy, firewall_policy_rule_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'locations/global/firewallPolicies/{firewallPolicy}/addRule', options)
          command.request_representation = Google::Apis::ComputeV1::FirewallPolicyRule::Representation
          command.request_object = firewall_policy_rule_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['firewallPolicy'] = firewall_policy unless firewall_policy.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Copies rules to the specified firewall policy.
        # @param [String] firewall_policy
        #   Name of the firewall policy to update.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] source_firewall_policy
        #   The firewall policy from which to copy rules.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def clone_firewall_policy_rules(firewall_policy, request_id: nil, source_firewall_policy: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'locations/global/firewallPolicies/{firewallPolicy}/cloneRules', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['firewallPolicy'] = firewall_policy unless firewall_policy.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['sourceFirewallPolicy'] = source_firewall_policy unless source_firewall_policy.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified policy.
        # @param [String] firewall_policy
        #   Name of the firewall policy to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_firewall_policy(firewall_policy, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'locations/global/firewallPolicies/{firewallPolicy}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['firewallPolicy'] = firewall_policy unless firewall_policy.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified firewall policy.
        # @param [String] firewall_policy
        #   Name of the firewall policy to get.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::FirewallPolicy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::FirewallPolicy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_firewall_policy(firewall_policy, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'locations/global/firewallPolicies/{firewallPolicy}', options)
          command.response_representation = Google::Apis::ComputeV1::FirewallPolicy::Representation
          command.response_class = Google::Apis::ComputeV1::FirewallPolicy
          command.params['firewallPolicy'] = firewall_policy unless firewall_policy.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets an association with the specified name.
        # @param [String] firewall_policy
        #   Name of the firewall policy to which the queried rule belongs.
        # @param [String] name
        #   The name of the association to get from the firewall policy.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::FirewallPolicyAssociation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::FirewallPolicyAssociation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_firewall_policy_association(firewall_policy, name: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'locations/global/firewallPolicies/{firewallPolicy}/getAssociation', options)
          command.response_representation = Google::Apis::ComputeV1::FirewallPolicyAssociation::Representation
          command.response_class = Google::Apis::ComputeV1::FirewallPolicyAssociation
          command.params['firewallPolicy'] = firewall_policy unless firewall_policy.nil?
          command.query['name'] = name unless name.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the access control policy for a resource. May be empty if no such policy
        # or resource exists.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Fixnum] options_requested_policy_version
        #   Requested IAM Policy version.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_firewall_policy_iam_policy(resource, options_requested_policy_version: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'locations/global/firewallPolicies/{resource}/getIamPolicy', options)
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['resource'] = resource unless resource.nil?
          command.query['optionsRequestedPolicyVersion'] = options_requested_policy_version unless options_requested_policy_version.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets a rule of the specified priority.
        # @param [String] firewall_policy
        #   Name of the firewall policy to which the queried rule belongs.
        # @param [Fixnum] priority
        #   The priority of the rule to get from the firewall policy.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::FirewallPolicyRule] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::FirewallPolicyRule]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_firewall_policy_rule(firewall_policy, priority: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'locations/global/firewallPolicies/{firewallPolicy}/getRule', options)
          command.response_representation = Google::Apis::ComputeV1::FirewallPolicyRule::Representation
          command.response_class = Google::Apis::ComputeV1::FirewallPolicyRule
          command.params['firewallPolicy'] = firewall_policy unless firewall_policy.nil?
          command.query['priority'] = priority unless priority.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a new policy in the specified project using the data included in the
        # request.
        # @param [Google::Apis::ComputeV1::FirewallPolicy] firewall_policy_object
        # @param [String] parent_id
        #   Parent ID for this request. The ID can be either be "folders/[FOLDER_ID]" if
        #   the parent is a folder or "organizations/[ORGANIZATION_ID]" if the parent is
        #   an organization.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_firewall_policy(firewall_policy_object = nil, parent_id: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'locations/global/firewallPolicies', options)
          command.request_representation = Google::Apis::ComputeV1::FirewallPolicy::Representation
          command.request_object = firewall_policy_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.query['parentId'] = parent_id unless parent_id.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all the policies that have been configured for the specified folder or
        # organization.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [String] parent_id
        #   Parent ID for this request. The ID can be either be "folders/[FOLDER_ID]" if
        #   the parent is a folder or "organizations/[ORGANIZATION_ID]" if the parent is
        #   an organization.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::FirewallPolicyList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::FirewallPolicyList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_firewall_policies(filter: nil, max_results: nil, order_by: nil, page_token: nil, parent_id: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'locations/global/firewallPolicies', options)
          command.response_representation = Google::Apis::ComputeV1::FirewallPolicyList::Representation
          command.response_class = Google::Apis::ComputeV1::FirewallPolicyList
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['parentId'] = parent_id unless parent_id.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists associations of a specified target, i.e., organization or folder.
        # @param [String] target_resource
        #   The target resource to list associations. It is an organization, or a folder.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::FirewallPoliciesListAssociationsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::FirewallPoliciesListAssociationsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_firewall_policy_associations(target_resource: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'locations/global/firewallPolicies/listAssociations', options)
          command.response_representation = Google::Apis::ComputeV1::FirewallPoliciesListAssociationsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::FirewallPoliciesListAssociationsResponse
          command.query['targetResource'] = target_resource unless target_resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Moves the specified firewall policy.
        # @param [String] firewall_policy
        #   Name of the firewall policy to update.
        # @param [String] parent_id
        #   The new parent of the firewall policy. The ID can be either be "folders/[
        #   FOLDER_ID]" if the parent is a folder or "organizations/[ORGANIZATION_ID]" if
        #   the parent is an organization.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def move_firewall_policy(firewall_policy, parent_id: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'locations/global/firewallPolicies/{firewallPolicy}/move', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['firewallPolicy'] = firewall_policy unless firewall_policy.nil?
          command.query['parentId'] = parent_id unless parent_id.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Patches the specified policy with the data included in the request.
        # @param [String] firewall_policy
        #   Name of the firewall policy to update.
        # @param [Google::Apis::ComputeV1::FirewallPolicy] firewall_policy_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_firewall_policy(firewall_policy, firewall_policy_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, 'locations/global/firewallPolicies/{firewallPolicy}', options)
          command.request_representation = Google::Apis::ComputeV1::FirewallPolicy::Representation
          command.request_object = firewall_policy_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['firewallPolicy'] = firewall_policy unless firewall_policy.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Patches a rule of the specified priority.
        # @param [String] firewall_policy
        #   Name of the firewall policy to update.
        # @param [Google::Apis::ComputeV1::FirewallPolicyRule] firewall_policy_rule_object
        # @param [Fixnum] priority
        #   The priority of the rule to patch.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_firewall_policy_rule(firewall_policy, firewall_policy_rule_object = nil, priority: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'locations/global/firewallPolicies/{firewallPolicy}/patchRule', options)
          command.request_representation = Google::Apis::ComputeV1::FirewallPolicyRule::Representation
          command.request_object = firewall_policy_rule_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['firewallPolicy'] = firewall_policy unless firewall_policy.nil?
          command.query['priority'] = priority unless priority.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Removes an association for the specified firewall policy.
        # @param [String] firewall_policy
        #   Name of the firewall policy to update.
        # @param [String] name
        #   Name for the attachment that will be removed.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def remove_firewall_policy_association(firewall_policy, name: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'locations/global/firewallPolicies/{firewallPolicy}/removeAssociation', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['firewallPolicy'] = firewall_policy unless firewall_policy.nil?
          command.query['name'] = name unless name.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a rule of the specified priority.
        # @param [String] firewall_policy
        #   Name of the firewall policy to update.
        # @param [Fixnum] priority
        #   The priority of the rule to remove from the firewall policy.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def remove_firewall_policy_rule(firewall_policy, priority: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'locations/global/firewallPolicies/{firewallPolicy}/removeRule', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['firewallPolicy'] = firewall_policy unless firewall_policy.nil?
          command.query['priority'] = priority unless priority.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the access control policy on the specified resource. Replaces any
        # existing policy.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::GlobalOrganizationSetPolicyRequest] global_organization_set_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_firewall_policy_iam_policy(resource, global_organization_set_policy_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'locations/global/firewallPolicies/{resource}/setIamPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalOrganizationSetPolicyRequest::Representation
          command.request_object = global_organization_set_policy_request_object
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_firewall_policy_iam_permissions(resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'locations/global/firewallPolicies/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified firewall.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] firewall
        #   Name of the firewall rule to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_firewall(project, firewall, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/global/firewalls/{firewall}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['firewall'] = firewall unless firewall.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified firewall.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] firewall
        #   Name of the firewall rule to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Firewall] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Firewall]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_firewall(project, firewall, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/firewalls/{firewall}', options)
          command.response_representation = Google::Apis::ComputeV1::Firewall::Representation
          command.response_class = Google::Apis::ComputeV1::Firewall
          command.params['project'] = project unless project.nil?
          command.params['firewall'] = firewall unless firewall.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a firewall rule in the specified project using the data included in
        # the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::Firewall] firewall_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_firewall(project, firewall_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/firewalls', options)
          command.request_representation = Google::Apis::ComputeV1::Firewall::Representation
          command.request_object = firewall_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of firewall rules available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::FirewallList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::FirewallList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_firewalls(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/firewalls', options)
          command.response_representation = Google::Apis::ComputeV1::FirewallList::Representation
          command.response_class = Google::Apis::ComputeV1::FirewallList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified firewall rule with the data included in the request.
        # This method supports PATCH semantics and uses the JSON merge patch format and
        # processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] firewall
        #   Name of the firewall rule to patch.
        # @param [Google::Apis::ComputeV1::Firewall] firewall_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_firewall(project, firewall, firewall_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, 'projects/{project}/global/firewalls/{firewall}', options)
          command.request_representation = Google::Apis::ComputeV1::Firewall::Representation
          command.request_object = firewall_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['firewall'] = firewall unless firewall.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified firewall rule with the data included in the request.
        # Note that all fields will be updated if using PUT, even fields that are not
        # specified. To update individual fields, please use PATCH instead.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] firewall
        #   Name of the firewall rule to update.
        # @param [Google::Apis::ComputeV1::Firewall] firewall_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_firewall(project, firewall, firewall_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:put, 'projects/{project}/global/firewalls/{firewall}', options)
          command.request_representation = Google::Apis::ComputeV1::Firewall::Representation
          command.request_object = firewall_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['firewall'] = firewall unless firewall.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of forwarding rules. To prevent failure, Google
        # recommends that you set the `returnPartialSuccess` parameter to `true`.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Boolean] include_all_scopes
        #   Indicates whether every visible scope for each scope type (zone, region,
        #   global) should be included in the response. For new resource types added after
        #   this field, the flag has no effect as new resource types will always include
        #   every visible scope for each scope type in response. For resource types which
        #   predate this field, if this flag is omitted or false, only scopes of the scope
        #   types where the resource type is expected to be found will be included.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [Fixnum] service_project_number
        #   The Shared VPC service project id or service project number for which
        #   aggregated list request is invoked for subnetworks list-usable api.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ForwardingRuleAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ForwardingRuleAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_forwarding_rules(project, filter: nil, include_all_scopes: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, service_project_number: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/aggregated/forwardingRules', options)
          command.response_representation = Google::Apis::ComputeV1::ForwardingRuleAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::ForwardingRuleAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['includeAllScopes'] = include_all_scopes unless include_all_scopes.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['serviceProjectNumber'] = service_project_number unless service_project_number.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified ForwardingRule resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_forwarding_rule(project, region, forwarding_rule, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/regions/{region}/forwardingRules/{forwardingRule}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified ForwardingRule resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ForwardingRule] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ForwardingRule]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_forwarding_rule(project, region, forwarding_rule, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/regions/{region}/forwardingRules/{forwardingRule}', options)
          command.response_representation = Google::Apis::ComputeV1::ForwardingRule::Representation
          command.response_class = Google::Apis::ComputeV1::ForwardingRule
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a ForwardingRule resource in the specified project and region using
        # the data included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [Google::Apis::ComputeV1::ForwardingRule] forwarding_rule_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_forwarding_rule(project, region, forwarding_rule_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/regions/{region}/forwardingRules', options)
          command.request_representation = Google::Apis::ComputeV1::ForwardingRule::Representation
          command.request_object = forwarding_rule_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of ForwardingRule resources available to the specified
        # project and region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ForwardingRuleList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ForwardingRuleList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_forwarding_rules(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/regions/{region}/forwardingRules', options)
          command.response_representation = Google::Apis::ComputeV1::ForwardingRuleList::Representation
          command.response_class = Google::Apis::ComputeV1::ForwardingRuleList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified forwarding rule with the data included in the request.
        # This method supports PATCH semantics and uses the JSON merge patch format and
        # processing rules. Currently, you can only patch the network_tier field.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource to patch.
        # @param [Google::Apis::ComputeV1::ForwardingRule] forwarding_rule_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_forwarding_rule(project, region, forwarding_rule, forwarding_rule_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, 'projects/{project}/regions/{region}/forwardingRules/{forwardingRule}', options)
          command.request_representation = Google::Apis::ComputeV1::ForwardingRule::Representation
          command.request_object = forwarding_rule_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on the specified resource. To learn more about labels, read
        # the Labeling Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The region for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::RegionSetLabelsRequest] region_set_labels_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_forwarding_rule_labels(project, region, resource, region_set_labels_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/regions/{region}/forwardingRules/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeV1::RegionSetLabelsRequest::Representation
          command.request_object = region_set_labels_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Changes target URL for forwarding rule. The new target should be of the same
        # type as the old target.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource in which target is to be set.
        # @param [Google::Apis::ComputeV1::TargetReference] target_reference_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_forwarding_rule_target(project, region, forwarding_rule, target_reference_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/regions/{region}/forwardingRules/{forwardingRule}/setTarget', options)
          command.request_representation = Google::Apis::ComputeV1::TargetReference::Representation
          command.request_object = target_reference_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified address resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] address
        #   Name of the address resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_global_address(project, address, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/global/addresses/{address}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['address'] = address unless address.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified address resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] address
        #   Name of the address resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Address] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Address]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_global_address(project, address, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/addresses/{address}', options)
          command.response_representation = Google::Apis::ComputeV1::Address::Representation
          command.response_class = Google::Apis::ComputeV1::Address
          command.params['project'] = project unless project.nil?
          command.params['address'] = address unless address.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an address resource in the specified project by using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::Address] address_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_global_address(project, address_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/addresses', options)
          command.request_representation = Google::Apis::ComputeV1::Address::Representation
          command.request_object = address_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of global addresses.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::AddressList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::AddressList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_global_addresses(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/addresses', options)
          command.response_representation = Google::Apis::ComputeV1::AddressList::Representation
          command.response_class = Google::Apis::ComputeV1::AddressList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Moves the specified address resource from one project to another project.
        # @param [String] project
        #   Source project ID which the Address is moved from.
        # @param [String] address
        #   Name of the address resource to move.
        # @param [Google::Apis::ComputeV1::GlobalAddressesMoveRequest] global_addresses_move_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def move_global_address(project, address, global_addresses_move_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/addresses/{address}/move', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalAddressesMoveRequest::Representation
          command.request_object = global_addresses_move_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['address'] = address unless address.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on a GlobalAddress. To learn more about labels, read the
        # Labeling Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::GlobalSetLabelsRequest] global_set_labels_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_global_address_labels(project, resource, global_set_labels_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/addresses/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalSetLabelsRequest::Representation
          command.request_object = global_set_labels_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified GlobalForwardingRule resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_global_forwarding_rule(project, forwarding_rule, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/global/forwardingRules/{forwardingRule}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified GlobalForwardingRule resource. Gets a list of available
        # forwarding rules by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ForwardingRule] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ForwardingRule]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_global_forwarding_rule(project, forwarding_rule, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/forwardingRules/{forwardingRule}', options)
          command.response_representation = Google::Apis::ComputeV1::ForwardingRule::Representation
          command.response_class = Google::Apis::ComputeV1::ForwardingRule
          command.params['project'] = project unless project.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a GlobalForwardingRule resource in the specified project using the
        # data included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::ForwardingRule] forwarding_rule_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_global_forwarding_rule(project, forwarding_rule_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/forwardingRules', options)
          command.request_representation = Google::Apis::ComputeV1::ForwardingRule::Representation
          command.request_object = forwarding_rule_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of GlobalForwardingRule resources available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ForwardingRuleList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ForwardingRuleList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_global_forwarding_rules(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/forwardingRules', options)
          command.response_representation = Google::Apis::ComputeV1::ForwardingRuleList::Representation
          command.response_class = Google::Apis::ComputeV1::ForwardingRuleList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified forwarding rule with the data included in the request.
        # This method supports PATCH semantics and uses the JSON merge patch format and
        # processing rules. Currently, you can only patch the network_tier field.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource to patch.
        # @param [Google::Apis::ComputeV1::ForwardingRule] forwarding_rule_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_global_forwarding_rule(project, forwarding_rule, forwarding_rule_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, 'projects/{project}/global/forwardingRules/{forwardingRule}', options)
          command.request_representation = Google::Apis::ComputeV1::ForwardingRule::Representation
          command.request_object = forwarding_rule_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on the specified resource. To learn more about labels, read
        # the Labeling resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::GlobalSetLabelsRequest] global_set_labels_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_global_forwarding_rule_labels(project, resource, global_set_labels_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/forwardingRules/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalSetLabelsRequest::Representation
          command.request_object = global_set_labels_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Changes target URL for the GlobalForwardingRule resource. The new target
        # should be of the same type as the old target.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource in which target is to be set.
        # @param [Google::Apis::ComputeV1::TargetReference] target_reference_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_global_forwarding_rule_target(project, forwarding_rule, target_reference_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/forwardingRules/{forwardingRule}/setTarget', options)
          command.request_representation = Google::Apis::ComputeV1::TargetReference::Representation
          command.request_object = target_reference_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Attach a network endpoint to the specified network endpoint group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network_endpoint_group
        #   The name of the network endpoint group where you are attaching network
        #   endpoints to. It should comply with RFC1035.
        # @param [Google::Apis::ComputeV1::GlobalNetworkEndpointGroupsAttachEndpointsRequest] global_network_endpoint_groups_attach_endpoints_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def attach_global_network_endpoint_group_network_endpoints(project, network_endpoint_group, global_network_endpoint_groups_attach_endpoints_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/networkEndpointGroups/{networkEndpointGroup}/attachNetworkEndpoints', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalNetworkEndpointGroupsAttachEndpointsRequest::Representation
          command.request_object = global_network_endpoint_groups_attach_endpoints_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['networkEndpointGroup'] = network_endpoint_group unless network_endpoint_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified network endpoint group.Note that the NEG cannot be
        # deleted if there are backend services referencing it.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network_endpoint_group
        #   The name of the network endpoint group to delete. It should comply with
        #   RFC1035.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_global_network_endpoint_group(project, network_endpoint_group, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/global/networkEndpointGroups/{networkEndpointGroup}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['networkEndpointGroup'] = network_endpoint_group unless network_endpoint_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Detach the network endpoint from the specified network endpoint group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network_endpoint_group
        #   The name of the network endpoint group where you are removing network
        #   endpoints. It should comply with RFC1035.
        # @param [Google::Apis::ComputeV1::GlobalNetworkEndpointGroupsDetachEndpointsRequest] global_network_endpoint_groups_detach_endpoints_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def detach_global_network_endpoint_group_network_endpoints(project, network_endpoint_group, global_network_endpoint_groups_detach_endpoints_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/networkEndpointGroups/{networkEndpointGroup}/detachNetworkEndpoints', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalNetworkEndpointGroupsDetachEndpointsRequest::Representation
          command.request_object = global_network_endpoint_groups_detach_endpoints_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['networkEndpointGroup'] = network_endpoint_group unless network_endpoint_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified network endpoint group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network_endpoint_group
        #   The name of the network endpoint group. It should comply with RFC1035.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NetworkEndpointGroup] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NetworkEndpointGroup]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_global_network_endpoint_group(project, network_endpoint_group, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/networkEndpointGroups/{networkEndpointGroup}', options)
          command.response_representation = Google::Apis::ComputeV1::NetworkEndpointGroup::Representation
          command.response_class = Google::Apis::ComputeV1::NetworkEndpointGroup
          command.params['project'] = project unless project.nil?
          command.params['networkEndpointGroup'] = network_endpoint_group unless network_endpoint_group.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a network endpoint group in the specified project using the parameters
        # that are included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::NetworkEndpointGroup] network_endpoint_group_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_global_network_endpoint_group(project, network_endpoint_group_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/networkEndpointGroups', options)
          command.request_representation = Google::Apis::ComputeV1::NetworkEndpointGroup::Representation
          command.request_object = network_endpoint_group_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of network endpoint groups that are located in the
        # specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NetworkEndpointGroupList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NetworkEndpointGroupList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_global_network_endpoint_groups(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/networkEndpointGroups', options)
          command.response_representation = Google::Apis::ComputeV1::NetworkEndpointGroupList::Representation
          command.response_class = Google::Apis::ComputeV1::NetworkEndpointGroupList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists the network endpoints in the specified network endpoint group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network_endpoint_group
        #   The name of the network endpoint group from which you want to generate a list
        #   of included network endpoints. It should comply with RFC1035.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NetworkEndpointGroupsListNetworkEndpoints] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NetworkEndpointGroupsListNetworkEndpoints]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_global_network_endpoint_group_network_endpoints(project, network_endpoint_group, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/networkEndpointGroups/{networkEndpointGroup}/listNetworkEndpoints', options)
          command.response_representation = Google::Apis::ComputeV1::NetworkEndpointGroupsListNetworkEndpoints::Representation
          command.response_class = Google::Apis::ComputeV1::NetworkEndpointGroupsListNetworkEndpoints
          command.params['project'] = project unless project.nil?
          command.params['networkEndpointGroup'] = network_endpoint_group unless network_endpoint_group.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of all operations. To prevent failure, Google
        # recommends that you set the `returnPartialSuccess` parameter to `true`.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Boolean] include_all_scopes
        #   Indicates whether every visible scope for each scope type (zone, region,
        #   global) should be included in the response. For new resource types added after
        #   this field, the flag has no effect as new resource types will always include
        #   every visible scope for each scope type in response. For resource types which
        #   predate this field, if this flag is omitted or false, only scopes of the scope
        #   types where the resource type is expected to be found will be included.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [Fixnum] service_project_number
        #   The Shared VPC service project id or service project number for which
        #   aggregated list request is invoked for subnetworks list-usable api.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::OperationAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::OperationAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_global_operation(project, filter: nil, include_all_scopes: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, service_project_number: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/aggregated/operations', options)
          command.response_representation = Google::Apis::ComputeV1::OperationAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::OperationAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['includeAllScopes'] = include_all_scopes unless include_all_scopes.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['serviceProjectNumber'] = service_project_number unless service_project_number.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified Operations resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] operation
        #   Name of the Operations resource to delete, or its unique numeric identifier.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [NilClass] No result returned for this method
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [void]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_global_operation(project, operation, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/global/operations/{operation}', options)
          command.params['project'] = project unless project.nil?
          command.params['operation'] = operation unless operation.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the specified Operations resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] operation
        #   Name of the Operations resource to return, or its unique numeric identifier.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_global_operation(project, operation, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/operations/{operation}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['operation'] = operation unless operation.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of Operation resources contained within the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::OperationList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::OperationList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_global_operations(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/operations', options)
          command.response_representation = Google::Apis::ComputeV1::OperationList::Representation
          command.response_class = Google::Apis::ComputeV1::OperationList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Waits for the specified Operation resource to return as `DONE` or for the
        # request to approach the 2 minute deadline, and retrieves the specified
        # Operation resource. This method differs from the `GET` method in that it waits
        # for no more than the default deadline (2 minutes) and then returns the current
        # state of the operation, which might be `DONE` or still in progress. This
        # method is called on a best-effort basis. Specifically: - In uncommon cases,
        # when the server is overloaded, the request might return before the default
        # deadline is reached, or might return after zero seconds. - If the default
        # deadline is reached, there is no guarantee that the operation is actually done
        # when the method returns. Be prepared to retry if the operation is not `DONE`.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] operation
        #   Name of the Operations resource to return, or its unique numeric identifier.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def wait_global_operation(project, operation, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/operations/{operation}/wait', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['operation'] = operation unless operation.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified Operations resource.
        # @param [String] operation
        #   Name of the Operations resource to delete, or its unique numeric identifier.
        # @param [String] parent_id
        #   Parent ID for this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [NilClass] No result returned for this method
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [void]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_global_organization_operation(operation, parent_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'locations/global/operations/{operation}', options)
          command.params['operation'] = operation unless operation.nil?
          command.query['parentId'] = parent_id unless parent_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the specified Operations resource. Gets a list of operations by
        # making a `list()` request.
        # @param [String] operation
        #   Name of the Operations resource to return. Parent is derived from this field.
        # @param [String] parent_id
        #   Parent ID for this request. Not used. Parent is derived from resource_id.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_global_organization_operation(operation, parent_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'locations/global/operations/{operation}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['operation'] = operation unless operation.nil?
          command.query['parentId'] = parent_id unless parent_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of Operation resources contained within the specified
        # organization.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [String] parent_id
        #   Parent ID for this request.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::OperationList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::OperationList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_global_organization_operations(filter: nil, max_results: nil, order_by: nil, page_token: nil, parent_id: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'locations/global/operations', options)
          command.response_representation = Google::Apis::ComputeV1::OperationList::Representation
          command.response_class = Google::Apis::ComputeV1::OperationList
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['parentId'] = parent_id unless parent_id.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified global PublicDelegatedPrefix.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] public_delegated_prefix
        #   Name of the PublicDelegatedPrefix resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_global_public_delegated_prefix(project, public_delegated_prefix, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/global/publicDelegatedPrefixes/{publicDelegatedPrefix}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['publicDelegatedPrefix'] = public_delegated_prefix unless public_delegated_prefix.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified global PublicDelegatedPrefix resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] public_delegated_prefix
        #   Name of the PublicDelegatedPrefix resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::PublicDelegatedPrefix] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::PublicDelegatedPrefix]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_global_public_delegated_prefix(project, public_delegated_prefix, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/publicDelegatedPrefixes/{publicDelegatedPrefix}', options)
          command.response_representation = Google::Apis::ComputeV1::PublicDelegatedPrefix::Representation
          command.response_class = Google::Apis::ComputeV1::PublicDelegatedPrefix
          command.params['project'] = project unless project.nil?
          command.params['publicDelegatedPrefix'] = public_delegated_prefix unless public_delegated_prefix.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a global PublicDelegatedPrefix in the specified project using the
        # parameters that are included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::PublicDelegatedPrefix] public_delegated_prefix_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_global_public_delegated_prefix(project, public_delegated_prefix_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/publicDelegatedPrefixes', options)
          command.request_representation = Google::Apis::ComputeV1::PublicDelegatedPrefix::Representation
          command.request_object = public_delegated_prefix_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists the global PublicDelegatedPrefixes for a project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::PublicDelegatedPrefixList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::PublicDelegatedPrefixList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_global_public_delegated_prefixes(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/publicDelegatedPrefixes', options)
          command.response_representation = Google::Apis::ComputeV1::PublicDelegatedPrefixList::Representation
          command.response_class = Google::Apis::ComputeV1::PublicDelegatedPrefixList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Patches the specified global PublicDelegatedPrefix resource with the data
        # included in the request. This method supports PATCH semantics and uses JSON
        # merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] public_delegated_prefix
        #   Name of the PublicDelegatedPrefix resource to patch.
        # @param [Google::Apis::ComputeV1::PublicDelegatedPrefix] public_delegated_prefix_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_global_public_delegated_prefix(project, public_delegated_prefix, public_delegated_prefix_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, 'projects/{project}/global/publicDelegatedPrefixes/{publicDelegatedPrefix}', options)
          command.request_representation = Google::Apis::ComputeV1::PublicDelegatedPrefix::Representation
          command.request_object = public_delegated_prefix_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['publicDelegatedPrefix'] = public_delegated_prefix unless public_delegated_prefix.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of all HealthCheck resources, regional and global,
        # available to the specified project. To prevent failure, Google recommends that
        # you set the `returnPartialSuccess` parameter to `true`.
        # @param [String] project
        #   Name of the project scoping this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Boolean] include_all_scopes
        #   Indicates whether every visible scope for each scope type (zone, region,
        #   global) should be included in the response. For new resource types added after
        #   this field, the flag has no effect as new resource types will always include
        #   every visible scope for each scope type in response. For resource types which
        #   predate this field, if this flag is omitted or false, only scopes of the scope
        #   types where the resource type is expected to be found will be included.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [Fixnum] service_project_number
        #   The Shared VPC service project id or service project number for which
        #   aggregated list request is invoked for subnetworks list-usable api.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::HealthChecksAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::HealthChecksAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_health_check_list(project, filter: nil, include_all_scopes: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, service_project_number: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/aggregated/healthChecks', options)
          command.response_representation = Google::Apis::ComputeV1::HealthChecksAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::HealthChecksAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['includeAllScopes'] = include_all_scopes unless include_all_scopes.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['serviceProjectNumber'] = service_project_number unless service_project_number.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified HealthCheck resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] health_check
        #   Name of the HealthCheck resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_health_check(project, health_check, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/global/healthChecks/{healthCheck}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['healthCheck'] = health_check unless health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified HealthCheck resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] health_check
        #   Name of the HealthCheck resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::HealthCheck] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::HealthCheck]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_health_check(project, health_check, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/healthChecks/{healthCheck}', options)
          command.response_representation = Google::Apis::ComputeV1::HealthCheck::Representation
          command.response_class = Google::Apis::ComputeV1::HealthCheck
          command.params['project'] = project unless project.nil?
          command.params['healthCheck'] = health_check unless health_check.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a HealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::HealthCheck] health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_health_check(project, health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/healthChecks', options)
          command.request_representation = Google::Apis::ComputeV1::HealthCheck::Representation
          command.request_object = health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of HealthCheck resources available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::HealthCheckList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::HealthCheckList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_health_checks(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/healthChecks', options)
          command.response_representation = Google::Apis::ComputeV1::HealthCheckList::Representation
          command.response_class = Google::Apis::ComputeV1::HealthCheckList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HealthCheck resource in the specified project using the data
        # included in the request. This method supports PATCH semantics and uses the
        # JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] health_check
        #   Name of the HealthCheck resource to patch.
        # @param [Google::Apis::ComputeV1::HealthCheck] health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_health_check(project, health_check, health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, 'projects/{project}/global/healthChecks/{healthCheck}', options)
          command.request_representation = Google::Apis::ComputeV1::HealthCheck::Representation
          command.request_object = health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['healthCheck'] = health_check unless health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] health_check
        #   Name of the HealthCheck resource to update.
        # @param [Google::Apis::ComputeV1::HealthCheck] health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_health_check(project, health_check, health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:put, 'projects/{project}/global/healthChecks/{healthCheck}', options)
          command.request_representation = Google::Apis::ComputeV1::HealthCheck::Representation
          command.request_object = health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['healthCheck'] = health_check unless health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified HttpHealthCheck resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] http_health_check
        #   Name of the HttpHealthCheck resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_http_health_check(project, http_health_check, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/global/httpHealthChecks/{httpHealthCheck}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpHealthCheck'] = http_health_check unless http_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified HttpHealthCheck resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] http_health_check
        #   Name of the HttpHealthCheck resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::HttpHealthCheck] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::HttpHealthCheck]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_http_health_check(project, http_health_check, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/httpHealthChecks/{httpHealthCheck}', options)
          command.response_representation = Google::Apis::ComputeV1::HttpHealthCheck::Representation
          command.response_class = Google::Apis::ComputeV1::HttpHealthCheck
          command.params['project'] = project unless project.nil?
          command.params['httpHealthCheck'] = http_health_check unless http_health_check.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a HttpHealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::HttpHealthCheck] http_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_http_health_check(project, http_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/httpHealthChecks', options)
          command.request_representation = Google::Apis::ComputeV1::HttpHealthCheck::Representation
          command.request_object = http_health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of HttpHealthCheck resources available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::HttpHealthCheckList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::HttpHealthCheckList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_http_health_checks(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/httpHealthChecks', options)
          command.response_representation = Google::Apis::ComputeV1::HttpHealthCheckList::Representation
          command.response_class = Google::Apis::ComputeV1::HttpHealthCheckList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HttpHealthCheck resource in the specified project using the data
        # included in the request. This method supports PATCH semantics and uses the
        # JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] http_health_check
        #   Name of the HttpHealthCheck resource to patch.
        # @param [Google::Apis::ComputeV1::HttpHealthCheck] http_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_http_health_check(project, http_health_check, http_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, 'projects/{project}/global/httpHealthChecks/{httpHealthCheck}', options)
          command.request_representation = Google::Apis::ComputeV1::HttpHealthCheck::Representation
          command.request_object = http_health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpHealthCheck'] = http_health_check unless http_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HttpHealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] http_health_check
        #   Name of the HttpHealthCheck resource to update.
        # @param [Google::Apis::ComputeV1::HttpHealthCheck] http_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_http_health_check(project, http_health_check, http_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:put, 'projects/{project}/global/httpHealthChecks/{httpHealthCheck}', options)
          command.request_representation = Google::Apis::ComputeV1::HttpHealthCheck::Representation
          command.request_object = http_health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpHealthCheck'] = http_health_check unless http_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified HttpsHealthCheck resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] https_health_check
        #   Name of the HttpsHealthCheck resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_https_health_check(project, https_health_check, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/global/httpsHealthChecks/{httpsHealthCheck}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpsHealthCheck'] = https_health_check unless https_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified HttpsHealthCheck resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] https_health_check
        #   Name of the HttpsHealthCheck resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::HttpsHealthCheck] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::HttpsHealthCheck]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_https_health_check(project, https_health_check, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/httpsHealthChecks/{httpsHealthCheck}', options)
          command.response_representation = Google::Apis::ComputeV1::HttpsHealthCheck::Representation
          command.response_class = Google::Apis::ComputeV1::HttpsHealthCheck
          command.params['project'] = project unless project.nil?
          command.params['httpsHealthCheck'] = https_health_check unless https_health_check.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a HttpsHealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::HttpsHealthCheck] https_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_https_health_check(project, https_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/httpsHealthChecks', options)
          command.request_representation = Google::Apis::ComputeV1::HttpsHealthCheck::Representation
          command.request_object = https_health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of HttpsHealthCheck resources available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::HttpsHealthCheckList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::HttpsHealthCheckList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_https_health_checks(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/httpsHealthChecks', options)
          command.response_representation = Google::Apis::ComputeV1::HttpsHealthCheckList::Representation
          command.response_class = Google::Apis::ComputeV1::HttpsHealthCheckList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HttpsHealthCheck resource in the specified project using the data
        # included in the request. This method supports PATCH semantics and uses the
        # JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] https_health_check
        #   Name of the HttpsHealthCheck resource to patch.
        # @param [Google::Apis::ComputeV1::HttpsHealthCheck] https_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_https_health_check(project, https_health_check, https_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, 'projects/{project}/global/httpsHealthChecks/{httpsHealthCheck}', options)
          command.request_representation = Google::Apis::ComputeV1::HttpsHealthCheck::Representation
          command.request_object = https_health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpsHealthCheck'] = https_health_check unless https_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HttpsHealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] https_health_check
        #   Name of the HttpsHealthCheck resource to update.
        # @param [Google::Apis::ComputeV1::HttpsHealthCheck] https_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_https_health_check(project, https_health_check, https_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:put, 'projects/{project}/global/httpsHealthChecks/{httpsHealthCheck}', options)
          command.request_representation = Google::Apis::ComputeV1::HttpsHealthCheck::Representation
          command.request_object = https_health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpsHealthCheck'] = https_health_check unless https_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the latest image that is part of an image family, is not deprecated
        # and is rolled out in the specified zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] family
        #   Name of the image family to search for.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ImageFamilyView] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ImageFamilyView]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_image_family_view(project, zone, family, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/imageFamilyViews/{family}', options)
          command.response_representation = Google::Apis::ComputeV1::ImageFamilyView::Representation
          command.response_class = Google::Apis::ComputeV1::ImageFamilyView
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['family'] = family unless family.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified image.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] image
        #   Name of the image resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_image(project, image, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/global/images/{image}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['image'] = image unless image.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the deprecation status of an image. If an empty request body is given,
        # clears the deprecation status instead.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] image
        #   Image name.
        # @param [Google::Apis::ComputeV1::DeprecationStatus] deprecation_status_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def deprecate_image(project, image, deprecation_status_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/images/{image}/deprecate', options)
          command.request_representation = Google::Apis::ComputeV1::DeprecationStatus::Representation
          command.request_object = deprecation_status_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['image'] = image unless image.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified image.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] image
        #   Name of the image resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Image] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Image]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_image(project, image, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/images/{image}', options)
          command.response_representation = Google::Apis::ComputeV1::Image::Representation
          command.response_class = Google::Apis::ComputeV1::Image
          command.params['project'] = project unless project.nil?
          command.params['image'] = image unless image.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the latest image that is part of an image family and is not deprecated.
        # For more information on image families, see Public image families
        # documentation.
        # @param [String] project
        #   The image project that the image belongs to. For example, to get a CentOS
        #   image, specify centos-cloud as the image project.
        # @param [String] family
        #   Name of the image family to search for.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Image] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Image]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_image_from_family(project, family, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/images/family/{family}', options)
          command.response_representation = Google::Apis::ComputeV1::Image::Representation
          command.response_class = Google::Apis::ComputeV1::Image
          command.params['project'] = project unless project.nil?
          command.params['family'] = family unless family.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the access control policy for a resource. May be empty if no such policy
        # or resource exists.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Fixnum] options_requested_policy_version
        #   Requested IAM Policy version.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_image_iam_policy(project, resource, options_requested_policy_version: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/images/{resource}/getIamPolicy', options)
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['optionsRequestedPolicyVersion'] = options_requested_policy_version unless options_requested_policy_version.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an image in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::Image] image_object
        # @param [Boolean] force_create
        #   Force image creation if true.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_image(project, image_object = nil, force_create: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/images', options)
          command.request_representation = Google::Apis::ComputeV1::Image::Representation
          command.request_object = image_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['forceCreate'] = force_create unless force_create.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of custom images available to the specified project. Custom
        # images are images you create that belong to your project. This method does not
        # get any images that belong to other projects, including publicly-available
        # images, like Debian 8. If you want to get a list of publicly-available images,
        # use this method to make a request to the respective image project, such as
        # debian-cloud or windows-cloud.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ImageList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ImageList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_images(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/images', options)
          command.response_representation = Google::Apis::ComputeV1::ImageList::Representation
          command.response_class = Google::Apis::ComputeV1::ImageList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Patches the specified image with the data included in the request. Only the
        # following fields can be modified: family, description, deprecation status.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] image
        #   Name of the image resource to patch.
        # @param [Google::Apis::ComputeV1::Image] image_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_image(project, image, image_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, 'projects/{project}/global/images/{image}', options)
          command.request_representation = Google::Apis::ComputeV1::Image::Representation
          command.request_object = image_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['image'] = image unless image.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the access control policy on the specified resource. Replaces any
        # existing policy.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::GlobalSetPolicyRequest] global_set_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_image_iam_policy(project, resource, global_set_policy_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/images/{resource}/setIamPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalSetPolicyRequest::Representation
          command.request_object = global_set_policy_request_object
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on an image. To learn more about labels, read the Labeling
        # Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::GlobalSetLabelsRequest] global_set_labels_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_image_labels(project, resource, global_set_labels_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/images/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalSetLabelsRequest::Representation
          command.request_object = global_set_labels_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_image_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/images/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Cancels the specified resize request and removes it from the queue. Cancelled
        # resize request does no longer wait for the resources to be provisioned. Cancel
        # is only possible for requests that are accepted in the queue.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located. The name
        #   should conform to RFC1035.
        # @param [String] instance_group_manager
        #   The name of the managed instance group. The name should conform to RFC1035 or
        #   be a resource ID.
        # @param [String] resize_request
        #   The name of the resize request to cancel. The name should conform to RFC1035
        #   or be a resource ID.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def cancel_instance_group_manager_resize_request(project, zone, instance_group_manager, resize_request, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/resizeRequests/{resizeRequest}/cancel', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.params['resizeRequest'] = resize_request unless resize_request.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified, inactive resize request. Requests that are still active
        # cannot be deleted. Deleting request does not delete instances that were
        # provisioned previously.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located. The name
        #   should conform to RFC1035.
        # @param [String] instance_group_manager
        #   The name of the managed instance group. The name should conform to RFC1035 or
        #   be a resource ID.
        # @param [String] resize_request
        #   The name of the resize request to delete. The name should conform to RFC1035
        #   or be a resource ID.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_group_manager_resize_request(project, zone, instance_group_manager, resize_request, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/resizeRequests/{resizeRequest}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.params['resizeRequest'] = resize_request unless resize_request.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns all of the details about the specified resize request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the href="/compute/docs/regions-zones/#available">zone scoping this
        #   request. Name should conform to RFC1035.
        # @param [String] instance_group_manager
        #   The name of the managed instance group. Name should conform to RFC1035 or be a
        #   resource ID.
        # @param [String] resize_request
        #   The name of the resize request. Name should conform to RFC1035 or be a
        #   resource ID.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupManagerResizeRequest] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupManagerResizeRequest]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_group_manager_resize_request(project, zone, instance_group_manager, resize_request, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/resizeRequests/{resizeRequest}', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupManagerResizeRequest::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupManagerResizeRequest
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.params['resizeRequest'] = resize_request unless resize_request.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a new resize request that starts provisioning VMs immediately or
        # queues VM creation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located and where the
        #   resize request will be created. Name should conform to RFC1035.
        # @param [String] instance_group_manager
        #   The name of the managed instance group to which the resize request will be
        #   added. Name should conform to RFC1035 or be a resource ID.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagerResizeRequest] instance_group_manager_resize_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_instance_group_manager_resize_request(project, zone, instance_group_manager, instance_group_manager_resize_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/resizeRequests', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagerResizeRequest::Representation
          command.request_object = instance_group_manager_resize_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of resize requests that are contained in the managed instance
        # group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located. The name
        #   should conform to RFC1035.
        # @param [String] instance_group_manager
        #   The name of the managed instance group. The name should conform to RFC1035.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupManagerResizeRequestsListResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupManagerResizeRequestsListResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_group_manager_resize_requests(project, zone, instance_group_manager, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/resizeRequests', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupManagerResizeRequestsListResponse::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupManagerResizeRequestsListResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Flags the specified instances to be removed from the managed instance group.
        # Abandoning an instance does not delete the instance, but it does remove the
        # instance from any target pools that are applied by the managed instance group.
        # This method reduces the targetSize of the managed instance group by the number
        # of instances that you abandon. This operation is marked as DONE when the
        # action is scheduled even if the instances have not yet been removed from the
        # group. You must separately verify the status of the abandoning action with the
        # listmanagedinstances method. If the group is part of a backend service that
        # has enabled connection draining, it can take up to 60 seconds after the
        # connection draining duration has elapsed before the VM instance is removed or
        # deleted. You can specify a maximum of 1000 instances with this method per
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersAbandonInstancesRequest] instance_group_managers_abandon_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def abandon_instance_group_manager_instances(project, zone, instance_group_manager, instance_group_managers_abandon_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/abandonInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersAbandonInstancesRequest::Representation
          command.request_object = instance_group_managers_abandon_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of managed instance groups and groups them by zone. To
        # prevent failure, Google recommends that you set the `returnPartialSuccess`
        # parameter to `true`.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Boolean] include_all_scopes
        #   Indicates whether every visible scope for each scope type (zone, region,
        #   global) should be included in the response. For new resource types added after
        #   this field, the flag has no effect as new resource types will always include
        #   every visible scope for each scope type in response. For resource types which
        #   predate this field, if this flag is omitted or false, only scopes of the scope
        #   types where the resource type is expected to be found will be included.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [Fixnum] service_project_number
        #   The Shared VPC service project id or service project number for which
        #   aggregated list request is invoked for subnetworks list-usable api.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupManagerAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupManagerAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_instance_group_managers(project, filter: nil, include_all_scopes: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, service_project_number: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/aggregated/instanceGroupManagers', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupManagerAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupManagerAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['includeAllScopes'] = include_all_scopes unless include_all_scopes.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['serviceProjectNumber'] = service_project_number unless service_project_number.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Applies changes to selected instances on the managed instance group. This
        # method can be used to apply new overrides and/or new versions.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located. Should
        #   conform to RFC1035.
        # @param [String] instance_group_manager
        #   The name of the managed instance group, should conform to RFC1035.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersApplyUpdatesRequest] instance_group_managers_apply_updates_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def apply_instance_group_manager_updates_to_instances(project, zone, instance_group_manager, instance_group_managers_apply_updates_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/applyUpdatesToInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersApplyUpdatesRequest::Representation
          command.request_object = instance_group_managers_apply_updates_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates instances with per-instance configurations in this managed instance
        # group. Instances are created using the current instance template. The create
        # instances operation is marked DONE if the createInstances request is
        # successful. The underlying actions take additional time. You must separately
        # verify the status of the creating or actions with the listmanagedinstances
        # method.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located. It should
        #   conform to RFC1035.
        # @param [String] instance_group_manager
        #   The name of the managed instance group. It should conform to RFC1035.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersCreateInstancesRequest] instance_group_managers_create_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. The request ID must be a valid UUID with the exception that
        #   zero UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_instance_group_manager_instances(project, zone, instance_group_manager, instance_group_managers_create_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/createInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersCreateInstancesRequest::Representation
          command.request_object = instance_group_managers_create_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified managed instance group and all of the instances in that
        # group. Note that the instance group must not belong to a backend service. Read
        # Deleting an instance group for more information.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_group_manager(project, zone, instance_group_manager, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Flags the specified instances in the managed instance group for immediate
        # deletion. The instances are also removed from any target pools of which they
        # were a member. This method reduces the targetSize of the managed instance
        # group by the number of instances that you delete. This operation is marked as
        # DONE when the action is scheduled even if the instances are still being
        # deleted. You must separately verify the status of the deleting action with the
        # listmanagedinstances method. If the group is part of a backend service that
        # has enabled connection draining, it can take up to 60 seconds after the
        # connection draining duration has elapsed before the VM instance is removed or
        # deleted. You can specify a maximum of 1000 instances with this method per
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersDeleteInstancesRequest] instance_group_managers_delete_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_group_manager_instances(project, zone, instance_group_manager, instance_group_managers_delete_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/deleteInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersDeleteInstancesRequest::Representation
          command.request_object = instance_group_managers_delete_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes selected per-instance configurations for the managed instance group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located. It should
        #   conform to RFC1035.
        # @param [String] instance_group_manager
        #   The name of the managed instance group. It should conform to RFC1035.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersDeletePerInstanceConfigsReq] instance_group_managers_delete_per_instance_configs_req_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_group_manager_per_instance_configs(project, zone, instance_group_manager, instance_group_managers_delete_per_instance_configs_req_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/deletePerInstanceConfigs', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersDeletePerInstanceConfigsReq::Representation
          command.request_object = instance_group_managers_delete_per_instance_configs_req_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns all of the details about the specified managed instance group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupManager] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupManager]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_group_manager(project, zone, instance_group_manager, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupManager::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupManager
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a managed instance group using the information that you specify in the
        # request. After the group is created, instances in the group are created using
        # the specified instance template. This operation is marked as DONE when the
        # group is created even if the instances in the group have not yet been created.
        # You must separately verify the status of the individual instances with the
        # listmanagedinstances method. A managed instance group can have up to 1000 VM
        # instances per group. Please contact Cloud Support if you need an increase in
        # this limit.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where you want to create the managed instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroupManager] instance_group_manager_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_instance_group_manager(project, zone, instance_group_manager_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManager::Representation
          command.request_object = instance_group_manager_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of managed instance groups that are contained within the
        # specified project and zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupManagerList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupManagerList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_group_managers(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instanceGroupManagers', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupManagerList::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupManagerList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all errors thrown by actions on instances for a given managed instance
        # group. The filter and orderBy query parameters are not supported.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located. It should
        #   conform to RFC1035.
        # @param [String] instance_group_manager
        #   The name of the managed instance group. It must be a string that meets the
        #   requirements in RFC1035, or an unsigned long integer: must match regexp
        #   pattern: (?:[a-z](?:[-a-z0-9]`0,61`[a-z0-9])?)|1-9`0,19`.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupManagersListErrorsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupManagersListErrorsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_group_manager_errors(project, zone, instance_group_manager, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/listErrors', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupManagersListErrorsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupManagersListErrorsResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all of the instances in the managed instance group. Each instance in the
        # list has a currentAction, which indicates the action that the managed instance
        # group is performing on the instance. For example, if the group is still
        # creating an instance, the currentAction is CREATING. If a previous action
        # failed, the list displays the errors for that failed action. The orderBy query
        # parameter is not supported. The `pageToken` query parameter is supported only
        # if the group's `listManagedInstancesResults` field is set to `PAGINATED`.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupManagersListManagedInstancesResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupManagersListManagedInstancesResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_group_manager_managed_instances(project, zone, instance_group_manager, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/listManagedInstances', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupManagersListManagedInstancesResponse::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupManagersListManagedInstancesResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all of the per-instance configurations defined for the managed instance
        # group. The orderBy query parameter is not supported.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located. It should
        #   conform to RFC1035.
        # @param [String] instance_group_manager
        #   The name of the managed instance group. It should conform to RFC1035.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupManagersListPerInstanceConfigsResp] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupManagersListPerInstanceConfigsResp]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_group_manager_per_instance_configs(project, zone, instance_group_manager, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/listPerInstanceConfigs', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupManagersListPerInstanceConfigsResp::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupManagersListPerInstanceConfigsResp
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a managed instance group using the information that you specify in the
        # request. This operation is marked as DONE when the group is patched even if
        # the instances in the group are still in the process of being patched. You must
        # separately verify the status of the individual instances with the
        # listManagedInstances method. This method supports PATCH semantics and uses the
        # JSON merge patch format and processing rules. If you update your group to
        # specify a new template or instance configuration, it's possible that your
        # intended specification for each VM in the group is different from the current
        # state of that VM. To learn how to apply an updated configuration to the VMs in
        # a MIG, see Updating instances in a MIG.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where you want to create the managed instance group.
        # @param [String] instance_group_manager
        #   The name of the instance group manager.
        # @param [Google::Apis::ComputeV1::InstanceGroupManager] instance_group_manager_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_instance_group_manager(project, zone, instance_group_manager, instance_group_manager_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManager::Representation
          command.request_object = instance_group_manager_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Inserts or patches per-instance configurations for the managed instance group.
        # perInstanceConfig.name serves as a key used to distinguish whether to perform
        # insert or patch.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located. It should
        #   conform to RFC1035.
        # @param [String] instance_group_manager
        #   The name of the managed instance group. It should conform to RFC1035.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersPatchPerInstanceConfigsReq] instance_group_managers_patch_per_instance_configs_req_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_instance_group_manager_per_instance_configs(project, zone, instance_group_manager, instance_group_managers_patch_per_instance_configs_req_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/patchPerInstanceConfigs', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersPatchPerInstanceConfigsReq::Representation
          command.request_object = instance_group_managers_patch_per_instance_configs_req_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Flags the specified VM instances in the managed instance group to be
        # immediately recreated. Each instance is recreated using the group's current
        # configuration. This operation is marked as DONE when the flag is set even if
        # the instances have not yet been recreated. You must separately verify the
        # status of each instance by checking its currentAction field; for more
        # information, see Checking the status of managed instances. If the group is
        # part of a backend service that has enabled connection draining, it can take up
        # to 60 seconds after the connection draining duration has elapsed before the VM
        # instance is removed or deleted. You can specify a maximum of 1000 instances
        # with this method per request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersRecreateInstancesRequest] instance_group_managers_recreate_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def recreate_instance_group_manager_instances(project, zone, instance_group_manager, instance_group_managers_recreate_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/recreateInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersRecreateInstancesRequest::Representation
          command.request_object = instance_group_managers_recreate_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Resizes the managed instance group. If you increase the size, the group
        # creates new instances using the current instance template. If you decrease the
        # size, the group deletes instances. The resize operation is marked DONE when
        # the resize actions are scheduled even if the group has not yet added or
        # deleted any instances. You must separately verify the status of the creating
        # or deleting actions with the listmanagedinstances method. When resizing down,
        # the instance group arbitrarily chooses the order in which VMs are deleted. The
        # group takes into account some VM attributes when making the selection
        # including: + The status of the VM instance. + The health of the VM instance. +
        # The instance template version the VM is based on. + For regional managed
        # instance groups, the location of the VM instance. This list is subject to
        # change. If the group is part of a backend service that has enabled connection
        # draining, it can take up to 60 seconds after the connection draining duration
        # has elapsed before the VM instance is removed or deleted.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Fixnum] size
        #   The number of running instances that the managed instance group should
        #   maintain at any given time. The group automatically adds or removes instances
        #   to maintain the number of instances specified by this parameter.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def resize_instance_group_manager(project, zone, instance_group_manager, size, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/resize', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['size'] = size unless size.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Flags the specified instances in the managed instance group to be resumed.
        # This method increases the targetSize and decreases the targetSuspendedSize of
        # the managed instance group by the number of instances that you resume. The
        # resumeInstances operation is marked DONE if the resumeInstances request is
        # successful. The underlying actions take additional time. You must separately
        # verify the status of the RESUMING action with the listmanagedinstances method.
        # In this request, you can only specify instances that are suspended. For
        # example, if an instance was previously suspended using the suspendInstances
        # method, it can be resumed using the resumeInstances method. If a health check
        # is attached to the managed instance group, the specified instances will be
        # verified as healthy after they are resumed. You can specify a maximum of 1000
        # instances with this method per request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersResumeInstancesRequest] instance_group_managers_resume_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def resume_instance_group_manager_instances(project, zone, instance_group_manager, instance_group_managers_resume_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/resumeInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersResumeInstancesRequest::Representation
          command.request_object = instance_group_managers_resume_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Specifies the instance template to use when creating new instances in this
        # group. The templates for existing instances in the group do not change unless
        # you run recreateInstances, run applyUpdatesToInstances, or set the group's
        # updatePolicy.type to PROACTIVE.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersSetInstanceTemplateRequest] instance_group_managers_set_instance_template_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_group_manager_instance_template(project, zone, instance_group_manager, instance_group_managers_set_instance_template_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/setInstanceTemplate', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersSetInstanceTemplateRequest::Representation
          command.request_object = instance_group_managers_set_instance_template_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Modifies the target pools to which all instances in this managed instance
        # group are assigned. The target pools automatically apply to all of the
        # instances in the managed instance group. This operation is marked DONE when
        # you make the request even if the instances have not yet been added to their
        # target pools. The change might take some time to apply to all of the instances
        # in the group depending on the size of the group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersSetTargetPoolsRequest] instance_group_managers_set_target_pools_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_group_manager_target_pools(project, zone, instance_group_manager, instance_group_managers_set_target_pools_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/setTargetPools', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersSetTargetPoolsRequest::Representation
          command.request_object = instance_group_managers_set_target_pools_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Flags the specified instances in the managed instance group to be started.
        # This method increases the targetSize and decreases the targetStoppedSize of
        # the managed instance group by the number of instances that you start. The
        # startInstances operation is marked DONE if the startInstances request is
        # successful. The underlying actions take additional time. You must separately
        # verify the status of the STARTING action with the listmanagedinstances method.
        # In this request, you can only specify instances that are stopped. For example,
        # if an instance was previously stopped using the stopInstances method, it can
        # be started using the startInstances method. If a health check is attached to
        # the managed instance group, the specified instances will be verified as
        # healthy after they are started. You can specify a maximum of 1000 instances
        # with this method per request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersStartInstancesRequest] instance_group_managers_start_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def start_instance_group_manager_instances(project, zone, instance_group_manager, instance_group_managers_start_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/startInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersStartInstancesRequest::Representation
          command.request_object = instance_group_managers_start_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Flags the specified instances in the managed instance group to be immediately
        # stopped. You can only specify instances that are running in this request. This
        # method reduces the targetSize and increases the targetStoppedSize of the
        # managed instance group by the number of instances that you stop. The
        # stopInstances operation is marked DONE if the stopInstances request is
        # successful. The underlying actions take additional time. You must separately
        # verify the status of the STOPPING action with the listmanagedinstances method.
        # If the standbyPolicy.initialDelaySec field is set, the group delays stopping
        # the instances until initialDelaySec have passed from instance.
        # creationTimestamp (that is, when the instance was created). This delay gives
        # your application time to set itself up and initialize on the instance. If more
        # than initialDelaySec seconds have passed since instance.creationTimestamp when
        # this method is called, there will be zero delay. If the group is part of a
        # backend service that has enabled connection draining, it can take up to 60
        # seconds after the connection draining duration has elapsed before the VM
        # instance is stopped. Stopped instances can be started using the startInstances
        # method. You can specify a maximum of 1000 instances with this method per
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersStopInstancesRequest] instance_group_managers_stop_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def stop_instance_group_manager_instances(project, zone, instance_group_manager, instance_group_managers_stop_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/stopInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersStopInstancesRequest::Representation
          command.request_object = instance_group_managers_stop_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Flags the specified instances in the managed instance group to be immediately
        # suspended. You can only specify instances that are running in this request.
        # This method reduces the targetSize and increases the targetSuspendedSize of
        # the managed instance group by the number of instances that you suspend. The
        # suspendInstances operation is marked DONE if the suspendInstances request is
        # successful. The underlying actions take additional time. You must separately
        # verify the status of the SUSPENDING action with the listmanagedinstances
        # method. If the standbyPolicy.initialDelaySec field is set, the group delays
        # suspension of the instances until initialDelaySec have passed from instance.
        # creationTimestamp (that is, when the instance was created). This delay gives
        # your application time to set itself up and initialize on the instance. If more
        # than initialDelaySec seconds have passed since instance.creationTimestamp when
        # this method is called, there will be zero delay. If the group is part of a
        # backend service that has enabled connection draining, it can take up to 60
        # seconds after the connection draining duration has elapsed before the VM
        # instance is suspended. Suspended instances can be resumed using the
        # resumeInstances method. You can specify a maximum of 1000 instances with this
        # method per request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersSuspendInstancesRequest] instance_group_managers_suspend_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def suspend_instance_group_manager_instances(project, zone, instance_group_manager, instance_group_managers_suspend_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/suspendInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersSuspendInstancesRequest::Representation
          command.request_object = instance_group_managers_suspend_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Inserts or updates per-instance configurations for the managed instance group.
        # perInstanceConfig.name serves as a key used to distinguish whether to perform
        # insert or patch.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located. It should
        #   conform to RFC1035.
        # @param [String] instance_group_manager
        #   The name of the managed instance group. It should conform to RFC1035.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersUpdatePerInstanceConfigsReq] instance_group_managers_update_per_instance_configs_req_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_instance_group_manager_per_instance_configs(project, zone, instance_group_manager, instance_group_managers_update_per_instance_configs_req_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/updatePerInstanceConfigs', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersUpdatePerInstanceConfigsReq::Representation
          command.request_object = instance_group_managers_update_per_instance_configs_req_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds a list of instances to the specified instance group. All of the instances
        # in the instance group must be in the same network/subnetwork. Read Adding
        # instances for more information.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group where you are adding instances.
        # @param [Google::Apis::ComputeV1::InstanceGroupsAddInstancesRequest] instance_groups_add_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_instance_group_instances(project, zone, instance_group, instance_groups_add_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroups/{instanceGroup}/addInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupsAddInstancesRequest::Representation
          command.request_object = instance_groups_add_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of instance groups and sorts them by zone. To prevent
        # failure, Google recommends that you set the `returnPartialSuccess` parameter
        # to `true`.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Boolean] include_all_scopes
        #   Indicates whether every visible scope for each scope type (zone, region,
        #   global) should be included in the response. For new resource types added after
        #   this field, the flag has no effect as new resource types will always include
        #   every visible scope for each scope type in response. For resource types which
        #   predate this field, if this flag is omitted or false, only scopes of the scope
        #   types where the resource type is expected to be found will be included.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [Fixnum] service_project_number
        #   The Shared VPC service project id or service project number for which
        #   aggregated list request is invoked for subnetworks list-usable api.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_instance_groups(project, filter: nil, include_all_scopes: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, service_project_number: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/aggregated/instanceGroups', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['includeAllScopes'] = include_all_scopes unless include_all_scopes.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['serviceProjectNumber'] = service_project_number unless service_project_number.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified instance group. The instances in the group are not
        # deleted. Note that instance group must not belong to a backend service. Read
        # Deleting an instance group for more information.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_group(project, zone, instance_group, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/zones/{zone}/instanceGroups/{instanceGroup}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified zonal instance group. Get a list of available zonal
        # instance groups by making a list() request. For managed instance groups, use
        # the instanceGroupManagers or regionInstanceGroupManagers methods instead.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroup] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroup]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_group(project, zone, instance_group, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instanceGroups/{instanceGroup}', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroup::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroup
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an instance group in the specified project using the parameters that
        # are included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where you want to create the instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroup] instance_group_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_instance_group(project, zone, instance_group_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroups', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroup::Representation
          command.request_object = instance_group_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of zonal instance group resources contained within the
        # specified zone. For managed instance groups, use the instanceGroupManagers or
        # regionInstanceGroupManagers methods instead.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_groups(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instanceGroups', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupList::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists the instances in the specified instance group. The orderBy query
        # parameter is not supported. The filter query parameter is supported, but only
        # for expressions that use `eq` (equal) or `ne` (not equal) operators.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group from which you want to generate a list of
        #   included instances.
        # @param [Google::Apis::ComputeV1::InstanceGroupsListInstancesRequest] instance_groups_list_instances_request_object
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupsListInstances] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupsListInstances]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_group_instances(project, zone, instance_group, instance_groups_list_instances_request_object = nil, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroups/{instanceGroup}/listInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupsListInstancesRequest::Representation
          command.request_object = instance_groups_list_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupsListInstances::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupsListInstances
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Removes one or more instances from the specified instance group, but does not
        # delete those instances. If the group is part of a backend service that has
        # enabled connection draining, it can take up to 60 seconds after the connection
        # draining duration before the VM instance is removed or deleted.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group where the specified instances will be removed.
        # @param [Google::Apis::ComputeV1::InstanceGroupsRemoveInstancesRequest] instance_groups_remove_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def remove_instance_group_instances(project, zone, instance_group, instance_groups_remove_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroups/{instanceGroup}/removeInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupsRemoveInstancesRequest::Representation
          command.request_object = instance_groups_remove_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the named ports for the specified instance group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group where the named ports are updated.
        # @param [Google::Apis::ComputeV1::InstanceGroupsSetNamedPortsRequest] instance_groups_set_named_ports_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_group_named_ports(project, zone, instance_group, instance_groups_set_named_ports_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instanceGroups/{instanceGroup}/setNamedPorts', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupsSetNamedPortsRequest::Representation
          command.request_object = instance_groups_set_named_ports_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Get Instance settings.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceSettings] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceSettings]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_setting(project, zone, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instanceSettings', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceSettings::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceSettings
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Patch Instance settings
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The zone scoping this request. It should conform to RFC1035.
        # @param [Google::Apis::ComputeV1::InstanceSettings] instance_settings_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] update_mask
        #   update_mask indicates fields to be updated as part of this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_instance_setting(project, zone, instance_settings_object = nil, request_id: nil, update_mask: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, 'projects/{project}/zones/{zone}/instanceSettings', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceSettings::Representation
          command.request_object = instance_settings_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['updateMask'] = update_mask unless update_mask.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of all InstanceTemplates resources, regional and global,
        # available to the specified project. To prevent failure, Google recommends that
        # you set the `returnPartialSuccess` parameter to `true`.
        # @param [String] project
        #   Name of the project scoping this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Boolean] include_all_scopes
        #   Indicates whether every visible scope for each scope type (zone, region,
        #   global) should be included in the response. For new resource types added after
        #   this field, the flag has no effect as new resource types will always include
        #   every visible scope for each scope type in response. For resource types which
        #   predate this field, if this flag is omitted or false, only scopes of the scope
        #   types where the resource type is expected to be found will be included.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [Fixnum] service_project_number
        #   The Shared VPC service project id or service project number for which
        #   aggregated list request is invoked for subnetworks list-usable api.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceTemplateAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceTemplateAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_instance_template_list(project, filter: nil, include_all_scopes: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, service_project_number: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/aggregated/instanceTemplates', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceTemplateAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceTemplateAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['includeAllScopes'] = include_all_scopes unless include_all_scopes.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['serviceProjectNumber'] = service_project_number unless service_project_number.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified instance template. Deleting an instance template is
        # permanent and cannot be undone. It is not possible to delete templates that
        # are already in use by a managed instance group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] instance_template
        #   The name of the instance template to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_template(project, instance_template, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/global/instanceTemplates/{instanceTemplate}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['instanceTemplate'] = instance_template unless instance_template.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified instance template.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] instance_template
        #   The name of the instance template.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceTemplate] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceTemplate]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_template(project, instance_template, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/instanceTemplates/{instanceTemplate}', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceTemplate::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceTemplate
          command.params['project'] = project unless project.nil?
          command.params['instanceTemplate'] = instance_template unless instance_template.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the access control policy for a resource. May be empty if no such policy
        # or resource exists.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Fixnum] options_requested_policy_version
        #   Requested IAM Policy version.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_template_iam_policy(project, resource, options_requested_policy_version: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/instanceTemplates/{resource}/getIamPolicy', options)
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['optionsRequestedPolicyVersion'] = options_requested_policy_version unless options_requested_policy_version.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an instance template in the specified project using the data that is
        # included in the request. If you are creating a new template to update an
        # existing instance group, your new instance template must use the same network
        # or, if applicable, the same subnetwork as the original template.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::InstanceTemplate] instance_template_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_instance_template(project, instance_template_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/instanceTemplates', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceTemplate::Representation
          command.request_object = instance_template_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of instance templates that are contained within the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceTemplateList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceTemplateList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_templates(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/global/instanceTemplates', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceTemplateList::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceTemplateList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the access control policy on the specified resource. Replaces any
        # existing policy.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::GlobalSetPolicyRequest] global_set_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_template_iam_policy(project, resource, global_set_policy_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/instanceTemplates/{resource}/setIamPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalSetPolicyRequest::Representation
          command.request_object = global_set_policy_request_object
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_instance_template_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/global/instanceTemplates/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds an access config to an instance's network interface.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [String] network_interface
        #   The name of the network interface to add to this instance.
        # @param [Google::Apis::ComputeV1::AccessConfig] access_config_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_instance_access_config(project, zone, instance, network_interface, access_config_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/addAccessConfig', options)
          command.request_representation = Google::Apis::ComputeV1::AccessConfig::Representation
          command.request_object = access_config_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['networkInterface'] = network_interface unless network_interface.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds existing resource policies to an instance. You can only add one policy
        # right now which will be applied to this instance for scheduling live
        # migrations.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [Google::Apis::ComputeV1::InstancesAddResourcePoliciesRequest] instances_add_resource_policies_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_instance_resource_policies(project, zone, instance, instances_add_resource_policies_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/addResourcePolicies', options)
          command.request_representation = Google::Apis::ComputeV1::InstancesAddResourcePoliciesRequest::Representation
          command.request_object = instances_add_resource_policies_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of all of the instances in your project across
        # all regions and zones. The performance of this method degrades when a filter
        # is specified on a project that has a very large number of instances. To
        # prevent failure, Google recommends that you set the `returnPartialSuccess`
        # parameter to `true`.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Boolean] include_all_scopes
        #   Indicates whether every visible scope for each scope type (zone, region,
        #   global) should be included in the response. For new resource types added after
        #   this field, the flag has no effect as new resource types will always include
        #   every visible scope for each scope type in response. For resource types which
        #   predate this field, if this flag is omitted or false, only scopes of the scope
        #   types where the resource type is expected to be found will be included.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [Fixnum] service_project_number
        #   The Shared VPC service project id or service project number for which
        #   aggregated list request is invoked for subnetworks list-usable api.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_instances(project, filter: nil, include_all_scopes: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, service_project_number: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/aggregated/instances', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['includeAllScopes'] = include_all_scopes unless include_all_scopes.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['serviceProjectNumber'] = service_project_number unless service_project_number.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Attaches an existing Disk resource to an instance. You must first create the
        # disk before you can attach it. It is not possible to create and attach a disk
        # at the same time. For more information, read Adding a persistent disk to your
        # instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [Google::Apis::ComputeV1::AttachedDisk] attached_disk_object
        # @param [Boolean] force_attach
        #   Whether to force attach the regional disk even if it's currently attached to
        #   another instance. If you try to force attach a zonal disk to an instance, you
        #   will receive an error.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def attach_disk(project, zone, instance, attached_disk_object = nil, force_attach: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/attachDisk', options)
          command.request_representation = Google::Apis::ComputeV1::AttachedDisk::Representation
          command.request_object = attached_disk_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['forceAttach'] = force_attach unless force_attach.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates multiple instances. Count specifies the number of instances to create.
        # For more information, see About bulk creation of VMs.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [Google::Apis::ComputeV1::BulkInsertInstanceResource] bulk_insert_instance_resource_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def bulk_instance_insert(project, zone, bulk_insert_instance_resource_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/bulkInsert', options)
          command.request_representation = Google::Apis::ComputeV1::BulkInsertInstanceResource::Representation
          command.request_object = bulk_insert_instance_resource_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified Instance resource. For more information, see Deleting an
        # instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance(project, zone, instance, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, 'projects/{project}/zones/{zone}/instances/{instance}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes an access config from an instance's network interface.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [String] access_config
        #   The name of the access config to delete.
        # @param [String] network_interface
        #   The name of the network interface.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_access_config(project, zone, instance, access_config, network_interface, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/deleteAccessConfig', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['accessConfig'] = access_config unless access_config.nil?
          command.query['networkInterface'] = network_interface unless network_interface.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Detaches a disk from an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Instance name for this request.
        # @param [String] device_name
        #   The device name of the disk to detach. Make a get() request on the instance to
        #   view currently attached disks and device names.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def detach_disk(project, zone, instance, device_name, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/detachDisk', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['deviceName'] = device_name unless device_name.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified Instance resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Instance] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Instance]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance(project, zone, instance, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instances/{instance}', options)
          command.response_representation = Google::Apis::ComputeV1::Instance::Representation
          command.response_class = Google::Apis::ComputeV1::Instance
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns effective firewalls applied to an interface of the instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [String] network_interface
        #   The name of the network interface to get the effective firewalls.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstancesGetEffectiveFirewallsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstancesGetEffectiveFirewallsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_effective_firewalls(project, zone, instance, network_interface, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instances/{instance}/getEffectiveFirewalls', options)
          command.response_representation = Google::Apis::ComputeV1::InstancesGetEffectiveFirewallsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::InstancesGetEffectiveFirewallsResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['networkInterface'] = network_interface unless network_interface.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified guest attributes entry.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [String] query_path
        #   Specifies the guest attributes path to be queried.
        # @param [String] variable_key
        #   Specifies the key for the guest attributes entry.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::GuestAttributes] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::GuestAttributes]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_guest_attributes(project, zone, instance, query_path: nil, variable_key: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instances/{instance}/getGuestAttributes', options)
          command.response_representation = Google::Apis::ComputeV1::GuestAttributes::Representation
          command.response_class = Google::Apis::ComputeV1::GuestAttributes
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['queryPath'] = query_path unless query_path.nil?
          command.query['variableKey'] = variable_key unless variable_key.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the access control policy for a resource. May be empty if no such policy
        # or resource exists.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Fixnum] options_requested_policy_version
        #   Requested IAM Policy version.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_iam_policy(project, zone, resource, options_requested_policy_version: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instances/{resource}/getIamPolicy', options)
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['optionsRequestedPolicyVersion'] = options_requested_policy_version unless options_requested_policy_version.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the screenshot from the specified instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Screenshot] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Screenshot]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_screenshot(project, zone, instance, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instances/{instance}/screenshot', options)
          command.response_representation = Google::Apis::ComputeV1::Screenshot::Representation
          command.response_class = Google::Apis::ComputeV1::Screenshot
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the last 1 MB of serial port output from the specified instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance for this request.
        # @param [Fixnum] port
        #   Specifies which COM or serial port to retrieve data from.
        # @param [Fixnum] start
        #   Specifies the starting byte position of the output to return. To start with
        #   the first byte of output to the specified port, omit this field or set it to `
        #   0`. If the output for that byte position is available, this field matches the `
        #   start` parameter sent with the request. If the amount of serial console output
        #   exceeds the size of the buffer (1 MB), the oldest output is discarded and is
        #   no longer available. If the requested start position refers to discarded
        #   output, the start position is adjusted to the oldest output still available,
        #   and the adjusted start position is returned as the `start` property value. You
        #   can also provide a negative start position, which translates to the most
        #   recent number of bytes written to the serial port. For example, -3 is
        #   interpreted as the most recent 3 bytes written to the serial console. Note
        #   that the negative start is bounded by the retained buffer size, and the
        #   returned serial console output will not exceed the max buffer size.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::SerialPortOutput] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::SerialPortOutput]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_serial_port_output(project, zone, instance, port: nil, start: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instances/{instance}/serialPort', options)
          command.response_representation = Google::Apis::ComputeV1::SerialPortOutput::Representation
          command.response_class = Google::Apis::ComputeV1::SerialPortOutput
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['port'] = port unless port.nil?
          command.query['start'] = start unless start.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the Shielded Instance Identity of an instance
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name or id of the instance scoping this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ShieldedInstanceIdentity] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ShieldedInstanceIdentity]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_shielded_instance_identity(project, zone, instance, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instances/{instance}/getShieldedInstanceIdentity', options)
          command.response_representation = Google::Apis::ComputeV1::ShieldedInstanceIdentity::Representation
          command.response_class = Google::Apis::ComputeV1::ShieldedInstanceIdentity
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an instance resource in the specified project using the data included
        # in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [Google::Apis::ComputeV1::Instance] instance_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] source_instance_template
        #   Specifies instance template to create the instance. This field is optional. It
        #   can be a full or partial URL. For example, the following are all valid URLs to
        #   an instance template: - https://www.googleapis.com/compute/v1/projects/project
        #   /global/instanceTemplates/instanceTemplate - projects/project/global/
        #   instanceTemplates/instanceTemplate - global/instanceTemplates/instanceTemplate
        # @param [String] source_machine_image
        #   Specifies the machine image to use to create the instance. This field is
        #   optional. It can be a full or partial URL. For example, the following are all
        #   valid URLs to a machine image: - https://www.googleapis.com/compute/v1/
        #   projects/project/global/global /machineImages/machineImage - projects/project/
        #   global/global/machineImages/machineImage - global/machineImages/machineImage
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_instance(project, zone, instance_object = nil, request_id: nil, source_instance_template: nil, source_machine_image: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances', options)
          command.request_representation = Google::Apis::ComputeV1::Instance::Representation
          command.request_object = instance_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['sourceInstanceTemplate'] = source_instance_template unless source_instance_template.nil?
          command.query['sourceMachineImage'] = source_machine_image unless source_machine_image.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of instances contained within the specified zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instances(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instances', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceList::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of resources that refer to the VM instance specified in the
        # request. For example, if the VM instance is part of a managed or unmanaged
        # instance group, the referrers list includes the instance group. For more
        # information, read Viewing referrers to VM instances.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the target instance scoping this request, or '-' if the request should
        #   span over all instances in the container.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. Most
        #   Compute resources support two types of filter expressions: expressions that
        #   support regular expressions and expressions that follow API improvement
        #   proposal AIP-160. These two types of filter expressions cannot be mixed in one
        #   request. If you want to use AIP-160, your expression must specify the field
        #   name, an operator, and the value that you want to use for filtering. The value
        #   must be a string, a number, or a boolean. The operator must be either `=`, `!=`
        #   , `>`, `<`, `<=`, `>=` or `:`. For example, if you are filtering Compute
        #   Engine instances, you can exclude instances named `example-instance` by
        #   specifying `name != example-instance`. The `:*` comparison can be used to test
        #   whether a key has been defined. For example, to find all objects with `owner`
        #   label use: ``` labels.owner:* ``` You can also filter nested fields. For
        #   example, you could specify `scheduling.automaticRestart = false` to include
        #   instances only if they are not scheduled for automatic restarts. You can use
        #   filtering on nested fields to filter based on resource labels. To filter on
        #   multiple expressions, provide each separate expression within parentheses. For
        #   example: ``` (scheduling.automaticRestart = true) (cpuPlatform = "Intel
        #   Skylake") ``` By default, each expression is an `AND` expression. However, you
        #   can include `AND` and `OR` expressions explicitly. For example: ``` (
        #   cpuPlatform = "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (
        #   scheduling.automaticRestart = true) ``` If you want to use a regular
        #   expression, use the `eq` (equal) or `ne` (not equal) operator against a single
        #   un-parenthesized expression with or without quotes or against multiple
        #   parenthesized expressions. Examples: `fieldname eq unquoted literal` `
        #   fieldname eq 'single quoted literal'` `fieldname eq "double quoted literal"` `(
        #   fieldname1 eq literal) (fieldname2 ne "literal")` The literal value is
        #   interpreted as a regular expression using Google RE2 library syntax. The
        #   literal value must match the entire field. For example, to filter for
        #   instances that do not end with name "instance", you would use `name ne .*
        #   instance`. You cannot combine constraints on multiple fields using regular
        #   expressions.
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than `maxResults`, Compute Engine returns a `
        #   nextPageToken` that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are `0` to `500`, inclusive. (Default: `500`)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name. You can also sort results in
        #   descending order based on the creation timestamp using `orderBy="
        #   creationTimestamp desc"`. This sorts results based on the `creationTimestamp`
        #   field in reverse chronological order (newest result first). Use this to sort
        #   resources like operations so that the newest operation is returned first.
        #   Currently, only sorting by `name` or `creationTimestamp desc` is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set `pageToken` to the `nextPageToken` returned
        #   by a previous list request to get the next page of results.
        # @param [Boolean] return_partial_success
        #   Opt-in for partial success behavior which provides partial results in case of
        #   failure. The default value is false. For example, when partial success
        #   behavior is enabled, aggregatedList for a single zone scope either returns all
        #   resources in the zone or no resources, with an error code.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceListReferrers] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceListReferrers]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_referrers(project, zone, instance, filter: nil, max_results: nil, order_by: nil, page_token: nil, return_partial_success: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, 'projects/{project}/zones/{zone}/instances/{instance}/referrers', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceListReferrers::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceListReferrers
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['returnPartialSuccess'] = return_partial_success unless return_partial_success.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Perform a manual maintenance on the instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def perform_instance_maintenance(project, zone, instance, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/performMaintenance', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Removes resource policies from an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [Google::Apis::ComputeV1::InstancesRemoveResourcePoliciesRequest] instances_remove_resource_policies_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def remove_instance_resource_policies(project, zone, instance, instances_remove_resource_policies_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/removeResourcePolicies', options)
          command.request_representation = Google::Apis::ComputeV1::InstancesRemoveResourcePoliciesRequest::Representation
          command.request_object = instances_remove_resource_policies_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Mark the host as faulty and try to restart the instance on a new host.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeV1::InstancesReportHostAsFaultyRequest] instances_report_host_as_faulty_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def report_instance_host_as_faulty(project, zone, instance, instances_report_host_as_faulty_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/reportHostAsFaulty', options)
          command.request_representation = Google::Apis::ComputeV1::InstancesReportHostAsFaultyRequest::Representation
          command.request_object = instances_report_host_as_faulty_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Performs a reset on the instance. This is a hard reset. The VM does not do a
        # graceful shutdown. For more information, see Resetting an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def reset_instance(project, zone, instance, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/reset', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Resumes an instance that was suspended using the instances().suspend method.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance resource to resume.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def resume_instance(project, zone, instance, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/resume', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sends diagnostic interrupt to the instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [NilClass] No result returned for this method
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [void]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def send_instance_diagnostic_interrupt(project, zone, instance, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/sendDiagnosticInterrupt', options)
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets deletion protection on the instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Boolean] deletion_protection
        #   Whether the resource should be protected against deletion.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_deletion_protection(project, zone, resource, deletion_protection: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{resource}/setDeletionProtection', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['deletionProtection'] = deletion_protection unless deletion_protection.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the auto-delete flag for a disk attached to an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [Boolean] auto_delete
        #   Whether to auto-delete the disk when the instance is deleted.
        # @param [String] device_name
        #   The device name of the disk to modify. Make a get() request on the instance to
        #   view currently attached disks and device names.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_disk_auto_delete(project, zone, instance, auto_delete, device_name, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/setDiskAutoDelete', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['autoDelete'] = auto_delete unless auto_delete.nil?
          command.query['deviceName'] = device_name unless device_name.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the access control policy on the specified resource. Replaces any
        # existing policy.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::ZoneSetPolicyRequest] zone_set_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_iam_policy(project, zone, resource, zone_set_policy_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{resource}/setIamPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::ZoneSetPolicyRequest::Representation
          command.request_object = zone_set_policy_request_object
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets labels on an instance. To learn more about labels, read the Labeling
        # Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeV1::InstancesSetLabelsRequest] instances_set_labels_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_labels(project, zone, instance, instances_set_labels_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/setLabels', options)
          command.request_representation = Google::Apis::ComputeV1::InstancesSetLabelsRequest::Representation
          command.request_object = instances_set_labels_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Changes the number and/or type of accelerator for a stopped instance to the
        # values specified in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeV1::InstancesSetMachineResourcesRequest] instances_set_machine_resources_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_machine_resources(project, zone, instance, instances_set_machine_resources_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/setMachineResources', options)
          command.request_representation = Google::Apis::ComputeV1::InstancesSetMachineResourcesRequest::Representation
          command.request_object = instances_set_machine_resources_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Changes the machine type for a stopped instance to the machine type specified
        # in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeV1::InstancesSetMachineTypeRequest] instances_set_machine_type_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_machine_type(project, zone, instance, instances_set_machine_type_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/setMachineType', options)
          command.request_representation = Google::Apis::ComputeV1::InstancesSetMachineTypeRequest::Representation
          command.request_object = instances_set_machine_type_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets metadata for the specified instance to the data included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeV1::Metadata] metadata_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_metadata(project, zone, instance, metadata_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/setMetadata', options)
          command.request_representation = Google::Apis::ComputeV1::Metadata::Representation
          command.request_object = metadata_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Changes the minimum CPU platform that this instance should use. This method
        # can only be called on a stopped instance. For more information, read
        # Specifying a Minimum CPU Platform.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeV1::InstancesSetMinCpuPlatformRequest] instances_set_min_cpu_platform_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_min_cpu_platform(project, zone, instance, instances_set_min_cpu_platform_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/setMinCpuPlatform', options)
          command.request_representation = Google::Apis::ComputeV1::InstancesSetMinCpuPlatformRequest::Representation
          command.request_object = instances_set_min_cpu_platform_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets name of an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [Google::Apis::ComputeV1::InstancesSetNameRequest] instances_set_name_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_name(project, zone, instance, instances_set_name_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/setName', options)
          command.request_representation = Google::Apis::ComputeV1::InstancesSetNameRequest::Representation
          command.request_object = instances_set_name_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets an instance's scheduling options. You can only call this method on a
        # stopped instance, that is, a VM instance that is in a `TERMINATED` state. See
        # Instance Life Cycle for more information on the possible instance states. For
        # more information about setting scheduling options for a VM, see Set VM host
        # maintenance policy.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Instance name for this request.
        # @param [Google::Apis::ComputeV1::Scheduling] scheduling_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_scheduling(project, zone, instance, scheduling_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/setScheduling', options)
          command.request_representation = Google::Apis::ComputeV1::Scheduling::Representation
          command.request_object = scheduling_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the Google Cloud Armor security policy for the specified instance. For
        # more information, see Google Cloud Armor Overview
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone scoping this request.
        # @param [String] instance
        #   Name of the Instance resource to which the security policy should be set. The
        #   name should conform to RFC1035.
        # @param [Google::Apis::ComputeV1::InstancesSetSecurityPolicyRequest] instances_set_security_policy_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed. For example, consider a situation
        #   where you make an initial request and the request times out. If you make the
        #   request again with the same request ID, the server can check if original
        #   operation with the same request ID was received, and if so, will ignore the
        #   second request. This prevents clients from accidentally creating duplicate
        #   commitments. The request ID must be a valid UUID with the exception that zero
        #   UUID is not supported ( 00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   Available to use for quota purposes for server-side applications. Can be any
        #   arbitrary string assigned to a user, but should not exceed 40 characters.
        # @param [String] user_ip
        #   Legacy name for parameter that has been superseded by `quotaUser`.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_security_policy(project, zone, instance, instances_set_security_policy_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, 'projects/{project}/zones/{zone}/instances/{instance}/setSecurityPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::InstancesSetSecurityPolicyRequest::Representation
          command.request_object = instances_set_security_policy_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the service account on the instance. For more information, read Changing
        # the service account and access scopes for an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance resource to start.
        # @param [Google::Apis::ComputeV1::InstancesSetServiceAccountRequest] instances_set_service_account_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your req