require "helper"
require "aarch64/parser"

class ParserTest < AArch64::Test

  def test_parse_error
    error = assert_raises(Racc::ParseError) do
      parse "mov x1, x0\nmov x1, #2 // comments not supported"
    end
    assert_equal 'parse error on value "/" (error) on line 2 at pos 23/48', error.message
  end

  def test_parse
    assert_bytes "movz x0, 0xCAFE", [0xc0, 0x5f, 0x99, 0xd2]
    assert_bytes "movz x0, #0xcafe", [0xc0, 0x5f, 0x99, 0xd2]
    assert_bytes "movz x0, #0xcafe, lsl #16", [0xc0, 0x5f, 0xb9, 0xd2]
    assert_bytes "movz w0, #0xcafe", [0xc0, 0x5f, 0x99, 0x52]
    assert_bytes "movz w0, #0xcafe, lsl #16", [0xc0, 0x5f, 0xb9, 0x52]
  end

  def test_autda
    # autda x1, sp
    assert_bytes "autda x1, sp", [0xe1, 0x1b, 0xc1, 0xda]
    assert_bytes "autda x1, x2", [0x41, 0x18, 0xc1, 0xda]
  end

  def parse input
    parser = AArch64::Parser.new
    begin
      asm = parser.parse input
    rescue Racc::ParseError
      raise
    end
    io = StringIO.new
    asm.write_to io
    io
  end

  def assert_bytes input, bytes
    io = parse input
    assert_equal bytes, io.string.bytes, ->() {
      pos          = 32.times.map { |i| (i % 0x10).to_s(16) }.join.reverse
      actual_bin   = sprintf("%032b", io.string.unpack1("L<"))
      expected_bin = sprintf("%032b", bytes.pack("C4").unpack1("L<"))
      broken = []
      actual_bin.bytes.zip(expected_bin.bytes).each_with_index { |(a, b), i|
        broken << (31 - i) unless a == b
      }
      expected_bin, actual_bin = color_diff(expected_bin, actual_bin)
      "idx: #{pos}\nexp: #{expected_bin}\nact: #{actual_bin}\n" +
        broken.reverse.map { |idx| "Bit #{idx} differs" }.join("\n") +
        "\n#{input}"
    }
  end

  def assert_round_trip input, output: [input]
    io = parse input
    assert_equal output, disasm(io.string).map { |x|
      "#{x.mnemonic} #{x.op_str}"
    }
  end

  def test_add
    assert_bytes "add sp, sp, #0x40", [0xff, 0x03, 0x01, 0x91]
    assert_bytes "add x8, sp, #1, lsl #12", [0xe8, 0x07, 0x40, 0x91]
    assert_bytes "add sp, sp, #1, lsl #12", [0xff, 0x07, 0x40, 0x91]
    assert_bytes "add x8, x10, #1", [0x48, 0x5, 00, 0x91]
  end

  def test_labels_already_defined
    error = assert_raises(RuntimeError) do
      parse "start:\nstart:\n"
    end
    assert_equal "symbol 'start' is already defined", error.message
  end

  def test_labels_empty
    assert_bytes "start:\n", []
    assert_bytes <<~ASM, []
      start:
      end:
    ASM
  end

  def test_labels_not_defined
    examples = [
      "b label",
      "b.eq label",
      "b.lo label",
      "b.lt label",
      "b.hs label",
      "b.gt label",
      "b.le label",
      "b.ne label",
      "b.mi label",
      "b.ge label",
      "b.pl label",
      "b.ls label",
      "b.hi label",
      "b.vc label",
      "b.vs label",
      "cbz x0, label",
      "cbz w0, label",
      "cbnz w0, label",
      "cbnz x0, label",
      "tbz w1, #1, label",
      "tbz x1, #1, label",
      "tbnz w1, #1, label",
      "tbnz x1, #1, label",
      "bl label",
    ]
    examples.each do |instruction|
      error = assert_raises(RuntimeError) do
        parse instruction
      end
      assert_match 'Label "label" not defined', error.message
    end
  end

  def test_labels_branch
    assert_bytes <<~ASM, [0x00, 0x04, 0x00, 0x91, 0xff, 0xff, 0xff, 0x17]
      loop:
        add x0, x0, #1
        b loop
    ASM
    assert_bytes <<~ASM, [0x20, 0x00, 0x80, 0xd2, 0xe0, 0xff, 0xff, 0x10]
      label:
        mov x0, #1
        adr x0, label
    ASM
  end

  def test_labels_forward_reference
    assert_bytes "b label\nlabel:", [0x01, 0x00, 0x00, 0x14]
    assert_bytes "b.eq label\nlabel:", [0x20, 0x00, 0x00, 0x54]
    assert_bytes "cbz w1, label\nlabel:", [0x21, 0x00, 0x00, 0x34]
    assert_bytes "cbz x1, label\nlabel:", [0x21, 0x00, 0x00, 0xb4]
    assert_bytes "cbnz w1, label\nlabel:", [0x21, 0x00, 0x00, 0x35]
    assert_bytes "cbnz x1, label\nlabel:", [0x21, 0x00, 0x00, 0xb5]
    assert_bytes "tbz w1, #1, label\nlabel:", [0x21, 0x00, 0x08, 0x36]
    assert_bytes "tbz x1, #1, label\nlabel:", [0x21, 0x00, 0x08, 0x36]
    assert_bytes "tbnz w1, #1, label\nlabel:", [0x21, 0x00, 0x08, 0x37]
    assert_bytes "tbnz x1, #1, label\nlabel:", [0x21, 0x00, 0x08, 0x37]
    assert_bytes "bl label\nlabel:", [0x01, 0x00, 0x00, 0x94]
    assert_bytes "adr x0, label\nlabel:", [0x20, 0x00, 0x00, 0x10]
    assert_bytes "adrp x0, label\nlabel:", [0x00, 0x00, 0x00, 0x90]
  end

  def test_bls
    assert_bytes "b.ls #0x10", [0x89, 0x00, 0000, 0x54]
  end

  def test_bhi
    assert_bytes "b.hi #0x28", [0x48, 0x01, 0000, 0x54]
  end

  def test_bvc
    assert_bytes "b.vc #0x2e8", [0x47, 0x17, 0000, 0x54]
    assert_bytes "b.vc #0x10", [0x87, 0000, 0000, 0x54]
  end

  def test_bvs
    assert_bytes "b.vs #0x64", [0x26, 0x03, 0000, 0x54]
    assert_bytes "b.vs #0x1c", [0xe6, 0000, 0000, 0x54]
    assert_bytes "b.vs #0xec", [0x66, 0x07, 0000, 0x54]
  end

  def test_brk
    assert_bytes "brk #0x1", [0x20, 0000, 0x20, 0xd4]
  end

  def test_cmn
    assert_bytes "cmn wsp, #0x555, lsl #12", [0xff, 0x57, 0x55, 0x31]
    assert_bytes "cmn x18, #0x555", [0x5f, 0x56, 0x15, 0xb1]
  end

  def test_cmp
    assert_bytes "cmp w0, #1", [0x1f, 0x04, 0000, 0x71]
  end

  def test_crc32
    assert_bytes "crc32b w5, w7, w20", [0xe5, 0x40, 0xd4, 0x1a]
    assert_bytes "crc32h w28, wzr, w30", [0xfc, 0x47, 0xde, 0x1a]
    assert_bytes "crc32w w0, w1, w2", [0x20, 0x48, 0xc2, 0x1a]
    assert_bytes "crc32x w7, w9, x20", [0x27, 0x4d, 0xd4, 0x9a]
  end

  def test_crc32c
    assert_bytes "crc32cb w9, w5, w4", [0xa9, 0x50, 0xc4, 0x1a]
    assert_bytes "crc32ch w13, w17, w25", [0x2d, 0x56, 0xd9, 0x1a]
    assert_bytes "crc32cw wzr, w3, w5", [0x7f, 0x58, 0xc5, 0x1a]
    assert_bytes "crc32cx w18, w16, xzr", [0x12, 0x5e, 0xdf, 0x9a]
  end

  def test_eon
    assert_bytes "eon w1, w2, w3", [0x41, 0000, 0x23, 0x4a]
    assert_bytes "eon x1, x2, x3", [0x41, 0000, 0x23, 0xca]
    assert_bytes "eon w1, w2, w3, lsl #4", [0x41, 0x10, 0x23, 0x4a]
    assert_bytes "eon x1, x2, x3, lsl #4", [0x41, 0x10, 0x23, 0xca]
    assert_bytes "eon w1, w2, w3, lsr #4", [0x41, 0x10, 0x63, 0x4a]
    assert_bytes "eon x1, x2, x3, lsr #4", [0x41, 0x10, 0x63, 0xca]
    assert_bytes "eon w1, w2, w3, asr #4", [0x41, 0x10, 0xa3, 0x4a]
    assert_bytes "eon x1, x2, x3, asr #4", [0x41, 0x10, 0xa3, 0xca]
    assert_bytes "eon w1, w2, w3, ror #4", [0x41, 0x10, 0xe3, 0x4a]
    assert_bytes "eon x1, x2, x3, ror #4", [0x41, 0x10, 0xe3, 0xca]
  end

  def test_eor
    assert_bytes "eor w1, w2, w3", [0x41, 0000, 0x03, 0x4a]
    assert_bytes "eor x1, x2, x3", [0x41, 0000, 0x03, 0xca]
    assert_bytes "eor w1, w2, w3, lsl #5", [0x41, 0x14, 0x03, 0x4a]
    assert_bytes "eor x1, x2, x3, lsl #5", [0x41, 0x14, 0x03, 0xca]
    assert_bytes "eor w1, w2, w3, lsr #5", [0x41, 0x14, 0x43, 0x4a]
    assert_bytes "eor x1, x2, x3, lsr #5", [0x41, 0x14, 0x43, 0xca]
    assert_bytes "eor w1, w2, w3, asr #5", [0x41, 0x14, 0x83, 0x4a]
    assert_bytes "eor x1, x2, x3, asr #5", [0x41, 0x14, 0x83, 0xca]
    assert_bytes "eor w1, w2, w3, ror #5", [0x41, 0x14, 0xc3, 0x4a]
    assert_bytes "eor x1, x2, x3, ror #5", [0x41, 0x14, 0xc3, 0xca]
  end

  def test_hlt
    assert_bytes "hlt #0x7b", [0x60, 0x0f, 0x40, 0xd4]
  end

  def test_hvc
    assert_bytes "hvc #0x1", [0x22, 0000, 0000, 0xd4]
  end

  def test_ldursh
    assert_bytes "ldursh x17, [x19, #-0x100]", [0x71, 0x02, 0x90, 0x78]
    assert_bytes "ldursh w15, [x21, #-0x100]", [0xaf, 0x02, 0xd0, 0x78]
  end

  def test_movz
    assert_bytes "movz x0, #0x2a", [0x40, 0x05, 0x80, 0xd2]
  end

  def test_movk
    assert_bytes "movk x0, #0x2a", [0x40, 0x05, 0x80, 0xf2]
    assert_bytes "movk x0, #0x2a, lsl #16", [0x40, 0x05, 0xa0, 0xf2]
  end

  def test_movn
    assert_bytes "movn w0, #0x2", [0x40, 0000, 0x80, 0x12]
    assert_bytes "movn x0, #0x2", [0x40, 0000, 0x80, 0x92]
    assert_bytes "movn w0, #0x2, lsl #16", [0x40, 0000, 0xa0, 0x12]
    assert_bytes "movn x0, #0x2, lsl #16", [0x40, 0000, 0xa0, 0x92]
    assert_bytes "movn w2, #0x4d2", [0x42, 0x9a, 0x80, 0x12]
  end

  def test_msr
    assert_bytes "msr dbgdtrtx_el0, x3", [0x03, 0x05, 0x13, 0xd5]
    assert_bytes "msr lorsa_el1, x0", [0000, 0xa4, 0x18, 0xd5]
    assert_bytes "msr lorea_el1, x0", [0x20, 0xa4, 0x18, 0xd5]
    assert_bytes "msr lorn_el1, x0", [0x40, 0xa4, 0x18, 0xd5]
    assert_bytes "msr lorc_el1, x0", [0x60, 0xa4, 0x18, 0xd5]
    assert_bytes "msr pan, x5", [0x65, 0x42, 0x18, 0xd5]
    assert_bytes "msr ttbr1_el2, x0", [0x20, 0x20, 0x1c, 0xd5]
    assert_bytes "msr s3_0_c5_c3_1, x0", [0x20, 0x53, 0x18, 0xd5]
  end

  def test_mrs
    assert_bytes "mrs x3, afsr0_el1", [0x03, 0x51, 0x38, 0xd5]
    assert_bytes "mrs x3, afsr0_el2", [0x03, 0x51, 0x3c, 0xd5]
    assert_bytes "mrs x3, afsr0_el3", [0x03, 0x51, 0x3e, 0xd5]
    assert_bytes "mrs x3, aidr_el1", [0xe3, 0000, 0x39, 0xd5]
    assert_bytes "mrs x3, afsr1_el1", [0x23, 0x51, 0x38, 0xd5]
    assert_bytes "mrs x3, afsr1_el2", [0x23, 0x51, 0x3c, 0xd5]
    assert_bytes "mrs x3, afsr1_el3", [0x23, 0x51, 0x3e, 0xd5]
    assert_bytes "mrs x3, amair_el1", [0x03, 0xa3, 0x38, 0xd5]
    assert_bytes "mrs x3, amair_el2", [0x03, 0xa3, 0x3c, 0xd5]
    assert_bytes "mrs x3, amair_el3", [0x03, 0xa3, 0x3e, 0xd5]
    assert_bytes "mrs x3, ccsidr_el1", [0x03, 0000, 0x39, 0xd5]
    assert_bytes "mrs x3, clidr_el1", [0x23, 0000, 0x39, 0xd5]
    assert_bytes "mrs x6, s3_0_c12_c1_1", [0x26, 0xc1, 0x38, 0xd5]
    assert_bytes "mrs x7, s3_4_c12_c1_1", [0x27, 0xc1, 0x3c, 0xd5]
  end

  def test_mvn
    assert_bytes "mvn w4, w9", [0xe4, 0x03, 0x29, 0x2a]
    assert_bytes "mvn x2, x3", [0xe2, 0x03, 0x23, 0xaa]
    assert_bytes "mvn w4, w9", [0xe4, 0x03, 0x29, 0x2a]
    assert_bytes "mvn w4, w9, lsl #1", [0xe4, 0x07, 0x29, 0x2a]
    assert_bytes "mvn x2, x3, lsl #1", [0xe2, 0x07, 0x23, 0xaa]
    assert_bytes "mvn w4, w9, lsl #1", [0xe4, 0x07, 0x29, 0x2a]
  end

  def test_orr
    assert_bytes "orr w1, w2, w3", [0x41, 0000, 0x03, 0x2a]
    assert_bytes "orr x1, x2, x3", [0x41, 0000, 0x03, 0xaa]
  end

  def test_prfm
    assert_bytes "prfm plil2keep, #4", [0x2a, 0000, 0000, 0xd8]
    assert_bytes "prfm #0x1f, [sp, #0x20]", [0xff, 0x13, 0x80, 0xf9]
    assert_bytes "prfm pldl1keep, [x2]", [0x40, 0000, 0x80, 0xf9]
    assert_bytes "prfm pldl1strm, [x2]", [0x41, 0000, 0x80, 0xf9]
    assert_bytes "prfm pldl2keep, [x2]", [0x42, 0000, 0x80, 0xf9]
    assert_bytes "prfm pldl2strm, [x2]", [0x43, 0000, 0x80, 0xf9]
    assert_bytes "prfm pldl3keep, [x2]", [0x44, 0000, 0x80, 0xf9]
    assert_bytes "prfm pldl3strm, [x2]", [0x45, 0000, 0x80, 0xf9]
    assert_bytes "prfm pstl1keep, [x2]", [0x50, 0000, 0x80, 0xf9]
    assert_bytes "prfm pstl1strm, [x2]", [0x51, 0000, 0x80, 0xf9]
    assert_bytes "prfm pstl2keep, [x2]", [0x52, 0000, 0x80, 0xf9]
    assert_bytes "prfm pstl2strm, [x2]", [0x53, 0000, 0x80, 0xf9]
    assert_bytes "prfm pstl3keep, [x2]", [0x54, 0000, 0x80, 0xf9]
    assert_bytes "prfm pstl3strm, [x2]", [0x55, 0000, 0x80, 0xf9]
  end

  def test_smc
    assert_bytes "smc #0xf", [0xe3, 0x01, 0000, 0xd4]
  end

  def test_stnp
    assert_bytes "stnp x4, x9, [sp, #-0x10]", [0xe4, 0x27, 0x3f, 0xa8]
  end

  def test_stp
    assert_bytes "stp x0, x1, [sp]", [0xe0, 0x07, 0000, 0xa9]
  end

  def test_strh
    assert_bytes "strh wzr, [x8, x9]", [0x1f, 0x69, 0x29, 0x78]
  end

  def test_hint
    assert_bytes "hint #0xc", [0x9f, 0x21, 0x03, 0xd5]
  end

  def test_tbz
    assert_bytes "tbz w3, #5, #4", [0x23, 0000, 0x28, 0x36]
  end

  def test_tbnz
    assert_bytes "tbnz w3, #5, #4", [0x23, 0000, 0x28, 0x37]
  end

  def test_tst
    assert_bytes "tst w1, w2", [0x3f, 0000, 0x02, 0x6a]
    assert_bytes "tst x1, x2", [0x3f, 0000, 0x02, 0xea]
  end

  def test_generated_dc
    assert_bytes "dc      cisw, x30", [0x5e,0x7e,0x08,0xd5]
    assert_bytes "dc      civac, x3", [0x23,0x7e,0x0b,0xd5]
    assert_bytes "dc      csw, x10", [0x4a,0x7a,0x08,0xd5]
    assert_bytes "dc      cvac, x9", [0x29,0x7a,0x0b,0xd5]
    assert_bytes "dc      cvau, x0", [0x20,0x7b,0x0b,0xd5]
    assert_bytes "dc      isw, x2", [0x42,0x76,0x08,0xd5]
    assert_bytes "dc      ivac, xzr", [0x3f,0x76,0x08,0xd5]
    assert_bytes "dc      zva, x12", [0x2c,0x74,0x0b,0xd5]
  end

  def test_generated_ic
    assert_bytes "ic      iallu", [0x1f,0x75,0x08,0xd5]
    assert_bytes "ic      ialluis", [0x1f,0x71,0x08,0xd5]
    assert_bytes "ic      ivau, x9", [0x29,0x75,0x0b,0xd5]
  end

  def test_generated_adc
    assert_bytes "adc      w20, w0, wzr", [0x14,0x00,0x1f,0x1a]
    assert_bytes "adc      w29, w27, w25", [0x7d,0x03,0x19,0x1a]
    assert_bytes "adc      w9, wzr, w10", [0xe9,0x03,0x0a,0x1a]
    assert_bytes "adc      wzr, w3, w4", [0x7f,0x00,0x04,0x1a]
    assert_bytes "adc      x20, x0, xzr", [0x14,0x00,0x1f,0x9a]
    assert_bytes "adc      x29, x27, x25", [0x7d,0x03,0x19,0x9a]
    assert_bytes "adc      x9, xzr, x10", [0xe9,0x03,0x0a,0x9a]
    assert_bytes "adc      xzr, x3, x4", [0x7f,0x00,0x04,0x9a]
  end

  def test_generated_adcs
    assert_bytes "adcs     w20, w0, wzr", [0x14,0x00,0x1f,0x3a]
    assert_bytes "adcs     w29, w27, w25", [0x7d,0x03,0x19,0x3a]
    assert_bytes "adcs     w9, wzr, w10", [0xe9,0x03,0x0a,0x3a]
    assert_bytes "adcs     wzr, w3, w4", [0x7f,0x00,0x04,0x3a]
    assert_bytes "adcs     x20, x0, xzr", [0x14,0x00,0x1f,0xba]
    assert_bytes "adcs     x29, x27, x25", [0x7d,0x03,0x19,0xba]
    assert_bytes "adcs     x9, xzr, x10", [0xe9,0x03,0x0a,0xba]
    assert_bytes "adcs     xzr, x3, x4", [0x7f,0x00,0x04,0xba]
  end

  def test_generated_add
    assert_bytes "add     x17, x29, x20, asr #58", [0xb1,0xeb,0x94,0x8b]
    assert_bytes "add     x17, x29, x20, lsl #58", [0xb1,0xeb,0x14,0x8b]
    assert_bytes "add     x17, x29, x20, lsr #58", [0xb1,0xeb,0x54,0x8b]
    assert_bytes "add      sp, x2, w3, sxtw", [0x5f,0xc0,0x23,0x8b]
    assert_bytes "add      sp, x29, #3816", [0xbf,0xa3,0x3b,0x91]
    assert_bytes "add      w0, w2, w3, sxtw", [0x40,0xc0,0x23,0x0b]
    assert_bytes "add      w11, w13, w15", [0xab,0x01,0x0f,0x0b]
    assert_bytes "add      w13, w5, #4095, lsl #12", [0xad,0xfc,0x7f,0x11]
    assert_bytes "add      w17, w19, w23, uxtw #2", [0x71,0x4a,0x37,0x0b]
    assert_bytes "add      w17, w29, w20, lsl #29", [0xb1,0x77,0x14,0x0b]
    assert_bytes "add      w17, w29, w20, lsl #31", [0xb1,0x7f,0x14,0x0b]
    assert_bytes "add      w19, w17, w1, uxtx", [0x33,0x62,0x21,0x0b]
    assert_bytes "add      w2, w3, #4095", [0x62,0xfc,0x3f,0x11]
    assert_bytes "add      w2, w3, w4, asr #0", [0x62,0x00,0x84,0x0b]
    assert_bytes "add      w2, w3, w5, sxtx", [0x62,0xe0,0x25,0x0b]
    assert_bytes "add      w2, w5, w1, sxtb", [0xa2,0x80,0x21,0x0b]
    assert_bytes "add      w2, w5, w7, uxtb", [0xa2,0x00,0x27,0x0b]
    assert_bytes "add      w2, wsp, w3, lsl #1", [0xe2,0x47,0x23,0x0b]
    assert_bytes "add      w20, wsp, #801", [0xf4,0x87,0x0c,0x11]
    assert_bytes "add      w20, wzr, w4", [0xf4,0x03,0x04,0x0b]
    assert_bytes "add      w21, w15, w17, uxth", [0xf5,0x21,0x31,0x0b]
    assert_bytes "add      w21, w22, w23, lsr #0", [0xd5,0x02,0x57,0x0b]
    assert_bytes "add      w24, w25, w26, lsr #18", [0x38,0x4b,0x5a,0x0b]
    assert_bytes "add      w26, w17, w19, sxth", [0x3a,0xa2,0x33,0x0b]
    assert_bytes "add      w27, w28, w29, lsr #29", [0x9b,0x77,0x5d,0x0b]
    assert_bytes "add      w27, w28, w29, lsr #31", [0x9b,0x7f,0x5d,0x0b]
    assert_bytes "add      w29, w23, w17, uxtx #1", [0xfd,0x66,0x31,0x0b]
    assert_bytes "add      w3, w5, w7", [0xa3,0x00,0x07,0x0b]
    assert_bytes "add      w30, w29, #1, lsl #12", [0xbe,0x07,0x40,0x11]
    assert_bytes "add      w30, w29, wzr, uxtw", [0xbe,0x43,0x3f,0x0b]
    assert_bytes "add      w4, w5, #0", [0xa4,0x00,0x00,0x11]
    assert_bytes "add      w4, w6, wzr", [0xc4,0x00,0x1f,0x0b]
    assert_bytes "add      w5, w6, w7, asr #21", [0xc5,0x54,0x87,0x0b]
    assert_bytes "add      w8, w9, w10, asr #29", [0x28,0x75,0x8a,0x0b]
    assert_bytes "add      w8, w9, w10, asr #31", [0x28,0x7d,0x8a,0x0b]
    assert_bytes "add      w9, w3, wzr, lsl #10", [0x69,0x28,0x1f,0x0b]
    assert_bytes "add      wsp, w30, #4084", [0xdf,0xd3,0x3f,0x11]
    assert_bytes "add      wsp, wsp, #1104", [0xff,0x43,0x11,0x11]
    assert_bytes "add      wzr, w3, w5", [0x7f,0x00,0x05,0x0b]
    assert_bytes "add      x0, x24, #291", [0x00,0x8f,0x04,0x91]
    assert_bytes "add      x11, x13, x15", [0xab,0x01,0x0f,0x8b]
    assert_bytes "add      x12, x1, w20, uxtw", [0x2c,0x40,0x34,0x8b]
    assert_bytes "add      x17, x25, w20, sxtb", [0x31,0x83,0x34,0x8b]
    assert_bytes "add      x17, x29, x20, lsl #63", [0xb1,0xff,0x14,0x8b]
    assert_bytes "add      x18, x13, w19, sxth", [0xb2,0xa1,0x33,0x8b]
    assert_bytes "add      x2, x3, w5, sxtb", [0x62,0x80,0x25,0x8b]
    assert_bytes "add      x2, x3, x4, asr #0", [0x62,0x00,0x84,0x8b]
    assert_bytes "add      x2, x4, w5, uxtb", [0x82,0x00,0x25,0x8b]
    assert_bytes "add      x20, sp, w19, uxth", [0xf4,0x23,0x33,0x8b]
    assert_bytes "add      x20, x3, x13, uxtx", [0x74,0x60,0x2d,0x8b]
    assert_bytes "add      x20, xzr, x4", [0xf4,0x03,0x04,0x8b]
    assert_bytes "add      x21, x22, x23, lsr #0", [0xd5,0x02,0x57,0x8b]
    assert_bytes "add      x24, x25, x26, lsr #18", [0x38,0x4b,0x5a,0x8b]
    assert_bytes "add      x27, x28, x29, lsr #63", [0x9b,0xff,0x5d,0x8b]
    assert_bytes "add      x3, x24, #4095, lsl #12", [0x03,0xff,0x7f,0x91]
    assert_bytes "add      x3, x5, x7", [0xa3,0x00,0x07,0x8b]
    assert_bytes "add      x3, x5, x9, sxtx", [0xa3,0xe0,0x29,0x8b]
    assert_bytes "add      x4, x6, xzr", [0xc4,0x00,0x1f,0x8b]
    assert_bytes "add      x5, x6, x7, asr #21", [0xc5,0x54,0x87,0x8b]
    assert_bytes "add      x5, x7, #1638", [0xe5,0x98,0x19,0x91]
    assert_bytes "add      x7, x11, w13, uxth #4", [0x67,0x31,0x2d,0x8b]
    assert_bytes "add      x8, sp, #1074", [0xe8,0xcb,0x10,0x91]
    assert_bytes "add      x8, x9, x10, asr #63", [0x28,0xfd,0x8a,0x8b]
    assert_bytes "add      x9, x3, xzr, lsl #10", [0x69,0x28,0x1f,0x8b]
    assert_bytes "add      xzr, x3, x5", [0x7f,0x00,0x05,0x8b]
  end

  def test_generated_adds
    assert_bytes "adds     w11, w13, w15", [0xab,0x01,0x0f,0x2b]
    assert_bytes "adds     w13, w23, #291, lsl #12", [0xed,0x8e,0x44,0x31]
    assert_bytes "adds     w17, w29, w20, lsl #31", [0xb1,0x7f,0x14,0x2b]
    assert_bytes "adds     w19, w17, w1, uxtx", [0x33,0x62,0x21,0x2b]
    assert_bytes "adds     w2, w3, w4, asr #0", [0x62,0x00,0x84,0x2b]
    assert_bytes "adds     w2, w3, w5, sxtx", [0x62,0xe0,0x25,0x2b]
    assert_bytes "adds     w2, w5, w1, sxtb #1", [0xa2,0x84,0x21,0x2b]
    assert_bytes "adds     w2, w5, w7, uxtb", [0xa2,0x00,0x27,0x2b]
    assert_bytes "adds     w20, wsp, #0", [0xf4,0x03,0x00,0x31]
    assert_bytes "adds     w20, wzr, w4", [0xf4,0x03,0x04,0x2b]
    assert_bytes "adds     w21, w15, w17, uxth", [0xf5,0x21,0x31,0x2b]
    assert_bytes "adds     w21, w22, w23, lsr #0", [0xd5,0x02,0x57,0x2b]
    assert_bytes "adds     w24, w25, w26, lsr #18", [0x38,0x4b,0x5a,0x2b]
    assert_bytes "adds     w26, wsp, w19, sxth", [0xfa,0xa3,0x33,0x2b]
    assert_bytes "adds     w27, w28, w29, lsr #31", [0x9b,0x7f,0x5d,0x2b]
    assert_bytes "adds     w3, w5, w7", [0xa3,0x00,0x07,0x2b]
    assert_bytes "adds     w30, w29, wzr, uxtw", [0xbe,0x43,0x3f,0x2b]
    assert_bytes "adds     w4, w6, wzr", [0xc4,0x00,0x1f,0x2b]
    assert_bytes "adds     w5, w6, w7, asr #21", [0xc5,0x54,0x87,0x2b]
    assert_bytes "adds     w8, w9, w10, asr #31", [0x28,0x7d,0x8a,0x2b]
    assert_bytes "adds     w9, w3, wzr, lsl #10", [0x69,0x28,0x1f,0x2b]
    assert_bytes "adds     x11, x13, x15", [0xab,0x01,0x0f,0xab]
    assert_bytes "adds     x12, x1, w20, uxtw", [0x2c,0x40,0x34,0xab]
    assert_bytes "adds     x17, x29, x20, lsl #63", [0xb1,0xff,0x14,0xab]
    assert_bytes "adds     x18, sp, w19, sxth", [0xf2,0xa3,0x33,0xab]
    assert_bytes "adds     x2, x3, x4, asr #0", [0x62,0x00,0x84,0xab]
    assert_bytes "adds     x2, x4, w5, uxtb #2", [0x82,0x08,0x25,0xab]
    assert_bytes "adds     x20, sp, w19, uxth #4", [0xf4,0x33,0x33,0xab]
    assert_bytes "adds     x20, x3, x13, uxtx", [0x74,0x60,0x2d,0xab]
    assert_bytes "adds     x20, xzr, x4", [0xf4,0x03,0x04,0xab]
    assert_bytes "adds     x21, x22, x23, lsr #0", [0xd5,0x02,0x57,0xab]
    assert_bytes "adds     x24, x25, x26, lsr #18", [0x38,0x4b,0x5a,0xab]
    assert_bytes "adds     x27, x28, x29, lsr #63", [0x9b,0xff,0x5d,0xab]
    assert_bytes "adds     x3, x5, x7", [0xa3,0x00,0x07,0xab]
    assert_bytes "adds     x3, x5, x9, sxtx #2", [0xa3,0xe8,0x29,0xab]
    assert_bytes "adds     x4, x6, xzr", [0xc4,0x00,0x1f,0xab]
    assert_bytes "adds     x5, x6, x7, asr #21", [0xc5,0x54,0x87,0xab]
    assert_bytes "adds     x8, x9, x10, asr #63", [0x28,0xfd,0x8a,0xab]
    assert_bytes "adds     x9, x3, xzr, lsl #10", [0x69,0x28,0x1f,0xab]
  end

  def test_generated_adr
    assert_bytes "adr     x2, #-1048576", [0x02,0x00,0x80,0x10]
    assert_bytes "adr     x20, #0", [0x14,0x00,0x00,0x10]
    assert_bytes "adr     x5, #1048575", [0xe5,0xff,0x7f,0x70]
    assert_bytes "adr     x9, #-1", [0xe9,0xff,0xff,0x70]
    assert_bytes "adr     x9, #1048575", [0xe9,0xff,0x7f,0x70]
  end

  def test_generated_adrp
    assert_bytes "adrp    x20, #-4294967296", [0x14,0x00,0x80,0x90]
    assert_bytes "adrp    x30, #4096", [0x1e,0x00,0x00,0xb0]
    assert_bytes "adrp    x9, #4294963200", [0xe9,0xff,0x7f,0xf0]
  end

  def test_generated_and
    assert_bytes "and  w2, w3, #0xfffffffd", [0x62,0x78,0x1e,0x12]
    assert_bytes "and      w0, w2, wzr, lsr #17", [0x40,0x44,0x5f,0x0a]
    assert_bytes "and      w11, wzr, #0x30003", [0xeb,0x87,0x00,0x12]
    assert_bytes "and      w12, w13, #0xffc3ffc3", [0xac,0xad,0x0a,0x12]
    assert_bytes "and      w12, w23, w21", [0xec,0x02,0x15,0x0a]
    assert_bytes "and      w14, w15, #0x80008000", [0xee,0x81,0x01,0x12]
    assert_bytes "and      w16, w15, w1, lsl #1", [0xf0,0x05,0x01,0x0a]
    assert_bytes "and      w3, w17, w19, ror #31", [0x23,0x7e,0xd3,0x0a]
    assert_bytes "and      w3, w30, w11", [0xc3,0x03,0x0b,0x0a]
    assert_bytes "and      w3, w30, w11, asr #0", [0xc3,0x03,0x8b,0x0a]
    assert_bytes "and      w3, wzr, w20, ror #0", [0xe3,0x03,0xd4,0x0a]
    assert_bytes "and      w9, w4, w10, lsl #31", [0x89,0x7c,0x0a,0x0a]
    assert_bytes "and      x11, xzr, #0x3000300030003", [0xeb,0x87,0x00,0x92]
    assert_bytes "and      x12, x13, #0xffc3ffc3ffc3ffc3", [0xac,0xad,0x0a,0x92]
    assert_bytes "and      x14, x15, #0x8000800080008000", [0xee,0x81,0x01,0x92]
    assert_bytes "and      x3, x5, x7, lsl #63", [0xa3,0xfc,0x07,0x8a]
    assert_bytes "and      x5, x14, x19, asr #4", [0xc5,0x11,0x93,0x8a]
    assert_bytes "and      x7, x20, xzr, asr #63", [0x87,0xfe,0x9f,0x8a]
    assert_bytes "and      x9, x10, #0x7fffffffffff", [0x49,0xb9,0x40,0x92]
    assert_bytes "and      xzr, x4, x26", [0x9f,0x00,0x1a,0x8a]
  end

  def test_generated_ands
    assert_bytes "ands  w19, w20, #0xfffffff0", [0x93,0x6e,0x1c,0x72]
    assert_bytes "ands     w19, w20, #0x33333333", [0x93,0xe6,0x00,0x72]
    assert_bytes "ands     w21, w22, #0x99999999", [0xd5,0xe6,0x01,0x72]
    assert_bytes "ands     w7, wzr, w9, lsl #1", [0xe7,0x07,0x09,0x6a]
    assert_bytes "ands     x19, x20, #0x3333333333333333", [0x93,0xe6,0x00,0xf2]
    assert_bytes "ands     x21, x22, #0x9999999999999999", [0xd5,0xe6,0x01,0xf2]
    assert_bytes "ands     x3, x5, x20, ror #63", [0xa3,0xfc,0xd4,0xea]
  end

  def test_generated_asr
    assert_bytes "asr  w18, w19, w20", [0x72,0x2a,0xd4,0x1a]
    assert_bytes "asr  w23, w24, w25", [0x17,0x2b,0xd9,0x1a]
    assert_bytes "asr  x21, x22, x23", [0xd5,0x2a,0xd7,0x9a]
    assert_bytes "asr  x26, x27, x28", [0x7a,0x2b,0xdc,0x9a]
    assert_bytes "asr      w1, wzr, #3", [0xe1,0x7f,0x03,0x13]
    assert_bytes "asr      w3, w2, #0", [0x43,0x7c,0x00,0x13]
    assert_bytes "asr      w9, w10, #31", [0x49,0x7d,0x1f,0x13]
    assert_bytes "asr      wzr, wzr, #31", [0xff,0x7f,0x1f,0x13]
    assert_bytes "asr      x20, x21, #63", [0xb4,0xfe,0x7f,0x93]
    assert_bytes "asr      x3, x4, #63", [0x83,0xfc,0x7f,0x93]
    assert_bytes "asr     w11, w12, #31", [0x8b,0x7d,0x1f,0x13]
    assert_bytes "asr     w13, w14, #29", [0xcd,0x7d,0x1d,0x13]
    assert_bytes "asr     w9, w10, #0", [0x49,0x7d,0x00,0x13]
    assert_bytes "asr     x19, x20, #0", [0x93,0xfe,0x40,0x93]
    assert_bytes "asr     x2, x3, #63", [0x62,0xfc,0x7f,0x93]
    assert_bytes "asr     x9, x10, #5", [0x49,0xfd,0x45,0x93]
    assert_bytes "asr    w9, w10, #0", [0x49,0x7d,0x00,0x13]
    assert_bytes "asr    x19, x20, #0", [0x93,0xfe,0x40,0x93]
  end

  def test_generated_at
    assert_bytes "at      s12e0r, x20", [0xd4,0x78,0x0c,0xd5]
    assert_bytes "at      s12e0w, x20", [0xf4,0x78,0x0c,0xd5]
    assert_bytes "at      s12e1r, x20", [0x94,0x78,0x0c,0xd5]
    assert_bytes "at      s12e1w, x20", [0xb4,0x78,0x0c,0xd5]
    assert_bytes "at      s1e0r, x19", [0x53,0x78,0x08,0xd5]
    assert_bytes "at      s1e0w, x19", [0x73,0x78,0x08,0xd5]
    assert_bytes "at      s1e1r, x19", [0x13,0x78,0x08,0xd5]
    assert_bytes "at      s1e1w, x19", [0x33,0x78,0x08,0xd5]
    assert_bytes "at      s1e2r, x19", [0x13,0x78,0x0c,0xd5]
    assert_bytes "at      s1e2w, x19", [0x33,0x78,0x0c,0xd5]
    assert_bytes "at      s1e3r, x19", [0x13,0x78,0x0e,0xd5]
    assert_bytes "at      s1e3w, x19", [0x33,0x78,0x0e,0xd5]
  end

  def test_generated_b
    assert_bytes "b       #134217724", [0xff,0xff,0xff,0x15]
    assert_bytes "b       #4", [0x01,0x00,0x00,0x14]
    assert_bytes "b.eq #0", [0x00,0x00,0x00,0x54]
    assert_bytes "b.lo #1048572", [0xe3,0xff,0x7f,0x54]
    assert_bytes "b.lt #-4", [0xeb,0xff,0xff,0x54]
  end

  def test_generated_bfi
    assert_bytes "bfi      w11, w12, #31, #1", [0x8b,0x01,0x01,0x33]
    assert_bytes "bfi      w13, w14, #29, #3", [0xcd,0x09,0x03,0x33]
    assert_bytes "bfi      x2, x3, #63, #1", [0x62,0x00,0x41,0xb3]
    assert_bytes "bfi      x4, x5, #52, #11", [0xa4,0x28,0x4c,0xb3]
    assert_bytes "bfi      x4, xzr, #1, #6", [0xe4,0x17,0x7f,0xb3]
    assert_bytes "bfi      x9, x10, #5, #59", [0x49,0xe9,0x7b,0xb3]
    assert_bytes "bfi      xzr, xzr, #10, #11", [0xff,0x2b,0x76,0xb3]
    assert_bytes "bfi wzr, wzr, #31, #1", [0xff,0x03,0x01,0x33]
    assert_bytes "bfi x0, xzr, #5, #9", [0xe0,0x23,0x7b,0xb3]
    assert_bytes "bfi xzr, xzr, #63, #1", [0xff,0x03,0x41,0xb3]
  end

  def test_generated_bfxil
    assert_bytes "bfxil    w11, w12, #31, #1", [0x8b,0x7d,0x1f,0x33]
    assert_bytes "bfxil    w13, w14, #29, #3", [0xcd,0x7d,0x1d,0x33]
    assert_bytes "bfxil    w9, w10, #0, #1", [0x49,0x01,0x00,0x33]
    assert_bytes "bfxil    w9, w10, #0, #32", [0x49,0x7d,0x00,0x33]
    assert_bytes "bfxil    x19, x20, #0, #64", [0x93,0xfe,0x40,0xb3]
    assert_bytes "bfxil    x2, x3, #63, #1", [0x62,0xfc,0x7f,0xb3]
    assert_bytes "bfxil    x5, x6, #12, #52", [0xc5,0xfc,0x4c,0xb3]
    assert_bytes "bfxil    x9, x10, #5, #59", [0x49,0xfd,0x45,0xb3]
    assert_bytes "bfxil    xzr, x4, #0, #1", [0x9f,0x00,0x40,0xb3]
    assert_bytes "bfxil    xzr, xzr, #10, #11", [0xff,0x53,0x4a,0xb3]
    assert_bytes "bfxil w3, wzr, #0, #32", [0xe3,0x7f,0x00,0x33]
  end

  def test_generated_bic
    assert_bytes "bic      w2, w7, w9", [0xe2,0x00,0x29,0x0a]
    assert_bytes "bic      x13, x20, x14, lsl #47", [0x8d,0xbe,0x2e,0x8a]
  end

  def test_generated_bics
    assert_bytes "bics     w3, w5, w7", [0xa3,0x00,0x27,0x6a]
    assert_bytes "bics     x3, xzr, x3, lsl #1", [0xe3,0x07,0x23,0xea]
  end

  def test_generated_bl
    assert_bytes "bl      #-134217728", [0x00,0x00,0x00,0x96]
    assert_bytes "bl      #0", [0x00,0x00,0x00,0x94]
  end

  def test_generated_blr
    assert_bytes "blr      xzr", [0xe0,0x03,0x3f,0xd6]
  end

  def test_generated_br
    assert_bytes "br       x20", [0x80,0x02,0x1f,0xd6]
  end

  def test_generated_cbnz
    assert_bytes "cbnz    x3, #-4", [0xe3,0xff,0xff,0xb5]
    assert_bytes "cbnz    xzr, #-1048576", [0x1f,0x00,0x80,0xb5]
  end

  def test_generated_cbz
    assert_bytes "cbz     w20, #1048572", [0xf4,0xff,0x7f,0x34]
    assert_bytes "cbz     w5, #0", [0x05,0x00,0x00,0x34]
  end

  def test_generated_ccmn
    assert_bytes "ccmn    w1, #31, #0, eq", [0x20,0x08,0x5f,0x3a]
    assert_bytes "ccmn    w1, wzr, #0, eq", [0x20,0x00,0x5f,0x3a]
    assert_bytes "ccmn    w3, #0, #15, hs", [0x6f,0x28,0x40,0x3a]
    assert_bytes "ccmn    w3, w0, #15, hs", [0x6f,0x20,0x40,0x3a]
    assert_bytes "ccmn    wzr, #15, #13, hs", [0xed,0x2b,0x4f,0x3a]
    assert_bytes "ccmn    wzr, w15, #13, hs", [0xed,0x23,0x4f,0x3a]
    assert_bytes "ccmn    x3, #0, #15, gt", [0x6f,0xc8,0x40,0xba]
    assert_bytes "ccmn    x3, x0, #15, gt", [0x6f,0xc0,0x40,0xba]
    assert_bytes "ccmn    x9, #31, #0, le", [0x20,0xd9,0x5f,0xba]
    assert_bytes "ccmn    x9, xzr, #0, le", [0x20,0xd1,0x5f,0xba]
    assert_bytes "ccmn    xzr, #5, #7, ne", [0xe7,0x1b,0x45,0xba]
    assert_bytes "ccmn    xzr, x5, #7, ne", [0xe7,0x13,0x45,0xba]
  end

  def test_generated_ccmp
    assert_bytes "ccmp    w1, #31, #0, eq", [0x20,0x08,0x5f,0x7a]
    assert_bytes "ccmp    w1, wzr, #0, eq", [0x20,0x00,0x5f,0x7a]
    assert_bytes "ccmp    w3, #0, #15, hs", [0x6f,0x28,0x40,0x7a]
    assert_bytes "ccmp    w3, w0, #15, hs", [0x6f,0x20,0x40,0x7a]
    assert_bytes "ccmp    wzr, #15, #13, hs", [0xed,0x2b,0x4f,0x7a]
    assert_bytes "ccmp    wzr, w15, #13, hs", [0xed,0x23,0x4f,0x7a]
    assert_bytes "ccmp    x3, #0, #15, gt", [0x6f,0xc8,0x40,0xfa]
    assert_bytes "ccmp    x3, x0, #15, gt", [0x6f,0xc0,0x40,0xfa]
    assert_bytes "ccmp    x9, #31, #0, le", [0x20,0xd9,0x5f,0xfa]
    assert_bytes "ccmp    x9, xzr, #0, le", [0x20,0xd1,0x5f,0xfa]
    assert_bytes "ccmp    xzr, #5, #7, ne", [0xe7,0x1b,0x45,0xfa]
    assert_bytes "ccmp    xzr, x5, #7, ne", [0xe7,0x13,0x45,0xfa]
  end

  def test_generated_cinc
    assert_bytes "cinc     x3, x5, gt", [0xa3,0xd4,0x85,0x9a]
    assert_bytes "cinc     xzr, x4, le", [0x9f,0xc4,0x84,0x9a]
    assert_bytes "cinc    w3, w5, gt", [0xa3,0xd4,0x85,0x1a]
    assert_bytes "cinc    wzr, w4, le", [0x9f,0xc4,0x84,0x1a]
  end

  def test_generated_cinv
    assert_bytes "cinv    w3, w5, gt", [0xa3,0xd0,0x85,0x5a]
    assert_bytes "cinv    wzr, w4, le", [0x9f,0xc0,0x84,0x5a]
    assert_bytes "cinv    x3, x5, gt", [0xa3,0xd0,0x85,0xda]
    assert_bytes "cinv    xzr, x4, le", [0x9f,0xc0,0x84,0xda]
  end

  def test_generated_clrex
    assert_bytes "clrex", [0x5f,0x3f,0x03,0xd5]
    assert_bytes "clrex   #0", [0x5f,0x30,0x03,0xd5]
    assert_bytes "clrex   #7", [0x5f,0x37,0x03,0xd5]
  end

  def test_generated_cls
    assert_bytes "cls  w3, w5", [0xa3,0x14,0xc0,0x5a]
    assert_bytes "cls  x20, x5", [0xb4,0x14,0xc0,0xda]
  end

  def test_generated_clz
    assert_bytes "clz  w24, w3", [0x78,0x10,0xc0,0x5a]
    assert_bytes "clz  w24, wzr", [0xf8,0x13,0xc0,0x5a]
    assert_bytes "clz  x26, x4", [0x9a,0x10,0xc0,0xda]
  end

  def test_generated_cmn
    assert_bytes "cmn      sp, #1092, lsl #12", [0xff,0x13,0x51,0xb1]
    assert_bytes "cmn      sp, w19, sxth", [0xff,0xa3,0x33,0xab]
    assert_bytes "cmn      sp, w19, uxth #4", [0xff,0x33,0x33,0xab]
    assert_bytes "cmn      sp, x6", [0xff,0x63,0x26,0xab]
    assert_bytes "cmn      w0, w3", [0x1f,0x00,0x03,0x2b]
    assert_bytes "cmn      w10, w11, lsl #31", [0x5f,0x7d,0x0b,0x2b]
    assert_bytes "cmn      w12, w13, lsr #0", [0x9f,0x01,0x4d,0x2b]
    assert_bytes "cmn      w14, w15, lsr #21", [0xdf,0x55,0x4f,0x2b]
    assert_bytes "cmn      w16, w17, lsr #31", [0x1f,0x7e,0x51,0x2b]
    assert_bytes "cmn      w18, w19, asr #0", [0x5f,0x02,0x93,0x2b]
    assert_bytes "cmn      w20, w21, asr #22", [0x9f,0x5a,0x95,0x2b]
    assert_bytes "cmn      w22, w23, asr #31", [0xdf,0x7e,0x97,0x2b]
    assert_bytes "cmn      w3, #291, lsl #12", [0x7f,0x8c,0x44,0x31]
    assert_bytes "cmn      w5, wzr", [0xbf,0x00,0x1f,0x2b]
    assert_bytes "cmn      w6, w7", [0xdf,0x00,0x07,0x2b]
    assert_bytes "cmn      w8, w9, lsl #15", [0x1f,0x3d,0x09,0x2b]
    assert_bytes "cmn      wsp, #1365", [0xff,0x57,0x15,0x31]
    assert_bytes "cmn      wsp, w6", [0xff,0x43,0x26,0x2b]
    assert_bytes "cmn      wzr, w4", [0xff,0x03,0x04,0x2b]
    assert_bytes "cmn      x0, x3", [0x1f,0x00,0x03,0xab]
    assert_bytes "cmn      x1, w20, uxtw", [0x3f,0x40,0x34,0xab]
    assert_bytes "cmn      x10, x11, lsl #63", [0x5f,0xfd,0x0b,0xab]
    assert_bytes "cmn      x12, x13, lsr #0", [0x9f,0x01,0x4d,0xab]
    assert_bytes "cmn      x14, x15, lsr #41", [0xdf,0xa5,0x4f,0xab]
    assert_bytes "cmn      x16, x17, lsr #63", [0x1f,0xfe,0x51,0xab]
    assert_bytes "cmn      x18, x19, asr #0", [0x5f,0x02,0x93,0xab]
    assert_bytes "cmn      x2, w3, sxtw", [0x5f,0xc0,0x23,0xab]
    assert_bytes "cmn      x20, x21, asr #55", [0x9f,0xde,0x95,0xab]
    assert_bytes "cmn      x22, x23, asr #63", [0xdf,0xfe,0x97,0xab]
    assert_bytes "cmn      x25, w20, sxtb #3", [0x3f,0x8f,0x34,0xab]
    assert_bytes "cmn      x3, x13, uxtx", [0x7f,0x60,0x2d,0xab]
    assert_bytes "cmn      x4, w5, uxtb #2", [0x9f,0x08,0x25,0xab]
    assert_bytes "cmn      x5, x9, sxtx #2", [0xbf,0xe8,0x29,0xab]
    assert_bytes "cmn      x5, xzr", [0xbf,0x00,0x1f,0xab]
    assert_bytes "cmn      x6, x7", [0xdf,0x00,0x07,0xab]
    assert_bytes "cmn      x8, x9, lsl #15", [0x1f,0x3d,0x09,0xab]
    assert_bytes "cmn      xzr, x4", [0xff,0x03,0x04,0xab]
    assert_bytes "cmn w2, w3, sxtw", [0x5f,0xc0,0x23,0x2b]
    assert_bytes "cmn wsp, w3, lsl #4", [0xff,0x53,0x23,0x2b]
  end

  def test_generated_cmp
    assert_bytes "cmp      sp, #200", [0xff,0x23,0x03,0xf1]
    assert_bytes "cmp      sp, w19, sxth", [0xff,0xa3,0x33,0xeb]
    assert_bytes "cmp      sp, w19, uxth #4", [0xff,0x33,0x33,0xeb]
    assert_bytes "cmp      sp, x6", [0xff,0x63,0x26,0xeb]
    assert_bytes "cmp      w0, w3", [0x1f,0x00,0x03,0x6b]
    assert_bytes "cmp      w10, w11, lsl #31", [0x5f,0x7d,0x0b,0x6b]
    assert_bytes "cmp      w12, w13, lsr #0", [0x9f,0x01,0x4d,0x6b]
    assert_bytes "cmp      w14, w15, lsr #21", [0xdf,0x55,0x4f,0x6b]
    assert_bytes "cmp      w15, w17, uxth", [0xff,0x21,0x31,0x6b]
    assert_bytes "cmp      w16, w17, lsr #31", [0x1f,0x7e,0x51,0x6b]
    assert_bytes "cmp      w17, w1, uxtx", [0x3f,0x62,0x21,0x6b]
    assert_bytes "cmp      w18, w19, asr #0", [0x5f,0x02,0x93,0x6b]
    assert_bytes "cmp      w2, w3, sxtw", [0x5f,0xc0,0x23,0x6b]
    assert_bytes "cmp      w20, w21, asr #22", [0x9f,0x5a,0x95,0x6b]
    assert_bytes "cmp      w22, w23, asr #31", [0xdf,0x7e,0x97,0x6b]
    assert_bytes "cmp      w29, wzr, uxtw", [0xbf,0x43,0x3f,0x6b]
    assert_bytes "cmp      w3, w5, sxtx", [0x7f,0xe0,0x25,0x6b]
    assert_bytes "cmp      w5, w1, sxtb #1", [0xbf,0x84,0x21,0x6b]
    assert_bytes "cmp      w5, w7, uxtb", [0xbf,0x00,0x27,0x6b]
    assert_bytes "cmp      w5, wzr", [0xbf,0x00,0x1f,0x6b]
    assert_bytes "cmp      w6, w7", [0xdf,0x00,0x07,0x6b]
    assert_bytes "cmp      w8, w9, lsl #15", [0x1f,0x3d,0x09,0x6b]
    assert_bytes "cmp      wsp, #500", [0xff,0xd3,0x07,0x71]
    assert_bytes "cmp      wsp, w1, uxtb", [0xff,0x03,0x21,0x6b]
    assert_bytes "cmp      wsp, w19, sxth", [0xff,0xa3,0x33,0x6b]
    assert_bytes "cmp      wsp, w6", [0xff,0x43,0x26,0x6b]
    assert_bytes "cmp      wsp, w9", [0xff,0x43,0x29,0x6b]
    assert_bytes "cmp      wzr, w0", [0xff,0x03,0x00,0x6b]
    assert_bytes "cmp      wzr, w4", [0xff,0x03,0x04,0x6b]
    assert_bytes "cmp      x0, x3", [0x1f,0x00,0x03,0xeb]
    assert_bytes "cmp      x1, w20, uxtw", [0x3f,0x40,0x34,0xeb]
    assert_bytes "cmp      x10, x11, lsl #63", [0x5f,0xfd,0x0b,0xeb]
    assert_bytes "cmp      x12, x13, lsr #0", [0x9f,0x01,0x4d,0xeb]
    assert_bytes "cmp      x12, x13, uxtx #4", [0x9f,0x71,0x2d,0xeb]
    assert_bytes "cmp      x14, x15, lsr #41", [0xdf,0xa5,0x4f,0xeb]
    assert_bytes "cmp      x16, x17, lsr #63", [0x1f,0xfe,0x51,0xeb]
    assert_bytes "cmp      x18, x19, asr #0", [0x5f,0x02,0x93,0xeb]
    assert_bytes "cmp      x2, w3, sxtw", [0x5f,0xc0,0x23,0xeb]
    assert_bytes "cmp      x20, w29, uxtb #3", [0x9f,0x0e,0x3d,0xeb]
    assert_bytes "cmp      x20, x21, asr #55", [0x9f,0xde,0x95,0xeb]
    assert_bytes "cmp      x22, x23, asr #63", [0xdf,0xfe,0x97,0xeb]
    assert_bytes "cmp      x25, w20, sxtb #3", [0x3f,0x8f,0x34,0xeb]
    assert_bytes "cmp      x3, x13, uxtx", [0x7f,0x60,0x2d,0xeb]
    assert_bytes "cmp      x4, #300, lsl #12", [0x9f,0xb0,0x44,0xf1]
    assert_bytes "cmp      x4, w5, uxtb #2", [0x9f,0x08,0x25,0xeb]
    assert_bytes "cmp      x5, x9, sxtx #2", [0xbf,0xe8,0x29,0xeb]
    assert_bytes "cmp      x5, xzr", [0xbf,0x00,0x1f,0xeb]
    assert_bytes "cmp      x6, x7", [0xdf,0x00,0x07,0xeb]
    assert_bytes "cmp      x8, x9, lsl #15", [0x1f,0x3d,0x09,0xeb]
    assert_bytes "cmp      xzr, x4", [0xff,0x03,0x04,0xeb]
    assert_bytes "cmp     xzr, x0", [0xff,0x03,0x00,0xeb]
  end

  def test_generated_cneg
    assert_bytes "cneg    w3, w5, gt", [0xa3,0xd4,0x85,0x5a]
    assert_bytes "cneg    w9, wzr, lt", [0xe9,0xa7,0x9f,0x5a]
    assert_bytes "cneg    wzr, w4, le", [0x9f,0xc4,0x84,0x5a]
    assert_bytes "cneg    x3, x5, gt", [0xa3,0xd4,0x85,0xda]
    assert_bytes "cneg    x9, xzr, lt", [0xe9,0xa7,0x9f,0xda]
    assert_bytes "cneg    xzr, x4, le", [0x9f,0xc4,0x84,0xda]
  end

  def test_generated_csel
    assert_bytes "csel     w1, w0, w19, ne", [0x01,0x10,0x93,0x1a]
    assert_bytes "csel     w1, w28, wzr, mi", [0x81,0x43,0x9f,0x1a]
    assert_bytes "csel     w9, wzr, w30, gt", [0xe9,0xc3,0x9e,0x1a]
    assert_bytes "csel     wzr, w5, w9, eq", [0xbf,0x00,0x89,0x1a]
    assert_bytes "csel     x19, x23, x29, lt", [0xf3,0xb2,0x9d,0x9a]
    assert_bytes "csel     x5, xzr, x6, hs", [0xe5,0x23,0x86,0x9a]
    assert_bytes "csel     x7, x8, xzr, lo", [0x07,0x31,0x9f,0x9a]
    assert_bytes "csel     xzr, x3, x4, ge", [0x7f,0xa0,0x84,0x9a]
  end

  def test_generated_cset
    assert_bytes "cset     x9, lt", [0xe9,0xa7,0x9f,0x9a]
    assert_bytes "cset    w3, eq", [0xe3,0x17,0x9f,0x1a]
    assert_bytes "cset    w9, lt", [0xe9,0xa7,0x9f,0x1a]
    assert_bytes "cset    x9, pl", [0xe9,0x47,0x9f,0x9a]
  end

  def test_generated_csetm
    assert_bytes "csetm    w20, ne", [0xf4,0x03,0x9f,0x5a]
    assert_bytes "csetm    w9, lt", [0xe9,0xa3,0x9f,0x5a]
    assert_bytes "csetm    x30, ge", [0xfe,0xb3,0x9f,0xda]
    assert_bytes "csetm    x9, lt", [0xe9,0xa3,0x9f,0xda]
  end

  def test_generated_csinc
    assert_bytes "csinc    w1, w0, w19, ne", [0x01,0x14,0x93,0x1a]
    assert_bytes "csinc    w1, w28, wzr, mi", [0x81,0x47,0x9f,0x1a]
    assert_bytes "csinc    w9, wzr, w30, gt", [0xe9,0xc7,0x9e,0x1a]
    assert_bytes "csinc    wzr, w5, w9, eq", [0xbf,0x04,0x89,0x1a]
    assert_bytes "csinc    x19, x23, x29, lt", [0xf3,0xb6,0x9d,0x9a]
    assert_bytes "csinc    x5, xzr, x6, hs", [0xe5,0x27,0x86,0x9a]
    assert_bytes "csinc    x7, x8, xzr, lo", [0x07,0x35,0x9f,0x9a]
    assert_bytes "csinc    xzr, x3, x4, ge", [0x7f,0xa4,0x84,0x9a]
  end

  def test_generated_csinv
    assert_bytes "csinv    w1, w0, w19, ne", [0x01,0x10,0x93,0x5a]
    assert_bytes "csinv    w1, w28, wzr, mi", [0x81,0x43,0x9f,0x5a]
    assert_bytes "csinv    w9, wzr, w30, gt", [0xe9,0xc3,0x9e,0x5a]
    assert_bytes "csinv    wzr, w5, w9, eq", [0xbf,0x00,0x89,0x5a]
    assert_bytes "csinv    x19, x23, x29, lt", [0xf3,0xb2,0x9d,0xda]
    assert_bytes "csinv    x5, xzr, x6, hs", [0xe5,0x23,0x86,0xda]
    assert_bytes "csinv    x7, x8, xzr, lo", [0x07,0x31,0x9f,0xda]
    assert_bytes "csinv    xzr, x3, x4, ge", [0x7f,0xa0,0x84,0xda]
  end

  def test_generated_csneg
    assert_bytes "csneg    w1, w0, w19, ne", [0x01,0x14,0x93,0x5a]
    assert_bytes "csneg    w1, w28, wzr, mi", [0x81,0x47,0x9f,0x5a]
    assert_bytes "csneg    w9, wzr, w30, gt", [0xe9,0xc7,0x9e,0x5a]
    assert_bytes "csneg    wzr, w5, w9, eq", [0xbf,0x04,0x89,0x5a]
    assert_bytes "csneg    x19, x23, x29, lt", [0xf3,0xb6,0x9d,0xda]
    assert_bytes "csneg    x5, xzr, x6, hs", [0xe5,0x27,0x86,0xda]
    assert_bytes "csneg    x7, x8, xzr, lo", [0x07,0x35,0x9f,0xda]
    assert_bytes "csneg    xzr, x3, x4, ge", [0x7f,0xa4,0x84,0xda]
  end

  def test_generated_dcps1
    assert_bytes "dcps1", [0x01,0x00,0xa0,0xd4]
  end

  def test_generated_dcps2
    assert_bytes "dcps2", [0x02,0x00,0xa0,0xd4]
  end

  def test_generated_dcps3
    assert_bytes "dcps3", [0x03,0x00,0xa0,0xd4]
  end

  def test_generated_dmb
    assert_bytes "dmb     #0", [0xbf,0x30,0x03,0xd5]
    assert_bytes "dmb     #12", [0xbf,0x3c,0x03,0xd5]
    assert_bytes "dmb     ish", [0xbf,0x3b,0x03,0xd5]
    assert_bytes "dmb     ishld", [0xbf,0x39,0x03,0xd5]
    assert_bytes "dmb     ishst", [0xbf,0x3a,0x03,0xd5]
    assert_bytes "dmb     ld", [0xbf,0x3d,0x03,0xd5]
    assert_bytes "dmb     nsh", [0xbf,0x37,0x03,0xd5]
    assert_bytes "dmb     nshld", [0xbf,0x35,0x03,0xd5]
    assert_bytes "dmb     nshst", [0xbf,0x36,0x03,0xd5]
    assert_bytes "dmb     osh", [0xbf,0x33,0x03,0xd5]
    assert_bytes "dmb     oshld", [0xbf,0x31,0x03,0xd5]
    assert_bytes "dmb     oshst", [0xbf,0x32,0x03,0xd5]
    assert_bytes "dmb     st", [0xbf,0x3e,0x03,0xd5]
    assert_bytes "dmb     sy", [0xbf,0x3f,0x03,0xd5]
  end

  def test_generated_drps
    assert_bytes "drps", [0xe0,0x03,0xbf,0xd6]
  end

  def test_generated_dsb
    assert_bytes "dsb     #12", [0x9f,0x3c,0x03,0xd5]
    assert_bytes "dsb     ish", [0x9f,0x3b,0x03,0xd5]
    assert_bytes "dsb     ishld", [0x9f,0x39,0x03,0xd5]
    assert_bytes "dsb     ishst", [0x9f,0x3a,0x03,0xd5]
    assert_bytes "dsb     ld", [0x9f,0x3d,0x03,0xd5]
    assert_bytes "dsb     nsh", [0x9f,0x37,0x03,0xd5]
    assert_bytes "dsb     nshld", [0x9f,0x35,0x03,0xd5]
    assert_bytes "dsb     nshst", [0x9f,0x36,0x03,0xd5]
    assert_bytes "dsb     osh", [0x9f,0x33,0x03,0xd5]
    assert_bytes "dsb     oshld", [0x9f,0x31,0x03,0xd5]
    assert_bytes "dsb     oshst", [0x9f,0x32,0x03,0xd5]
    assert_bytes "dsb     st", [0x9f,0x3e,0x03,0xd5]
    assert_bytes "dsb     sy", [0x9f,0x3f,0x03,0xd5]
  end

  def test_generated_eor
    assert_bytes "eor  w16, w17, #0xfffffff9", [0x30,0x76,0x1d,0x52]
    assert_bytes "eor      sp, xzr, #0x303030303030303", [0xff,0xc7,0x00,0xd2]
    assert_bytes "eor      w16, w17, #0x81818181", [0x30,0xc6,0x01,0x52]
    assert_bytes "eor      w3, w6, #0xe0e0e0e0", [0xc3,0xc8,0x03,0x52]
    assert_bytes "eor      wsp, wzr, #0x3030303", [0xff,0xc7,0x00,0x52]
    assert_bytes "eor      x16, x17, #0x8181818181818181", [0x30,0xc6,0x01,0xd2]
    assert_bytes "eor      x3, x5, #0xffffffffc000000", [0xa3,0x84,0x66,0xd2]
    assert_bytes "eor      x3, x6, #0xe0e0e0e0e0e0e0e0", [0xc3,0xc8,0x03,0xd2]
  end

  def test_generated_eret
    assert_bytes "eret", [0xe0,0x03,0x9f,0xd6]
  end

  def test_generated_extr
    assert_bytes "extr     w11, w13, w17, #31", [0xab,0x7d,0x91,0x13]
    assert_bytes "extr     w3, w5, w7, #0", [0xa3,0x00,0x87,0x13]
    assert_bytes "extr     x11, x13, x17, #63", [0xab,0xfd,0xd1,0x93]
    assert_bytes "extr     x3, x5, x7, #15", [0xa3,0x3c,0xc7,0x93]
  end

  def test_generated_isb
    assert_bytes "isb", [0xdf,0x3f,0x03,0xd5]
    assert_bytes "isb     #12", [0xdf,0x3c,0x03,0xd5]
  end

  def test_generated_ldar
    assert_bytes "ldar     wzr, [x1]", [0x3f,0xfc,0xdf,0x88]
    assert_bytes "ldar     x1, [x2]", [0x41,0xfc,0xdf,0xc8]
  end

  def test_generated_ldarb
    assert_bytes "ldarb    w29, [sp]", [0xfd,0xff,0xdf,0x08]
  end

  def test_generated_ldarh
    assert_bytes "ldarh    w30, [x0]", [0x1e,0xfc,0xdf,0x48]
  end

  def test_generated_ldaxp
    assert_bytes "ldaxp    w26, wzr, [sp]", [0xfa,0xff,0x7f,0x88]
    assert_bytes "ldaxp    x27, x28, [x30]", [0xdb,0xf3,0x7f,0xc8]
  end

  def test_generated_ldaxr
    assert_bytes "ldaxr    wzr, [x22]", [0xdf,0xfe,0x5f,0x88]
    assert_bytes "ldaxr    x21, [x23]", [0xf5,0xfe,0x5f,0xc8]
  end

  def test_generated_ldaxrb
    assert_bytes "ldaxrb   w19, [x21]", [0xb3,0xfe,0x5f,0x08]
  end

  def test_generated_ldaxrh
    assert_bytes "ldaxrh   w20, [sp]", [0xf4,0xff,0x5f,0x48]
  end

  def test_generated_ldnp
    assert_bytes "ldnp      w2, wzr, [sp, #-256]", [0xe2,0x7f,0x60,0x28]
    assert_bytes "ldnp      w3, w5, [sp]", [0xe3,0x17,0x40,0x28]
    assert_bytes "ldnp      w9, w10, [sp, #4]", [0xe9,0xab,0x40,0x28]
    assert_bytes "ldnp      x21, x29, [x2, #504]", [0x55,0xf4,0x5f,0xa8]
    assert_bytes "ldnp      x22, x23, [x3, #-512]", [0x76,0x5c,0x60,0xa8]
    assert_bytes "ldnp      x24, x25, [x4, #8]", [0x98,0xe4,0x40,0xa8]
  end

  def test_generated_ldp
    assert_bytes "ldp      w2, wzr, [sp, #-256]", [0xe2,0x7f,0x60,0x29]
    assert_bytes "ldp      w2, wzr, [sp, #-256]!", [0xe2,0x7f,0xe0,0x29]
    assert_bytes "ldp      w2, wzr, [sp], #-256", [0xe2,0x7f,0xe0,0x28]
    assert_bytes "ldp      w3, w5, [sp, #0]!", [0xe3,0x17,0xc0,0x29]
    assert_bytes "ldp      w3, w5, [sp]", [0xe3,0x17,0x40,0x29]
    assert_bytes "ldp      w3, w5, [sp], #0", [0xe3,0x17,0xc0,0x28]
    assert_bytes "ldp      w9, w10, [sp, #4]", [0xe9,0xab,0x40,0x29]
    assert_bytes "ldp      w9, w10, [sp, #4]!", [0xe9,0xab,0xc0,0x29]
    assert_bytes "ldp      w9, w10, [sp], #4", [0xe9,0xab,0xc0,0x28]
    assert_bytes "ldp      x21, x29, [x2, #504]", [0x55,0xf4,0x5f,0xa9]
    assert_bytes "ldp      x21, x29, [x2, #504]!", [0x55,0xf4,0xdf,0xa9]
    assert_bytes "ldp      x21, x29, [x2], #504", [0x55,0xf4,0xdf,0xa8]
    assert_bytes "ldp      x22, x23, [x3, #-512]", [0x76,0x5c,0x60,0xa9]
    assert_bytes "ldp      x22, x23, [x3, #-512]!", [0x76,0x5c,0xe0,0xa9]
    assert_bytes "ldp      x22, x23, [x3], #-512", [0x76,0x5c,0xe0,0xa8]
    assert_bytes "ldp      x24, x25, [x4, #8]", [0x98,0xe4,0x40,0xa9]
    assert_bytes "ldp      x24, x25, [x4, #8]!", [0x98,0xe4,0xc0,0xa9]
    assert_bytes "ldp      x24, x25, [x4], #8", [0x98,0xe4,0xc0,0xa8]
  end

  def test_generated_ldpsw
    assert_bytes "ldpsw    x20, x30, [sp, #252]", [0xf4,0xfb,0x5f,0x69]
    assert_bytes "ldpsw    x20, x30, [sp, #252]!", [0xf4,0xfb,0xdf,0x69]
    assert_bytes "ldpsw    x20, x30, [sp], #252", [0xf4,0xfb,0xdf,0x68]
    assert_bytes "ldpsw    x9, x10, [sp, #4]", [0xe9,0xab,0x40,0x69]
    assert_bytes "ldpsw    x9, x10, [sp, #4]!", [0xe9,0xab,0xc0,0x69]
    assert_bytes "ldpsw    x9, x10, [sp], #4", [0xe9,0xab,0xc0,0x68]
    assert_bytes "ldpsw    x9, x10, [x2, #-256]", [0x49,0x28,0x60,0x69]
    assert_bytes "ldpsw    x9, x10, [x2, #-256]!", [0x49,0x28,0xe0,0x69]
    assert_bytes "ldpsw    x9, x10, [x2], #-256", [0x49,0x28,0xe0,0x68]
  end

  def test_generated_ldr
    assert_bytes "ldr      w10, [x30, x7, lsl #2]", [0xca,0x7b,0x67,0xb8]
    assert_bytes "ldr      w11, [x29, x3, sxtx]", [0xab,0xeb,0x63,0xb8]
    assert_bytes "ldr      w13, [x2, #4]", [0x4d,0x04,0x40,0xb9]
    assert_bytes "ldr      w15, [x25, w7, uxtw]", [0x2f,0x4b,0x67,0xb8]
    assert_bytes "ldr      w16, [x24, w8, uxtw #2]", [0x10,0x5b,0x68,0xb8]
    assert_bytes "ldr      w17, [sp, #16380]", [0xf1,0xff,0x7f,0xb9]
    assert_bytes "ldr      w18, [x22, w10, sxtw]", [0xd2,0xca,0x6a,0xb8]
    assert_bytes "ldr      w19, [sp, #255]!", [0xf3,0xff,0x4f,0xb8]
    assert_bytes "ldr      w19, [sp], #255", [0xf3,0xf7,0x4f,0xb8]
    assert_bytes "ldr      w2, [sp]", [0xe2,0x03,0x40,0xb9]
    assert_bytes "ldr      w20, [x30, #1]!", [0xd4,0x1f,0x40,0xb8]
    assert_bytes "ldr      w20, [x30], #1", [0xd4,0x17,0x40,0xb8]
    assert_bytes "ldr      w21, [x12, #-256]!", [0x95,0x0d,0x50,0xb8]
    assert_bytes "ldr      w21, [x12], #-256", [0x95,0x05,0x50,0xb8]
    assert_bytes "ldr      w3, [sp, x5]", [0xe3,0x6b,0x65,0xb8]
    assert_bytes "ldr      x0, [x0]", [0x00,0x00,0x40,0xf9]
    assert_bytes "ldr      x12, [x28, xzr, sxtx]", [0x8c,0xeb,0x7f,0xf8]
    assert_bytes "ldr      x13, [x27, x5, sxtx #3]", [0x6d,0xfb,0x65,0xf8]
    assert_bytes "ldr      x15, [x25, w7, uxtw]", [0x2f,0x4b,0x67,0xf8]
    assert_bytes "ldr      x16, [x24, w8, uxtw #3]", [0x10,0x5b,0x68,0xf8]
    assert_bytes "ldr      x17, [x23, w9, sxtw]", [0xf1,0xca,0x69,0xf8]
    assert_bytes "ldr      x18, [x22, w10, sxtw]", [0xd2,0xca,0x6a,0xf8]
    assert_bytes "ldr      x19, [x12, #-256]!", [0x93,0x0d,0x50,0xf8]
    assert_bytes "ldr      x19, [x12], #-256", [0x93,0x05,0x50,0xf8]
    assert_bytes "ldr      x2, [x3, #1]!", [0x62,0x1c,0x40,0xf8]
    assert_bytes "ldr      x2, [x3], #1", [0x62,0x14,0x40,0xf8]
    assert_bytes "ldr      x20, [sp, #8]", [0xf4,0x07,0x40,0xf9]
    assert_bytes "ldr      x3, [sp, x5]", [0xe3,0x6b,0x65,0xf8]
    assert_bytes "ldr      x3, [x4, #0]!", [0x83,0x0c,0x40,0xf8]
    assert_bytes "ldr      x30, [x12, #32760]", [0x9e,0xfd,0x7f,0xf9]
    assert_bytes "ldr      x4, [x29]", [0xa4,0x03,0x40,0xf9]
    assert_bytes "ldr      xzr, [sp, #0]!", [0xff,0x0f,0x40,0xf8]
    assert_bytes "ldr      xzr, [sp]", [0xff,0x03,0x40,0xf9]
    assert_bytes "ldr      xzr, [x9, #255]!", [0x3f,0xfd,0x4f,0xf8]
    assert_bytes "ldr      xzr, [x9], #255", [0x3f,0xf5,0x4f,0xf8]
    assert_bytes "ldr     w0, #1048572", [0xe0,0xff,0x7f,0x18]
    assert_bytes "ldr     x10, #-1048576", [0x0a,0x00,0x80,0x58]
  end

  def test_generated_ldrb
    assert_bytes "ldrb     w10, [x3, #-256]!", [0x6a,0x0c,0x50,0x38]
    assert_bytes "ldrb     w10, [x3, #1]!", [0x6a,0x1c,0x40,0x38]
    assert_bytes "ldrb     w10, [x3], #-256", [0x6a,0x04,0x50,0x38]
    assert_bytes "ldrb     w10, [x3], #1", [0x6a,0x14,0x40,0x38]
    assert_bytes "ldrb     w11, [x29, x3, sxtx]", [0xab,0xeb,0x63,0x38]
    assert_bytes "ldrb     w12, [x2]", [0x4c,0x00,0x40,0x39]
    assert_bytes "ldrb     w14, [x26, w6, uxtw]", [0x4e,0x4b,0x66,0x38]
    assert_bytes "ldrb     w17, [x23, w9, sxtw]", [0xf1,0xca,0x69,0x38]
    assert_bytes "ldrb     w26, [x3, #121]", [0x7a,0xe4,0x41,0x39]
    assert_bytes "ldrb     w3, [sp, x5]", [0xe3,0x6b,0x65,0x38]
    assert_bytes "ldrb     w9, [x2, #255]!", [0x49,0xfc,0x4f,0x38]
    assert_bytes "ldrb     w9, [x27, x6, lsl #0]", [0x69,0x7b,0x66,0x38]
    assert_bytes "ldrb     w9, [x2], #255", [0x49,0xf4,0x4f,0x38]
  end

  def test_generated_ldrh
    assert_bytes "ldrh     w10, [x3, #-256]!", [0x6a,0x0c,0x50,0x78]
    assert_bytes "ldrh     w10, [x30, x7, lsl #1]", [0xca,0x7b,0x67,0x78]
    assert_bytes "ldrh     w10, [x3], #-256", [0x6a,0x04,0x50,0x78]
    assert_bytes "ldrh     w12, [x28, xzr, sxtx]", [0x8c,0xeb,0x7f,0x78]
    assert_bytes "ldrh     w14, [x26, w6, uxtw]", [0x4e,0x4b,0x66,0x78]
    assert_bytes "ldrh     w15, [x25, w7, uxtw]", [0x2f,0x4b,0x67,0x78]
    assert_bytes "ldrh     w17, [x23, w9, sxtw]", [0xf1,0xca,0x69,0x78]
    assert_bytes "ldrh     w18, [x22, w10, sxtw]", [0xd2,0xca,0x6a,0x78]
    assert_bytes "ldrh     w2, [x4]", [0x82,0x00,0x40,0x79]
    assert_bytes "ldrh     w9, [x2, #1]!", [0x49,0x1c,0x40,0x78]
    assert_bytes "ldrh     w9, [x2, #255]!", [0x49,0xfc,0x4f,0x78]
    assert_bytes "ldrh     w9, [x2], #1", [0x49,0x14,0x40,0x78]
    assert_bytes "ldrh     w9, [x2], #255", [0x49,0xf4,0x4f,0x78]
  end

  def test_generated_ldrsb
    assert_bytes "ldrsb    w10, [x30, x7]", [0xca,0x6b,0xe7,0x38]
    assert_bytes "ldrsb    w15, [x25, w7, uxtw #0]", [0x2f,0x5b,0xe7,0x38]
    assert_bytes "ldrsb    w19, [x12, #-256]!", [0x93,0x0d,0xd0,0x38]
    assert_bytes "ldrsb    w19, [x12], #-256", [0x93,0x05,0xd0,0x38]
    assert_bytes "ldrsb    w2, [x3, #1]!", [0x62,0x1c,0xc0,0x38]
    assert_bytes "ldrsb    w2, [x3], #1", [0x62,0x14,0xc0,0x38]
    assert_bytes "ldrsb    w27, [sp, #4095]", [0xfb,0xff,0xff,0x39]
    assert_bytes "ldrsb    wzr, [x9, #255]!", [0x3f,0xfd,0xcf,0x38]
    assert_bytes "ldrsb    wzr, [x9], #255", [0x3f,0xf5,0xcf,0x38]
    assert_bytes "ldrsb    x18, [x22, w10, sxtw #0]", [0xd2,0xda,0xaa,0x38]
    assert_bytes "ldrsb    x19, [x12, #-256]!", [0x93,0x0d,0x90,0x38]
    assert_bytes "ldrsb    x19, [x12], #-256", [0x93,0x05,0x90,0x38]
    assert_bytes "ldrsb    x2, [x3, #1]!", [0x62,0x1c,0x80,0x38]
    assert_bytes "ldrsb    x2, [x3], #1", [0x62,0x14,0x80,0x38]
    assert_bytes "ldrsb    xzr, [x15]", [0xff,0x01,0x80,0x39]
    assert_bytes "ldrsb    xzr, [x9, #255]!", [0x3f,0xfd,0x8f,0x38]
    assert_bytes "ldrsb    xzr, [x9], #255", [0x3f,0xf5,0x8f,0x38]
  end

  def test_generated_ldrsh
    assert_bytes "ldrsh    w16, [x24, w8, uxtw #1]", [0x10,0x5b,0xe8,0x78]
    assert_bytes "ldrsh    w19, [x12, #-256]!", [0x93,0x0d,0xd0,0x78]
    assert_bytes "ldrsh    w19, [x12], #-256", [0x93,0x05,0xd0,0x78]
    assert_bytes "ldrsh    w2, [x3, #1]!", [0x62,0x1c,0xc0,0x78]
    assert_bytes "ldrsh    w2, [x3], #1", [0x62,0x14,0xc0,0x78]
    assert_bytes "ldrsh    w23, [x6, #8190]", [0xd7,0xfc,0xff,0x79]
    assert_bytes "ldrsh    w3, [sp, x5]", [0xe3,0x6b,0xe5,0x78]
    assert_bytes "ldrsh    w9, [x27, x6]", [0x69,0x6b,0xe6,0x78]
    assert_bytes "ldrsh    wzr, [sp, #2]", [0xff,0x07,0xc0,0x79]
    assert_bytes "ldrsh    wzr, [x9, #255]!", [0x3f,0xfd,0xcf,0x78]
    assert_bytes "ldrsh    wzr, [x9], #255", [0x3f,0xf5,0xcf,0x78]
    assert_bytes "ldrsh    x13, [x27, x5, sxtx #1]", [0x6d,0xfb,0xa5,0x78]
    assert_bytes "ldrsh    x19, [x12, #-256]!", [0x93,0x0d,0x90,0x78]
    assert_bytes "ldrsh    x19, [x12], #-256", [0x93,0x05,0x90,0x78]
    assert_bytes "ldrsh    x2, [x3, #1]!", [0x62,0x1c,0x80,0x78]
    assert_bytes "ldrsh    x2, [x3], #1", [0x62,0x14,0x80,0x78]
    assert_bytes "ldrsh    x29, [x2, #2]", [0x5d,0x04,0x80,0x79]
    assert_bytes "ldrsh    xzr, [x9, #255]!", [0x3f,0xfd,0x8f,0x78]
    assert_bytes "ldrsh    xzr, [x9], #255", [0x3f,0xf5,0x8f,0x78]
  end

  def test_generated_ldrsw
    assert_bytes "ldrsw    x17, [x23, w9, sxtw]", [0xf1,0xca,0xa9,0xb8]
    assert_bytes "ldrsw    x19, [x12, #-256]!", [0x93,0x0d,0x90,0xb8]
    assert_bytes "ldrsw    x19, [x12], #-256", [0x93,0x05,0x90,0xb8]
    assert_bytes "ldrsw    x19, [x21, wzr, sxtw #2]", [0xb3,0xda,0xbf,0xb8]
    assert_bytes "ldrsw    x2, [x3, #1]!", [0x62,0x1c,0x80,0xb8]
    assert_bytes "ldrsw    x2, [x3], #1", [0x62,0x14,0x80,0xb8]
    assert_bytes "ldrsw    x2, [x5, #4]", [0xa2,0x04,0x80,0xb9]
    assert_bytes "ldrsw    x23, [sp, #16380]", [0xf7,0xff,0xbf,0xb9]
    assert_bytes "ldrsw    xzr, [x9, #255]!", [0x3f,0xfd,0x8f,0xb8]
    assert_bytes "ldrsw    xzr, [x9], #255", [0x3f,0xf5,0x8f,0xb8]
  end

  def test_generated_ldtr
    assert_bytes "ldtr     w12, [sp, #255]", [0xec,0xfb,0x4f,0xb8]
    assert_bytes "ldtr     xzr, [x12, #255]", [0x9f,0xf9,0x4f,0xf8]
  end

  def test_generated_ldtrb
    assert_bytes "ldtrb    w1, [x20, #255]", [0x81,0xfa,0x4f,0x38]
  end

  def test_generated_ldtrh
    assert_bytes "ldtrh    w20, [x1, #255]", [0x34,0xf8,0x4f,0x78]
  end

  def test_generated_ldtrsb
    assert_bytes "ldtrsb   w19, [x1, #-256]", [0x33,0x08,0xd0,0x38]
    assert_bytes "ldtrsb   x9, [x7, #-256]", [0xe9,0x08,0x90,0x38]
  end

  def test_generated_ldtrsh
    assert_bytes "ldtrsh   w15, [x21, #-256]", [0xaf,0x0a,0xd0,0x78]
    assert_bytes "ldtrsh   x17, [x19, #-256]", [0x71,0x0a,0x90,0x78]
  end

  def test_generated_ldtrsw
    assert_bytes "ldtrsw   x20, [x15, #-256]", [0xf4,0x09,0x90,0xb8]
  end

  def test_generated_ldur
    assert_bytes "ldur     w12, [sp, #255]", [0xec,0xf3,0x4f,0xb8]
    assert_bytes "ldur     xzr, [x12, #255]", [0x9f,0xf1,0x4f,0xf8]
  end

  def test_generated_ldurb
    assert_bytes "ldurb    w1, [x20, #255]", [0x81,0xf2,0x4f,0x38]
  end

  def test_generated_ldurh
    assert_bytes "ldurh    w20, [x1, #255]", [0x34,0xf0,0x4f,0x78]
  end

  def test_generated_ldursb
    assert_bytes "ldursb   w19, [x1, #-256]", [0x33,0x00,0xd0,0x38]
    assert_bytes "ldursb   x9, [x7, #-256]", [0xe9,0x00,0x90,0x38]
  end

  def test_generated_ldursh
    assert_bytes "ldursh   w15, [x21, #-256]", [0xaf,0x02,0xd0,0x78]
    assert_bytes "ldursh   x17, [x19, #-256]", [0x71,0x02,0x90,0x78]
  end

  def test_generated_ldursw
    assert_bytes "ldursw   x13, [x2]", [0x4d,0x00,0x80,0xb8]
    assert_bytes "ldursw   x20, [x15, #-256]", [0xf4,0x01,0x90,0xb8]
  end

  def test_generated_ldxp
    assert_bytes "ldxp     w12, wzr, [sp]", [0xec,0x7f,0x7f,0x88]
    assert_bytes "ldxp     x13, x14, [x15]", [0xed,0x39,0x7f,0xc8]
  end

  def test_generated_ldxr
    assert_bytes "ldxr     w9, [sp]", [0xe9,0x7f,0x5f,0x88]
    assert_bytes "ldxr     x10, [x11]", [0x6a,0x7d,0x5f,0xc8]
  end

  def test_generated_ldxrb
    assert_bytes "ldxrb    w7, [x9]", [0x27,0x7d,0x5f,0x08]
  end

  def test_generated_ldxrh
    assert_bytes "ldxrh    wzr, [x10]", [0x5f,0x7d,0x5f,0x48]
  end

  def test_generated_lsl
    assert_bytes "lsl  w11, w12, w13", [0x8b,0x21,0xcd,0x1a]
    assert_bytes "lsl  w6, w7, w8", [0xe6,0x20,0xc8,0x1a]
    assert_bytes "lsl  x14, x15, x16", [0xee,0x21,0xd0,0x9a]
    assert_bytes "lsl  x9, x10, x11", [0x49,0x21,0xcb,0x9a]
    assert_bytes "lsl      w1, wzr, #3", [0xe1,0x73,0x1d,0x53]
    assert_bytes "lsl      w11, w12, #31", [0x8b,0x01,0x01,0x53]
    assert_bytes "lsl      w13, w14, #29", [0xcd,0x09,0x03,0x53]
    assert_bytes "lsl      w9, w10, #31", [0x49,0x01,0x01,0x53]
    assert_bytes "lsl      x2, x3, #63", [0x62,0x00,0x41,0xd3]
    assert_bytes "lsl      x20, x21, #63", [0xb4,0x02,0x41,0xd3]
    assert_bytes "lsl      x9, x10, #5", [0x49,0xe9,0x7b,0xd3]
  end

  def test_generated_lsr
    assert_bytes "lsr  w12, w13, w14", [0xac,0x25,0xce,0x1a]
    assert_bytes "lsr  w17, w18, w19", [0x51,0x26,0xd3,0x1a]
    assert_bytes "lsr  x15, x16, x17", [0x0f,0x26,0xd1,0x9a]
    assert_bytes "lsr  x20, x21, x22", [0xb4,0x26,0xd6,0x9a]
    assert_bytes "lsr      w3, w2, #0", [0x43,0x7c,0x00,0x53]
    assert_bytes "lsr      w9, w10, #0", [0x49,0x7d,0x00,0x53]
    assert_bytes "lsr      w9, w10, #31", [0x49,0x7d,0x1f,0x53]
    assert_bytes "lsr      wzr, wzr, #3", [0xff,0x7f,0x03,0x53]
    assert_bytes "lsr      x19, x20, #0", [0x93,0xfe,0x40,0xd3]
    assert_bytes "lsr      x20, x21, #63", [0xb4,0xfe,0x7f,0xd3]
    assert_bytes "lsr      x5, x6, #12", [0xc5,0xfc,0x4c,0xd3]
    assert_bytes "lsr     w11, w12, #31", [0x8b,0x7d,0x1f,0x53]
    assert_bytes "lsr     w13, w14, #29", [0xcd,0x7d,0x1d,0x53]
    assert_bytes "lsr     w9, w10, #0", [0x49,0x7d,0x00,0x53]
    assert_bytes "lsr     x19, x20, #0", [0x93,0xfe,0x40,0xd3]
    assert_bytes "lsr     x2, x3, #63", [0x62,0xfc,0x7f,0xd3]
    assert_bytes "lsr     x9, x10, #5", [0x49,0xfd,0x45,0xd3]
  end

  def test_generated_madd
    assert_bytes "madd     w1, w3, w7, w4", [0x61,0x10,0x07,0x1b]
    assert_bytes "madd     w13, wzr, w4, w4", [0xed,0x13,0x04,0x1b]
    assert_bytes "madd     w19, w30, wzr, w29", [0xd3,0x77,0x1f,0x1b]
    assert_bytes "madd     wzr, w0, w9, w11", [0x1f,0x2c,0x09,0x1b]
    assert_bytes "madd     x1, x3, x7, x4", [0x61,0x10,0x07,0x9b]
    assert_bytes "madd     x13, xzr, x4, x4", [0xed,0x13,0x04,0x9b]
    assert_bytes "madd     x19, x30, xzr, x29", [0xd3,0x77,0x1f,0x9b]
    assert_bytes "madd     xzr, x0, x9, x11", [0x1f,0x2c,0x09,0x9b]
  end

  def test_generated_mneg
    assert_bytes "mneg     w21, w22, w23", [0xd5,0xfe,0x17,0x1b]
    assert_bytes "mneg     w26, wzr, w27", [0xfa,0xff,0x1b,0x1b]
    assert_bytes "mneg     w28, w29, wzr", [0xbc,0xff,0x1f,0x1b]
    assert_bytes "mneg     w4, w5, w6", [0xa4,0xfc,0x06,0x1b]
    assert_bytes "mneg     wzr, w24, w25", [0x1f,0xff,0x19,0x1b]
    assert_bytes "mneg     x4, x5, x6", [0xa4,0xfc,0x06,0x9b]
  end

  def test_generated_mov
    assert_bytes "mov      sp, x30", [0xdf,0x03,0x00,0x91]
    assert_bytes "mov      w24, wsp", [0xf8,0x03,0x00,0x11]
    assert_bytes "mov      w3, w5", [0xe3,0x03,0x05,0x2a]
    assert_bytes "mov      wsp, w20", [0x9f,0x02,0x00,0x11]
    assert_bytes "mov      wzr, w2", [0xff,0x03,0x02,0x2a]
    assert_bytes "mov      x11, sp", [0xeb,0x03,0x00,0x91]
    assert_bytes "mov      x2, #5299989643264", [0x42,0x9a,0xc0,0xd2]
    assert_bytes "mov      x3, x6", [0xe3,0x03,0x06,0xaa]
    assert_bytes "mov      x3, xzr", [0xe3,0x03,0x1f,0xaa]
    assert_bytes "mov     w1, #65535", [0xe1,0xff,0x9f,0x52]
    assert_bytes "mov     w2, #-1235", [0x42,0x9a,0x80,0x12]
    assert_bytes "mov w3, #983055", [0xe3,0x8f,0x00,0x32]
    assert_bytes "mov x10, #-6148914691236517206", [0xea,0xf3,0x01,0xb2]
  end

  def test_generated_movz
    assert_bytes "movz     w2, #0, lsl #16", [0x02,0x00,0xa0,0x52]
  end

  def test_generated_msub
    assert_bytes "msub     w1, w3, w7, w4", [0x61,0x90,0x07,0x1b]
    assert_bytes "msub     w13, wzr, w4, w4", [0xed,0x93,0x04,0x1b]
    assert_bytes "msub     w19, w30, wzr, w29", [0xd3,0xf7,0x1f,0x1b]
    assert_bytes "msub     wzr, w0, w9, w11", [0x1f,0xac,0x09,0x1b]
    assert_bytes "msub     x1, x3, x7, x4", [0x61,0x90,0x07,0x9b]
    assert_bytes "msub     x13, xzr, x4, x4", [0xed,0x93,0x04,0x9b]
    assert_bytes "msub     x19, x30, xzr, x29", [0xd3,0xf7,0x1f,0x9b]
    assert_bytes "msub     xzr, x0, x9, x11", [0x1f,0xac,0x09,0x9b]
  end

  def test_generated_mul
    assert_bytes "mul      w10, w11, wzr", [0x6a,0x7d,0x1f,0x1b]
    assert_bytes "mul      w3, w4, w5", [0x83,0x7c,0x05,0x1b]
    assert_bytes "mul      w4, w5, w6", [0xa4,0x7c,0x06,0x1b]
    assert_bytes "mul      w8, wzr, w9", [0xe8,0x7f,0x09,0x1b]
    assert_bytes "mul      wzr, w6, w7", [0xdf,0x7c,0x07,0x1b]
    assert_bytes "mul      x12, x13, x14", [0xac,0x7d,0x0e,0x9b]
    assert_bytes "mul      x17, xzr, x18", [0xf1,0x7f,0x12,0x9b]
    assert_bytes "mul      x19, x20, xzr", [0x93,0x7e,0x1f,0x9b]
    assert_bytes "mul      x4, x5, x6", [0xa4,0x7c,0x06,0x9b]
    assert_bytes "mul      xzr, x15, x16", [0xff,0x7d,0x10,0x9b]
  end

  def test_generated_neg
    assert_bytes "neg      w12, w11, asr #31", [0xec,0x7f,0x8b,0x4b]
    assert_bytes "neg      w14, w13, asr #12", [0xee,0x33,0x8d,0x4b]
    assert_bytes "neg      w16, w15, asr #0", [0xf0,0x03,0x8f,0x4b]
    assert_bytes "neg      w18, w17, lsr #31", [0xf2,0x7f,0x51,0x4b]
    assert_bytes "neg      w20, w19, lsr #1", [0xf4,0x07,0x53,0x4b]
    assert_bytes "neg      w20, w4", [0xf4,0x03,0x04,0x4b]
    assert_bytes "neg      w22, w21, lsr #0", [0xf6,0x03,0x55,0x4b]
    assert_bytes "neg      w24, w23, lsl #31", [0xf8,0x7f,0x17,0x4b]
    assert_bytes "neg      w26, w25, lsl #29", [0xfa,0x77,0x19,0x4b]
    assert_bytes "neg      w28, w27", [0xfc,0x03,0x1b,0x4b]
    assert_bytes "neg      w29, w30", [0xfd,0x03,0x1e,0x4b]
    assert_bytes "neg      w30, wzr", [0xfe,0x03,0x1f,0x4b]
    assert_bytes "neg      wzr, w0", [0xff,0x03,0x00,0x4b]
    assert_bytes "neg      x12, x11, asr #31", [0xec,0x7f,0x8b,0xcb]
    assert_bytes "neg      x14, x13, asr #12", [0xee,0x33,0x8d,0xcb]
    assert_bytes "neg      x16, x15, asr #0", [0xf0,0x03,0x8f,0xcb]
    assert_bytes "neg      x18, x17, lsr #31", [0xf2,0x7f,0x51,0xcb]
    assert_bytes "neg      x20, x19, lsr #1", [0xf4,0x07,0x53,0xcb]
    assert_bytes "neg      x20, x4", [0xf4,0x03,0x04,0xcb]
    assert_bytes "neg      x22, x21, lsr #0", [0xf6,0x03,0x55,0xcb]
    assert_bytes "neg      x24, x23, lsl #31", [0xf8,0x7f,0x17,0xcb]
    assert_bytes "neg      x26, x25, lsl #29", [0xfa,0x77,0x19,0xcb]
    assert_bytes "neg      x28, x27", [0xfc,0x03,0x1b,0xcb]
    assert_bytes "neg      x29, x30", [0xfd,0x03,0x1e,0xcb]
    assert_bytes "neg      x30, xzr", [0xfe,0x03,0x1f,0xcb]
    assert_bytes "neg      xzr, x0", [0xff,0x03,0x00,0xcb]
  end

  def test_generated_negs
    assert_bytes "negs     w12, w11, asr #31", [0xec,0x7f,0x8b,0x6b]
    assert_bytes "negs     w14, w13, asr #12", [0xee,0x33,0x8d,0x6b]
    assert_bytes "negs     w16, w15, asr #0", [0xf0,0x03,0x8f,0x6b]
    assert_bytes "negs     w18, w17, lsr #31", [0xf2,0x7f,0x51,0x6b]
    assert_bytes "negs     w20, w19, lsr #1", [0xf4,0x07,0x53,0x6b]
    assert_bytes "negs     w20, w4", [0xf4,0x03,0x04,0x6b]
    assert_bytes "negs     w22, w21, lsr #0", [0xf6,0x03,0x55,0x6b]
    assert_bytes "negs     w24, w23, lsl #31", [0xf8,0x7f,0x17,0x6b]
    assert_bytes "negs     w26, w25, lsl #29", [0xfa,0x77,0x19,0x6b]
    assert_bytes "negs     w28, w27", [0xfc,0x03,0x1b,0x6b]
    assert_bytes "negs     w29, w30", [0xfd,0x03,0x1e,0x6b]
    assert_bytes "negs     w30, wzr", [0xfe,0x03,0x1f,0x6b]
    assert_bytes "negs     x12, x11, asr #31", [0xec,0x7f,0x8b,0xeb]
    assert_bytes "negs     x14, x13, asr #12", [0xee,0x33,0x8d,0xeb]
    assert_bytes "negs     x16, x15, asr #0", [0xf0,0x03,0x8f,0xeb]
    assert_bytes "negs     x18, x17, lsr #31", [0xf2,0x7f,0x51,0xeb]
    assert_bytes "negs     x20, x19, lsr #1", [0xf4,0x07,0x53,0xeb]
    assert_bytes "negs     x20, x4", [0xf4,0x03,0x04,0xeb]
    assert_bytes "negs     x22, x21, lsr #0", [0xf6,0x03,0x55,0xeb]
    assert_bytes "negs     x24, x23, lsl #31", [0xf8,0x7f,0x17,0xeb]
    assert_bytes "negs     x26, x25, lsl #29", [0xfa,0x77,0x19,0xeb]
    assert_bytes "negs     x28, x27", [0xfc,0x03,0x1b,0xeb]
    assert_bytes "negs     x29, x30", [0xfd,0x03,0x1e,0xeb]
    assert_bytes "negs     x30, xzr", [0xfe,0x03,0x1f,0xeb]
  end

  def test_generated_ngc
    assert_bytes "ngc      w23, wzr", [0xf7,0x03,0x1f,0x5a]
    assert_bytes "ngc      w3, w12", [0xe3,0x03,0x0c,0x5a]
    assert_bytes "ngc      w9, w10", [0xe9,0x03,0x0a,0x5a]
    assert_bytes "ngc      wzr, w9", [0xff,0x03,0x09,0x5a]
    assert_bytes "ngc      x0, xzr", [0xe0,0x03,0x1f,0xda]
    assert_bytes "ngc      x29, x30", [0xfd,0x03,0x1e,0xda]
    assert_bytes "ngc      x9, x10", [0xe9,0x03,0x0a,0xda]
    assert_bytes "ngc      xzr, x0", [0xff,0x03,0x00,0xda]
  end

  def test_generated_ngcs
    assert_bytes "ngcs     w23, wzr", [0xf7,0x03,0x1f,0x7a]
    assert_bytes "ngcs     w3, w12", [0xe3,0x03,0x0c,0x7a]
    assert_bytes "ngcs     w9, w10", [0xe9,0x03,0x0a,0x7a]
    assert_bytes "ngcs     wzr, w9", [0xff,0x03,0x09,0x7a]
    assert_bytes "ngcs     x0, xzr", [0xe0,0x03,0x1f,0xfa]
    assert_bytes "ngcs     x29, x30", [0xfd,0x03,0x1e,0xfa]
    assert_bytes "ngcs     x9, x10", [0xe9,0x03,0x0a,0xfa]
    assert_bytes "ngcs     xzr, x0", [0xff,0x03,0x00,0xfa]
  end

  def test_generated_nop
    assert_bytes "nop", [0x1f,0x20,0x03,0xd5]
  end

  def test_generated_orn
    assert_bytes "orn      w2, w5, w29", [0xa2,0x00,0x3d,0x2a]
    assert_bytes "orn      x3, x5, x7, asr #0", [0xa3,0x00,0xa7,0xaa]
  end

  def test_generated_orr
    assert_bytes "orr  w0, w1, #0xfffffffd", [0x20,0x78,0x1e,0x32]
    assert_bytes "orr      sp, x10, #0xe00000ffe00000ff", [0x5f,0x29,0x03,0xb2]
    assert_bytes "orr      w2, w7, w0, asr #31", [0xe2,0x7c,0x80,0x2a]
    assert_bytes "orr      w3, w9, #0xffff0000", [0x23,0x3d,0x10,0x32]
    assert_bytes "orr      w9, w10, #0x3ff", [0x49,0x25,0x00,0x32]
    assert_bytes "orr      wsp, w10, #0xe00000ff", [0x5f,0x29,0x03,0x32]
    assert_bytes "orr      x11, x12, #0x8000000000000fff", [0x8b,0x31,0x41,0xb2]
    assert_bytes "orr      x3, x9, #0xffff0000ffff0000", [0x23,0x3d,0x10,0xb2]
    assert_bytes "orr      x8, x9, x10, lsl #12", [0x28,0x31,0x0a,0xaa]
    assert_bytes "orr      x9, x10, #0x3ff000003ff", [0x49,0x25,0x00,0xb2]
  end

  def test_generated_prfm
    assert_bytes "prfm     pldl1keep, [x26, w6, uxtw]", [0x40,0x4b,0xa6,0xf8]
    assert_bytes "prfm    pldl1keep, [sp, #8]", [0xe0,0x07,0x80,0xf9]
    assert_bytes "prfm    pldl2keep, [x5, #16]", [0xa2,0x08,0x80,0xf9]
    assert_bytes "prfm    plil1keep, [sp, #8]", [0xe8,0x07,0x80,0xf9]
    assert_bytes "prfm    plil2keep, [x5, #16]", [0xaa,0x08,0x80,0xf9]
    assert_bytes "prfm    pstl1keep, [sp, #8]", [0xf0,0x07,0x80,0xf9]
    assert_bytes "prfm    pstl2keep, [x5, #16]", [0xb2,0x08,0x80,0xf9]
  end

  def test_generated_prfum
    assert_bytes "prfum    pldl2keep, [sp, #-256]", [0xe2,0x03,0x90,0xf8]
  end

  def test_generated_pssbb
    assert_bytes "pssbb", [0x9f,0x34,0x03,0xd5]
  end

  def test_generated_rbit
    assert_bytes "rbit  w0, w7", [0xe0,0x00,0xc0,0x5a]
    assert_bytes "rbit  x18, x3", [0x72,0x00,0xc0,0xda]
  end

  def test_generated_ret
    assert_bytes "ret", [0xc0,0x03,0x5f,0xd6]
    assert_bytes "ret      x10", [0x40,0x01,0x5f,0xd6]
  end

  def test_generated_rev
    assert_bytes "rev  w18, w0", [0x12,0x08,0xc0,0x5a]
    assert_bytes "rev  w7, wzr", [0xe7,0x0b,0xc0,0x5a]
    assert_bytes "rev  x13, x12", [0x8d,0x0d,0xc0,0xda]
    assert_bytes "rev  x18, xzr", [0xf2,0x0f,0xc0,0xda]
    assert_bytes "rev  x22, x2", [0x56,0x0c,0xc0,0xda]
    assert_bytes "rev  x22, xzr", [0xf6,0x0f,0xc0,0xda]
  end

  def test_generated_rev16
    assert_bytes "rev16  x5, x2", [0x45,0x04,0xc0,0xda]
    assert_bytes "rev16 w17, w1", [0x31,0x04,0xc0,0x5a]
  end

  def test_generated_rev32
    assert_bytes "rev32  x20, x1", [0x34,0x08,0xc0,0xda]
    assert_bytes "rev32  x20, xzr", [0xf4,0x0b,0xc0,0xda]
  end

  def test_generated_ror
    assert_bytes "ror  w0, w1, w2", [0x20,0x2c,0xc2,0x1a]
    assert_bytes "ror  w24, w25, w26", [0x38,0x2f,0xda,0x1a]
    assert_bytes "ror  x27, x28, x29", [0x9b,0x2f,0xdd,0x9a]
    assert_bytes "ror     w9, w13, #31", [0xa9,0x7d,0x8d,0x13]
    assert_bytes "ror     x19, x23, #24", [0xf3,0x62,0xd7,0x93]
    assert_bytes "ror     x29, xzr, #63", [0xfd,0xff,0xdf,0x93]
    assert_bytes "ror  x3, x4, x5", [0x83,0x2c,0xc5,0x9a]
  end

  def test_generated_sbc
    assert_bytes "sbc      w20, w0, wzr", [0x14,0x00,0x1f,0x5a]
    assert_bytes "sbc      w29, w27, w25", [0x7d,0x03,0x19,0x5a]
    assert_bytes "sbc      wzr, w3, w4", [0x7f,0x00,0x04,0x5a]
    assert_bytes "sbc      x20, x0, xzr", [0x14,0x00,0x1f,0xda]
    assert_bytes "sbc      x29, x27, x25", [0x7d,0x03,0x19,0xda]
    assert_bytes "sbc      xzr, x3, x4", [0x7f,0x00,0x04,0xda]
  end

  def test_generated_sbcs
    assert_bytes "sbcs     w20, w0, wzr", [0x14,0x00,0x1f,0x7a]
    assert_bytes "sbcs     w29, w27, w25", [0x7d,0x03,0x19,0x7a]
    assert_bytes "sbcs     wzr, w3, w4", [0x7f,0x00,0x04,0x7a]
    assert_bytes "sbcs     x20, x0, xzr", [0x14,0x00,0x1f,0xfa]
    assert_bytes "sbcs     x29, x27, x25", [0x7d,0x03,0x19,0xfa]
    assert_bytes "sbcs     xzr, x3, x4", [0x7f,0x00,0x04,0xfa]
  end

  def test_generated_sbfiz
    assert_bytes "sbfiz    w11, w12, #31, #1", [0x8b,0x01,0x01,0x13]
    assert_bytes "sbfiz    w13, w14, #29, #3", [0xcd,0x09,0x03,0x13]
    assert_bytes "sbfiz    x2, x3, #63, #1", [0x62,0x00,0x41,0x93]
    assert_bytes "sbfiz    x9, x10, #5, #59", [0x49,0xe9,0x7b,0x93]
    assert_bytes "sbfiz    xzr, xzr, #10, #11", [0xff,0x2b,0x76,0x93]
  end

  def test_generated_sbfx
    assert_bytes "sbfx     w12, w9, #0, #1", [0x2c,0x01,0x00,0x13]
    assert_bytes "sbfx     w9, w10, #0, #1", [0x49,0x01,0x00,0x13]
    assert_bytes "sbfx     x1, x2, #3, #2", [0x41,0x10,0x43,0x93]
    assert_bytes "sbfx     xzr, xzr, #10, #11", [0xff,0x53,0x4a,0x93]
  end

  def test_generated_sdiv
    assert_bytes "sdiv  w12, w21, w0", [0xac,0x0e,0xc0,0x1a]
    assert_bytes "sdiv  x13, x2, x1", [0x4d,0x0c,0xc1,0x9a]
  end

  def test_generated_sev
    assert_bytes "sev", [0x9f,0x20,0x03,0xd5]
  end

  def test_generated_sevl
    assert_bytes "sevl", [0xbf,0x20,0x03,0xd5]
  end

  def test_generated_smaddl
    assert_bytes "smaddl   x13, wzr, w14, x15", [0xed,0x3f,0x2e,0x9b]
    assert_bytes "smaddl   x16, w17, wzr, x18", [0x30,0x4a,0x3f,0x9b]
    assert_bytes "smaddl   x3, w5, w2, x9", [0xa3,0x24,0x22,0x9b]
    assert_bytes "smaddl   xzr, w10, w11, x12", [0x5f,0x31,0x2b,0x9b]
  end

  def test_generated_smnegl
    assert_bytes "smnegl   x11, w13, w17", [0xab,0xfd,0x31,0x9b]
    assert_bytes "smnegl   x19, w20, w21", [0x93,0xfe,0x35,0x9b]
  end

  def test_generated_smsubl
    assert_bytes "smsubl   x13, wzr, w14, x15", [0xed,0xbf,0x2e,0x9b]
    assert_bytes "smsubl   x16, w17, wzr, x18", [0x30,0xca,0x3f,0x9b]
    assert_bytes "smsubl   x3, w5, w2, x9", [0xa3,0xa4,0x22,0x9b]
    assert_bytes "smsubl   xzr, w10, w11, x12", [0x5f,0xb1,0x2b,0x9b]
  end

  def test_generated_smulh
    assert_bytes "smulh    x23, x22, xzr", [0xd7,0x7e,0x5f,0x9b]
    assert_bytes "smulh    x25, xzr, x24", [0xf9,0x7f,0x58,0x9b]
    assert_bytes "smulh    x30, x29, x28", [0xbe,0x7f,0x5c,0x9b]
    assert_bytes "smulh    xzr, x27, x26", [0x7f,0x7f,0x5a,0x9b]
  end

  def test_generated_smull
    assert_bytes "smull    x11, w13, w17", [0xab,0x7d,0x31,0x9b]
    assert_bytes "smull    x19, w20, w21", [0x93,0x7e,0x35,0x9b]
  end

  def test_generated_ssbb
    assert_bytes "ssbb", [0x9f,0x30,0x03,0xd5]
  end

  def test_generated_stlr
    assert_bytes "stlr     wzr, [x1]", [0x3f,0xfc,0x9f,0x88]
    assert_bytes "stlr     x30, [x2]", [0x5e,0xfc,0x9f,0xc8]
  end

  def test_generated_stlrb
    assert_bytes "stlrb    w27, [sp]", [0xfb,0xff,0x9f,0x08]
  end

  def test_generated_stlrh
    assert_bytes "stlrh    w28, [x0]", [0x1c,0xfc,0x9f,0x48]
  end

  def test_generated_stlxp
    assert_bytes "stlxp    w25, x26, x27, [sp]", [0xfa,0xef,0x39,0xc8]
    assert_bytes "stlxp    wzr, w22, w23, [x24]", [0x16,0xdf,0x3f,0x88]
  end

  def test_generated_stlxr
    assert_bytes "stlxr    w18, x19, [x20]", [0x93,0xfe,0x12,0xc8]
    assert_bytes "stlxr    wzr, w17, [sp]", [0xf1,0xff,0x1f,0x88]
  end

  def test_generated_stlxrb
    assert_bytes "stlxrb   w14, w15, [x16]", [0x0f,0xfe,0x0e,0x08]
  end

  def test_generated_stlxrh
    assert_bytes "stlxrh   w15, w16, [x17]", [0x30,0xfe,0x0f,0x48]
  end

  def test_generated_stnp
    assert_bytes "stnp      wzr, w9, [sp, #252]", [0xff,0xa7,0x1f,0x28]
  end

  def test_generated_stp
    assert_bytes "stp      wzr, w9, [sp, #252]", [0xff,0xa7,0x1f,0x29]
    assert_bytes "stp      wzr, w9, [sp, #252]!", [0xff,0xa7,0x9f,0x29]
    assert_bytes "stp      wzr, w9, [sp], #252", [0xff,0xa7,0x9f,0x28]
  end

  def test_generated_str
    assert_bytes "str      w13, [x27, x5, sxtx #2]", [0x6d,0xfb,0x25,0xb8]
    assert_bytes "str      w14, [x26, w6, uxtw]", [0x4e,0x4b,0x26,0xb8]
    assert_bytes "str      w19, [sp, #255]!", [0xf3,0xff,0x0f,0xb8]
    assert_bytes "str      w19, [sp], #255", [0xf3,0xf7,0x0f,0xb8]
    assert_bytes "str      w20, [x30, #1]!", [0xd4,0x1f,0x00,0xb8]
    assert_bytes "str      w20, [x30], #1", [0xd4,0x17,0x00,0xb8]
    assert_bytes "str      w20, [x4, #16380]", [0x94,0xfc,0x3f,0xb9]
    assert_bytes "str      w21, [x12, #-256]!", [0x95,0x0d,0x10,0xb8]
    assert_bytes "str      w21, [x12], #-256", [0x95,0x05,0x10,0xb8]
    assert_bytes "str      x11, [x29, x3, sxtx]", [0xab,0xeb,0x23,0xf8]
    assert_bytes "str      x19, [x12, #-256]!", [0x93,0x0d,0x10,0xf8]
    assert_bytes "str      x19, [x12], #-256", [0x93,0x05,0x10,0xf8]
    assert_bytes "str      x2, [x3, #1]!", [0x62,0x1c,0x00,0xf8]
    assert_bytes "str      x2, [x3], #1", [0x62,0x14,0x00,0xf8]
    assert_bytes "str      x30, [sp]", [0xfe,0x03,0x00,0xf9]
    assert_bytes "str      x9, [x27, x6]", [0x69,0x6b,0x26,0xf8]
    assert_bytes "str      xzr, [x9, #255]!", [0x3f,0xfd,0x0f,0xf8]
    assert_bytes "str      xzr, [x9], #255", [0x3f,0xf5,0x0f,0xf8]
  end

  def test_generated_strb
    assert_bytes "strb     w10, [x3, #-256]!", [0x6a,0x0c,0x10,0x38]
    assert_bytes "strb     w10, [x3, #1]!", [0x6a,0x1c,0x00,0x38]
    assert_bytes "strb     w10, [x3], #-256", [0x6a,0x04,0x10,0x38]
    assert_bytes "strb     w10, [x3], #1", [0x6a,0x14,0x00,0x38]
    assert_bytes "strb     w12, [x28, xzr, sxtx #0]", [0x8c,0xfb,0x3f,0x38]
    assert_bytes "strb     w23, [x3, #4095]", [0x77,0xfc,0x3f,0x39]
    assert_bytes "strb     w9, [x2, #255]!", [0x49,0xfc,0x0f,0x38]
    assert_bytes "strb     w9, [x2], #255", [0x49,0xf4,0x0f,0x38]
    assert_bytes "strb     wzr, [x2]", [0x5f,0x00,0x00,0x39]
  end

  def test_generated_strh
    assert_bytes "strh     w10, [x3, #-256]!", [0x6a,0x0c,0x10,0x78]
    assert_bytes "strh     w10, [x3], #-256", [0x6a,0x04,0x10,0x78]
    assert_bytes "strh     w11, [x29, x3, sxtx]", [0xab,0xeb,0x23,0x78]
    assert_bytes "strh     w17, [sp, #8190]", [0xf1,0xff,0x3f,0x79]
    assert_bytes "strh     w19, [x21, wzr, sxtw #1]", [0xb3,0xda,0x3f,0x78]
    assert_bytes "strh     w20, [x10, #14]", [0x54,0x1d,0x00,0x79]
    assert_bytes "strh     w9, [x2, #1]!", [0x49,0x1c,0x00,0x78]
    assert_bytes "strh     w9, [x2, #255]!", [0x49,0xfc,0x0f,0x78]
    assert_bytes "strh     w9, [x2], #1", [0x49,0x14,0x00,0x78]
    assert_bytes "strh     w9, [x2], #255", [0x49,0xf4,0x0f,0x78]
  end

  def test_generated_sttr
    assert_bytes "sttr     w16, [x0, #-256]", [0x10,0x08,0x10,0xb8]
    assert_bytes "sttr     x28, [x14, #1]", [0xdc,0x19,0x00,0xf8]
  end

  def test_generated_sttrb
    assert_bytes "sttrb    w9, [sp]", [0xe9,0x0b,0x00,0x38]
  end

  def test_generated_sttrh
    assert_bytes "sttrh    wzr, [x12, #255]", [0x9f,0xf9,0x0f,0x78]
  end

  def test_generated_stur
    assert_bytes "stur     w16, [x0, #-256]", [0x10,0x00,0x10,0xb8]
    assert_bytes "stur     x28, [x14, #1]", [0xdc,0x11,0x00,0xf8]
  end

  def test_generated_sturb
    assert_bytes "sturb    w9, [sp]", [0xe9,0x03,0x00,0x38]
  end

  def test_generated_sturh
    assert_bytes "sturh    wzr, [x12, #255]", [0x9f,0xf1,0x0f,0x78]
  end

  def test_generated_stxp
    assert_bytes "stxp     w11, w12, w13, [x14]", [0xcc,0x35,0x2b,0x88]
    assert_bytes "stxp     wzr, x23, x14, [x15]", [0xf7,0x39,0x3f,0xc8]
  end

  def test_generated_stxr
    assert_bytes "stxr     w5, x6, [x7]", [0xe6,0x7c,0x05,0xc8]
    assert_bytes "stxr     wzr, w4, [sp]", [0xe4,0x7f,0x1f,0x88]
  end

  def test_generated_stxrb
    assert_bytes "stxrb    w1, w2, [x3]", [0x62,0x7c,0x01,0x08]
  end

  def test_generated_stxrh
    assert_bytes "stxrh    w2, w3, [x4]", [0x83,0x7c,0x02,0x48]
  end

  def test_generated_sub
    assert_bytes "sub      sp, sp, #288", [0xff,0x83,0x04,0xd1]
    assert_bytes "sub      sp, x2, w3, sxtw", [0x5f,0xc0,0x23,0xcb]
    assert_bytes "sub      sp, x3, x7, lsl #4", [0x7f,0x70,0x27,0xcb]
    assert_bytes "sub      w0, wsp, #4077", [0xe0,0xb7,0x3f,0x51]
    assert_bytes "sub      w11, w13, w15", [0xab,0x01,0x0f,0x4b]
    assert_bytes "sub      w17, w29, w20, lsl #31", [0xb1,0x7f,0x14,0x4b]
    assert_bytes "sub      w19, w17, w1, uxtx", [0x33,0x62,0x21,0x4b]
    assert_bytes "sub      w2, w3, w4, asr #0", [0x62,0x00,0x84,0x4b]
    assert_bytes "sub      w2, w3, w5, sxtx", [0x62,0xe0,0x25,0x4b]
    assert_bytes "sub      w2, w5, w1, sxtb", [0xa2,0x80,0x21,0x4b]
    assert_bytes "sub      w2, w5, w7, uxtb", [0xa2,0x00,0x27,0x4b]
    assert_bytes "sub      w21, w15, w17, uxth", [0xf5,0x21,0x31,0x4b]
    assert_bytes "sub      w21, w22, w23, lsr #0", [0xd5,0x02,0x57,0x4b]
    assert_bytes "sub      w24, w25, w26, lsr #18", [0x38,0x4b,0x5a,0x4b]
    assert_bytes "sub      w26, wsp, w19, sxth", [0xfa,0xa3,0x33,0x4b]
    assert_bytes "sub      w27, w28, w29, lsr #31", [0x9b,0x7f,0x5d,0x4b]
    assert_bytes "sub      w3, w5, w7", [0xa3,0x00,0x07,0x4b]
    assert_bytes "sub      w30, w29, wzr, uxtw", [0xbe,0x43,0x3f,0x4b]
    assert_bytes "sub      w4, w20, #546, lsl #12", [0x84,0x8a,0x48,0x51]
    assert_bytes "sub      w4, w6, wzr", [0xc4,0x00,0x1f,0x4b]
    assert_bytes "sub      w5, w6, w7, asr #21", [0xc5,0x54,0x87,0x4b]
    assert_bytes "sub      w8, w9, w10, asr #31", [0x28,0x7d,0x8a,0x4b]
    assert_bytes "sub      w9, w3, wzr, lsl #10", [0x69,0x28,0x1f,0x4b]
    assert_bytes "sub      wsp, w19, #16", [0x7f,0x42,0x00,0x51]
    assert_bytes "sub      wsp, w2, w3, sxtw", [0x5f,0xc0,0x23,0x4b]
    assert_bytes "sub      wzr, w3, w5", [0x7f,0x00,0x05,0x4b]
    assert_bytes "sub      x11, x13, x15", [0xab,0x01,0x0f,0xcb]
    assert_bytes "sub      x12, x1, w20, uxtw", [0x2c,0x40,0x34,0xcb]
    assert_bytes "sub      x17, x25, w20, sxtb", [0x31,0x83,0x34,0xcb]
    assert_bytes "sub      x17, x29, x20, lsl #63", [0xb1,0xff,0x14,0xcb]
    assert_bytes "sub      x18, x13, w19, sxth", [0xb2,0xa1,0x33,0xcb]
    assert_bytes "sub      x2, x3, x4, asr #0", [0x62,0x00,0x84,0xcb]
    assert_bytes "sub      x2, x4, w5, uxtb #2", [0x82,0x08,0x25,0xcb]
    assert_bytes "sub      x20, sp, w19, uxth #4", [0xf4,0x33,0x33,0xcb]
    assert_bytes "sub      x20, x3, x13, uxtx", [0x74,0x60,0x2d,0xcb]
    assert_bytes "sub      x21, x22, x23, lsr #0", [0xd5,0x02,0x57,0xcb]
    assert_bytes "sub      x24, x25, x26, lsr #18", [0x38,0x4b,0x5a,0xcb]
    assert_bytes "sub      x27, x28, x29, lsr #63", [0x9b,0xff,0x5d,0xcb]
    assert_bytes "sub      x3, x5, x7", [0xa3,0x00,0x07,0xcb]
    assert_bytes "sub      x3, x5, x9, sxtx", [0xa3,0xe0,0x29,0xcb]
    assert_bytes "sub      x4, x6, xzr", [0xc4,0x00,0x1f,0xcb]
    assert_bytes "sub      x5, x6, x7, asr #21", [0xc5,0x54,0x87,0xcb]
    assert_bytes "sub      x8, x9, x10, asr #63", [0x28,0xfd,0x8a,0xcb]
    assert_bytes "sub      x9, x3, xzr, lsl #10", [0x69,0x28,0x1f,0xcb]
    assert_bytes "sub      xzr, x3, x5", [0x7f,0x00,0x05,0xcb]
  end

  def test_generated_subs
    assert_bytes "subs     w11, w13, w15", [0xab,0x01,0x0f,0x6b]
    assert_bytes "subs     w17, w29, w20, lsl #31", [0xb1,0x7f,0x14,0x6b]
    assert_bytes "subs     w19, w17, w1, uxtx", [0x33,0x62,0x21,0x6b]
    assert_bytes "subs     w2, w3, w4, asr #0", [0x62,0x00,0x84,0x6b]
    assert_bytes "subs     w2, w3, w5, sxtx", [0x62,0xe0,0x25,0x6b]
    assert_bytes "subs     w2, w5, w1, sxtb #1", [0xa2,0x84,0x21,0x6b]
    assert_bytes "subs     w2, w5, w7, uxtb", [0xa2,0x00,0x27,0x6b]
    assert_bytes "subs     w21, w15, w17, uxth", [0xf5,0x21,0x31,0x6b]
    assert_bytes "subs     w21, w22, w23, lsr #0", [0xd5,0x02,0x57,0x6b]
    assert_bytes "subs     w24, w25, w26, lsr #18", [0x38,0x4b,0x5a,0x6b]
    assert_bytes "subs     w26, wsp, w19, sxth", [0xfa,0xa3,0x33,0x6b]
    assert_bytes "subs     w27, w28, w29, lsr #31", [0x9b,0x7f,0x5d,0x6b]
    assert_bytes "subs     w3, w5, w7", [0xa3,0x00,0x07,0x6b]
    assert_bytes "subs     w30, w29, wzr, uxtw", [0xbe,0x43,0x3f,0x6b]
    assert_bytes "subs     w4, w6, wzr", [0xc4,0x00,0x1f,0x6b]
    assert_bytes "subs     w5, w6, w7, asr #21", [0xc5,0x54,0x87,0x6b]
    assert_bytes "subs     w8, w9, w10, asr #31", [0x28,0x7d,0x8a,0x6b]
    assert_bytes "subs     w9, w3, wzr, lsl #10", [0x69,0x28,0x1f,0x6b]
    assert_bytes "subs     x11, x13, x15", [0xab,0x01,0x0f,0xeb]
    assert_bytes "subs     x12, x1, w20, uxtw", [0x2c,0x40,0x34,0xeb]
    assert_bytes "subs     x17, x29, x20, lsl #63", [0xb1,0xff,0x14,0xeb]
    assert_bytes "subs     x18, sp, w19, sxth", [0xf2,0xa3,0x33,0xeb]
    assert_bytes "subs     x2, x3, x4, asr #0", [0x62,0x00,0x84,0xeb]
    assert_bytes "subs     x2, x4, w5, uxtb #2", [0x82,0x08,0x25,0xeb]
    assert_bytes "subs     x20, sp, w19, uxth #4", [0xf4,0x33,0x33,0xeb]
    assert_bytes "subs     x20, x3, x13, uxtx", [0x74,0x60,0x2d,0xeb]
    assert_bytes "subs     x21, x22, x23, lsr #0", [0xd5,0x02,0x57,0xeb]
    assert_bytes "subs     x24, x25, x26, lsr #18", [0x38,0x4b,0x5a,0xeb]
    assert_bytes "subs     x27, x28, x29, lsr #63", [0x9b,0xff,0x5d,0xeb]
    assert_bytes "subs     x3, sp, x9, lsl #2", [0xe3,0x6b,0x29,0xeb]
    assert_bytes "subs     x3, x5, x7", [0xa3,0x00,0x07,0xeb]
    assert_bytes "subs     x3, x5, x9, sxtx #2", [0xa3,0xe8,0x29,0xeb]
    assert_bytes "subs     x4, sp, #3822", [0xe4,0xbb,0x3b,0xf1]
    assert_bytes "subs     x4, x6, xzr", [0xc4,0x00,0x1f,0xeb]
    assert_bytes "subs     x5, x6, x7, asr #21", [0xc5,0x54,0x87,0xeb]
    assert_bytes "subs     x8, x9, x10, asr #63", [0x28,0xfd,0x8a,0xeb]
    assert_bytes "subs     x9, x3, xzr, lsl #10", [0x69,0x28,0x1f,0xeb]
  end

  def test_generated_svc
    assert_bytes "svc      #0", [0x01,0x00,0x00,0xd4]
  end

  def test_generated_sxtb
    assert_bytes "sxtb     w1, w2", [0x41,0x1c,0x00,0x13]
    assert_bytes "sxtb     xzr, w3", [0x7f,0x1c,0x40,0x93]
  end

  def test_generated_sxth
    assert_bytes "sxth     w9, w10", [0x49,0x3d,0x00,0x13]
    assert_bytes "sxth     x0, w1", [0x20,0x3c,0x40,0x93]
  end

  def test_generated_sxtw
    assert_bytes "sxtw     x3, w30", [0xc3,0x7f,0x40,0x93]
  end

  def test_generated_sys
    assert_bytes "sys     #0, c15, c15, #2", [0x5f,0xff,0x08,0xd5]
    assert_bytes "sys     #7, c5, c9, #7, x5", [0xe5,0x59,0x0f,0xd5]
  end

  def test_generated_sysl
    assert_bytes "sysl    x1, #0, c15, c15, #2", [0x41,0xff,0x28,0xd5]
    assert_bytes "sysl    x9, #7, c5, c9, #7", [0xe9,0x59,0x2f,0xd5]
  end

  def test_generated_tlbi
    assert_bytes "tlbi    alle1", [0x9f,0x87,0x0c,0xd5]
    assert_bytes "tlbi    alle1is", [0x9f,0x83,0x0c,0xd5]
    assert_bytes "tlbi    alle2", [0x1f,0x87,0x0c,0xd5]
    assert_bytes "tlbi    alle2is", [0x1f,0x83,0x0c,0xd5]
    assert_bytes "tlbi    alle3", [0x1f,0x87,0x0e,0xd5]
    assert_bytes "tlbi    alle3is", [0x1f,0x83,0x0e,0xd5]
    assert_bytes "tlbi    aside1, x20", [0x54,0x87,0x08,0xd5]
    assert_bytes "tlbi    aside1is, x5", [0x45,0x83,0x08,0xd5]
    assert_bytes "tlbi    ipas2e1, x15", [0x2f,0x84,0x0c,0xd5]
    assert_bytes "tlbi    ipas2e1is, x4", [0x24,0x80,0x0c,0xd5]
    assert_bytes "tlbi    ipas2le1, x16", [0xb0,0x84,0x0c,0xd5]
    assert_bytes "tlbi    ipas2le1is, x9", [0xa9,0x80,0x0c,0xd5]
    assert_bytes "tlbi    vaae1, x21", [0x75,0x87,0x08,0xd5]
    assert_bytes "tlbi    vaae1is, x9", [0x69,0x83,0x08,0xd5]
    assert_bytes "tlbi    vaale1, x25", [0xf9,0x87,0x08,0xd5]
    assert_bytes "tlbi    vaale1is, x14", [0xee,0x83,0x08,0xd5]
    assert_bytes "tlbi    vae1, x17", [0x31,0x87,0x08,0xd5]
    assert_bytes "tlbi    vae1is, x1", [0x21,0x83,0x08,0xd5]
    assert_bytes "tlbi    vae2, x18", [0x32,0x87,0x0c,0xd5]
    assert_bytes "tlbi    vae2is, x2", [0x22,0x83,0x0c,0xd5]
    assert_bytes "tlbi    vae3, x19", [0x33,0x87,0x0e,0xd5]
    assert_bytes "tlbi    vae3is, x3", [0x23,0x83,0x0e,0xd5]
    assert_bytes "tlbi    vale1, x22", [0xb6,0x87,0x08,0xd5]
    assert_bytes "tlbi    vale1is, x10", [0xaa,0x83,0x08,0xd5]
    assert_bytes "tlbi    vale2, x23", [0xb7,0x87,0x0c,0xd5]
    assert_bytes "tlbi    vale2is, x11", [0xab,0x83,0x0c,0xd5]
    assert_bytes "tlbi    vale3, x24", [0xb8,0x87,0x0e,0xd5]
    assert_bytes "tlbi    vale3is, x13", [0xad,0x83,0x0e,0xd5]
    assert_bytes "tlbi    vmalle1", [0x1f,0x87,0x08,0xd5]
    assert_bytes "tlbi    vmalle1is", [0x1f,0x83,0x08,0xd5]
    assert_bytes "tlbi    vmalls12e1", [0xdf,0x87,0x0c,0xd5]
    assert_bytes "tlbi    vmalls12e1is", [0xdf,0x83,0x0c,0xd5]
  end

  def test_generated_tst
    assert_bytes "tst      w3, w7, lsl #31", [0x7f,0x7c,0x07,0x6a]
    assert_bytes "tst      x2, x20, asr #0", [0x5f,0x00,0x94,0xea]
  end

  def test_generated_ubfiz
    assert_bytes "ubfiz    x4, x5, #52, #11", [0xa4,0x28,0x4c,0xd3]
    assert_bytes "ubfiz    x4, xzr, #1, #6", [0xe4,0x17,0x7f,0xd3]
    assert_bytes "ubfiz    xzr, xzr, #10, #11", [0xff,0x2b,0x76,0xd3]
  end

  def test_generated_ubfx
    assert_bytes "ubfx     w9, w10, #0, #1", [0x49,0x01,0x00,0x53]
    assert_bytes "ubfx     xzr, x4, #0, #1", [0x9f,0x00,0x40,0xd3]
    assert_bytes "ubfx    w9, w10, #0, #1", [0x49,0x01,0x00,0x53]
    assert_bytes "ubfx    xzr, xzr, #10, #11", [0xff,0x53,0x4a,0xd3]
  end

  def test_generated_udiv
    assert_bytes "udiv  w0, w7, w10", [0xe0,0x08,0xca,0x1a]
    assert_bytes "udiv  x9, x22, x4", [0xc9,0x0a,0xc4,0x9a]
  end

  def test_generated_umaddl
    assert_bytes "umaddl   x13, wzr, w14, x15", [0xed,0x3f,0xae,0x9b]
    assert_bytes "umaddl   x16, w17, wzr, x18", [0x30,0x4a,0xbf,0x9b]
    assert_bytes "umaddl   x3, w5, w2, x9", [0xa3,0x24,0xa2,0x9b]
    assert_bytes "umaddl   xzr, w10, w11, x12", [0x5f,0x31,0xab,0x9b]
  end

  def test_generated_umnegl
    assert_bytes "umnegl   x11, w13, w17", [0xab,0xfd,0xb1,0x9b]
    assert_bytes "umnegl   x19, w20, w21", [0x93,0xfe,0xb5,0x9b]
  end

  def test_generated_umsubl
    assert_bytes "umsubl   x13, wzr, w14, x15", [0xed,0xbf,0xae,0x9b]
    assert_bytes "umsubl   x16, w17, wzr, x18", [0x30,0xca,0xbf,0x9b]
    assert_bytes "umsubl   x3, w5, w2, x9", [0xa3,0xa4,0xa2,0x9b]
    assert_bytes "umsubl   xzr, w10, w11, x12", [0x5f,0xb1,0xab,0x9b]
  end

  def test_generated_umulh
    assert_bytes "umulh    x23, x22, xzr", [0xd7,0x7e,0xdf,0x9b]
    assert_bytes "umulh    x25, xzr, x24", [0xf9,0x7f,0xd8,0x9b]
    assert_bytes "umulh    x30, x29, x28", [0xbe,0x7f,0xdc,0x9b]
    assert_bytes "umulh    xzr, x27, x26", [0x7f,0x7f,0xda,0x9b]
  end

  def test_generated_umull
    assert_bytes "umull    x11, w13, w17", [0xab,0x7d,0xb1,0x9b]
    assert_bytes "umull    x19, w20, w21", [0x93,0x7e,0xb5,0x9b]
  end

  def test_generated_uxtb
    assert_bytes "uxtb     w1, w2", [0x41,0x1c,0x00,0x53]
  end

  def test_generated_uxth
    assert_bytes "uxth     w9, w10", [0x49,0x3d,0x00,0x53]
  end

  def test_generated_wfe
    assert_bytes "wfe", [0x5f,0x20,0x03,0xd5]
  end

  def test_generated_wfi
    assert_bytes "wfi", [0x7f,0x20,0x03,0xd5]
  end

  def test_generated_yield
    assert_bytes "yield", [0x3f,0x20,0x03,0xd5]
  end

end
