// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_BROWSER_CODE_CACHE_GENERATED_CODE_CACHE_CONTEXT_H_
#define CONTENT_BROWSER_CODE_CACHE_GENERATED_CODE_CACHE_CONTEXT_H_

#include "base/files/file_path.h"
#include "base/memory/ref_counted.h"
#include "base/task/sequenced_task_runner.h"
#include "base/thread_annotations.h"
#include "content/common/content_export.h"
#include "content/public/browser/browser_thread.h"

namespace content {

class GeneratedCodeCache;

// One instance exists per disk-backed (non in-memory) storage contexts. This
// owns the instance of GeneratedCodeCache that is used to store the data
// generated by the renderer (for ex: code caches for script resources). This
// initializes and closes the code cache on the code cache thread. The
// instance of this class (|this|) itself is constructed on the UI thread.
class CONTENT_EXPORT GeneratedCodeCacheContext
    : public base::RefCountedThreadSafe<GeneratedCodeCacheContext> {
 public:
  REQUIRE_ADOPTION_FOR_REFCOUNTED_TYPE();

  // Runs a task on the code cache thread, or immediately if already on the code
  // cache thread.
  static void RunOrPostTask(scoped_refptr<GeneratedCodeCacheContext> context,
                            const base::Location& location,
                            base::OnceClosure task);

  // Gets the task runner for the code cache thread.
  static scoped_refptr<base::SequencedTaskRunner> GetTaskRunner(
      scoped_refptr<GeneratedCodeCacheContext> context);

  GeneratedCodeCacheContext();

  GeneratedCodeCacheContext(const GeneratedCodeCacheContext&) = delete;
  GeneratedCodeCacheContext& operator=(const GeneratedCodeCacheContext&) =
      delete;

  // Initialize is called on the UI thread when the StoragePartition is
  // being setup.
  void Initialize(const base::FilePath& path, int max_bytes);

  void Shutdown();

  // Call on the code cache thread to get the code cache instances.
  GeneratedCodeCache* generated_js_code_cache() const;
  GeneratedCodeCache* generated_wasm_code_cache() const;
  GeneratedCodeCache* generated_webui_js_code_cache() const;

 private:
  friend class base::RefCountedThreadSafe<GeneratedCodeCacheContext>;
  ~GeneratedCodeCacheContext();

  void InitializeOnThread(const base::FilePath& path, int max_bytes);
  void ShutdownOnThread();

  // Created, used and deleted on the code cache thread.
  std::unique_ptr<GeneratedCodeCache, base::OnTaskRunnerDeleter>
      generated_js_code_cache_ GUARDED_BY_CONTEXT(sequence_checker_) = {
          nullptr, base::OnTaskRunnerDeleter(nullptr)};
  std::unique_ptr<GeneratedCodeCache, base::OnTaskRunnerDeleter>
      generated_wasm_code_cache_ GUARDED_BY_CONTEXT(sequence_checker_) = {
          nullptr, base::OnTaskRunnerDeleter(nullptr)};
  std::unique_ptr<GeneratedCodeCache, base::OnTaskRunnerDeleter>
      generated_webui_js_code_cache_ GUARDED_BY_CONTEXT(sequence_checker_) = {
          nullptr, base::OnTaskRunnerDeleter(nullptr)};
  scoped_refptr<base::SequencedTaskRunner> task_runner_;

  SEQUENCE_CHECKER(sequence_checker_);
};

}  // namespace content

#endif  // CONTENT_BROWSER_CODE_CACHE_GENERATED_CODE_CACHE_CONTEXT_H_
