// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_BROWSER_ATTRIBUTION_REPORTING_ATTRIBUTION_RESOLVER_IMPL_H_
#define CONTENT_BROWSER_ATTRIBUTION_REPORTING_ATTRIBUTION_RESOLVER_IMPL_H_

#include <stdint.h>

#include <memory>
#include <optional>
#include <string>
#include <variant>
#include <vector>

#include "base/sequence_checker.h"
#include "base/thread_annotations.h"
#include "base/time/time.h"
#include "base/types/expected.h"
#include "content/browser/attribution_reporting/aggregatable_result.mojom-forward.h"
#include "content/browser/attribution_reporting/attribution_report.h"
#include "content/browser/attribution_reporting/attribution_resolver.h"
#include "content/browser/attribution_reporting/attribution_storage_sql.h"
#include "content/browser/attribution_reporting/create_report_result.h"
#include "content/browser/attribution_reporting/rate_limit_result.h"
#include "content/browser/attribution_reporting/stored_source.h"
#include "content/common/content_export.h"
#include "content/public/browser/attribution_data_model.h"
#include "content/public/browser/storage_partition.h"

namespace base {
class FilePath;
}  // namespace base

namespace content {

class AttributionResolverDelegate;
class AttributionTrigger;
class StorableSource;

struct AttributionInfo;

// This class may be constructed on any sequence but must be accessed and
// destroyed on the same sequence. The sequence must outlive |this|.
class CONTENT_EXPORT AttributionResolverImpl : public AttributionResolver {
 public:
  AttributionResolverImpl(
      const base::FilePath& user_data_directory,
      std::unique_ptr<AttributionResolverDelegate> delegate);
  AttributionResolverImpl(const AttributionResolverImpl&) = delete;
  AttributionResolverImpl& operator=(const AttributionResolverImpl&) = delete;
  AttributionResolverImpl(AttributionResolverImpl&&) = delete;
  AttributionResolverImpl& operator=(AttributionResolverImpl&&) = delete;
  ~AttributionResolverImpl() override;

 private:
  // AttributionResolver:
  StoreSourceResult StoreSource(StorableSource source) override;
  CreateReportResult MaybeCreateAndStoreReport(
      AttributionTrigger trigger) override;
  std::vector<AttributionReport> GetAttributionReportsWithLimit(
      base::Time max_report_time,
      int limit) override;
  std::vector<AttributionReport> GetAttributionReports(
      base::Time max_report_time) override;
  std::optional<base::Time> GetNextReportTime(base::Time time) override;
  std::optional<AttributionReport> GetReport(AttributionReport::Id) override;
  std::vector<StoredSource> GetActiveSourcesWithLimit(int limit) override;
  std::vector<StoredSource> GetActiveSources() override;
  std::set<AttributionDataModel::DataKey> GetAllDataKeys() override;
  void DeleteByDataKey(const AttributionDataModel::DataKey& datakey) override;
  bool DeleteReport(AttributionReport::Id report_id) override;
  bool UpdateReportForSendFailure(AttributionReport::Id report_id,
                                  base::Time new_report_time) override;
  std::optional<base::Time> AdjustOfflineReportTimes() override;
  std::optional<base::Time> AdjustNavigationRetryReportTimes() override;
  void ClearData(base::Time delete_begin,
                 base::Time delete_end,
                 StoragePartition::StorageKeyMatcherFunction filter,
                 bool delete_rate_limit_data) override;
  void ClearDataIncludingRateLimit(
      base::Time delete_begin,
      base::Time delete_end,
      StoragePartition::StorageKeyMatcherFunction filter) override;
  ProcessAggregatableDebugReportResult ProcessAggregatableDebugReport(
      AggregatableDebugReport,
      std::optional<int> remaining_budget,
      std::optional<StoredSource::Id>) override;
  void StoreOsRegistrations(const base::flat_set<url::Origin>&) override;
  void SetDelegate(std::unique_ptr<AttributionResolverDelegate>) override;

  CreateReportResult::EventLevel MaybeCreateEventLevelReport(
      const AttributionInfo& attribution_info,
      const StoredSource& source,
      const AttributionTrigger& trigger,
      std::optional<uint64_t>& dedup_key)
      VALID_CONTEXT_REQUIRED(sequence_checker_);

  CreateReportResult::Aggregatable MaybeCreateAggregatableAttributionReport(
      const AttributionInfo& attribution_info,
      const StoredSource& source,
      const AttributionTrigger& trigger,
      std::optional<uint64_t>& dedup_keys)
      VALID_CONTEXT_REQUIRED(sequence_checker_);

  // Generates null aggregatable reports for the given trigger and stores all
  // those reports.
  [[nodiscard]] bool GenerateNullAggregatableReportsAndStoreReports(
      const AttributionTrigger&,
      const AttributionInfo&,
      const StoredSource* source,
      AttributionReport* new_aggregatable_report,
      std::optional<base::Time>& min_null_aggregatable_report_time)
      VALID_CONTEXT_REQUIRED(sequence_checker_);

  base::Time GetAggregatableReportTime(const AttributionTrigger& trigger,
                                       base::Time trigger_time) const
      VALID_CONTEXT_REQUIRED(sequence_checker_);

  struct ReplaceReportError {};
  struct AddNewReport {};
  struct DropNewReport {
    bool source_deactivated;
  };
  struct ReplaceOldReport {
    AttributionReport replaced_report;
  };

  using ReplaceReportResult = std::variant<ReplaceReportError,
                                           AddNewReport,
                                           DropNewReport,
                                           ReplaceOldReport>;

  [[nodiscard]] ReplaceReportResult MaybeReplaceLowerPriorityEventLevelReport(
      const AttributionReport& report,
      const StoredSource& source,
      int num_attributions) VALID_CONTEXT_REQUIRED(sequence_checker_);

  CreateReportResult::EventLevel MaybeStoreEventLevelReport(
      const StoredSource& source,
      std::optional<uint64_t> dedup_key,
      int num_attributions,
      CreateReportResult::EventLevelSuccess)
      VALID_CONTEXT_REQUIRED(sequence_checker_);

  std::unique_ptr<AttributionResolverDelegate> delegate_
      GUARDED_BY_CONTEXT(sequence_checker_);

  AttributionStorageSql storage_ GUARDED_BY_CONTEXT(sequence_checker_);

  // Time at which `DeleteExpiredSources()` was last called. Initialized to
  // the NULL time.
  base::Time last_deleted_expired_sources_
      GUARDED_BY_CONTEXT(sequence_checker_);

  SEQUENCE_CHECKER(sequence_checker_);
};

}  // namespace content

#endif  // CONTENT_BROWSER_ATTRIBUTION_REPORTING_ATTRIBUTION_RESOLVER_IMPL_H_
