/*
 * // Copyright (c) Radzivon Bartoshyk 8/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
Poly numerators for digamma with step 0.25 from 0 to 1: [-0.99999999,-0.75], [-0.75, -0.5] etc.
Generated by Wolfram:
<<FunctionApproximations`
ClearAll["Global`*"]
f[x_]:=PolyGamma[x+1]
{err0,approx,err1}=MiniMaxApproximation[f[z],{z,{-0.99999999,-0.75},11,11},WorkingPrecision->75,MaxIterations->100]
num=Numerator[approx];
den=Denominator[approx];
poly=num;
coeffs=CoefficientList[poly,z];
TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
poly=den;
coeffs=CoefficientList[poly,z];
TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
*/
static DIGAMMA_P: [[(u64, u64); 12]; 5] = [
    [
        (0x3c56cb8ed2e8f89d, 0xbfe2788cfc6fb619),
        (0xbc3f4a887143e1a0, 0x3face2294e40a23d),
        (0x3c66cbeeb663e02b, 0x3ff93da24c048557),
        (0xbc99dce69443c3a1, 0x3ffb9ca4e7e3b448),
        (0x3c86642ccc010b7c, 0x3febdebcbbc26740),
        (0xbc6b0b2f44dae75c, 0x3fcfec9b12222c41),
        (0xbbf4498c8a1b0940, 0x3fa62080e59486a8),
        (0x3c0b9dc350a33198, 0x3f72bc74facca682),
        (0xbbcd99184e051c10, 0x3f32c94b7b2fb002),
        (0xbb8738e809ec0cee, 0x3ee48ad4af286d1f),
        (0x3b130f69e7502da5, 0x3e84788fd8c782e7),
        (0xbaabf9ce6f6a39c0, 0x3e075df82b55c605),
    ],
    [
        (0x3c56cb907008b543, 0xbfe2788cfc6fb619),
        (0xbc100842dc7f9e7f, 0x3fb01301a3e98b73),
        (0xbc823b1b4a0eb7aa, 0x3ff938b063dae04f),
        (0xbc9ad21ad2ab96c5, 0x3ffb55c29a135ea7),
        (0xbc7d5ba993767236, 0x3feb4f810b04c753),
        (0x3c64772eb3244da3, 0x3fcee4afd0031184),
        (0x3c31c1810f303099, 0x3fa513f613ed0729),
        (0x3c0bd0845bf7f471, 0x3f717f05f5d10c47),
        (0xbbd5725001e2660d, 0x3f311a7e461bfdbc),
        (0xbb8598e1d6bf14c5, 0x3ee21971cbf5769f),
        (0x3b2c987b53229a3f, 0x3e81461991174455),
        (0xbaa40e6fc1c3cdbd, 0x3e029575e1cf2d2e),
    ],
    [
        (0x3c56cb90701fbfa6, 0xbfe2788cfc6fb619),
        (0xbc5fccfe13017bdd, 0x3fb1afc092258787),
        (0xbc5e1697dc2cc4ab, 0x3ff9337b6bb944de),
        (0x3c7ec613451b31b9, 0x3ffb0fd93c22113b),
        (0x3c69294dd477ad47, 0x3feac3c7b64adea4),
        (0xbc69ac8bff5565b5, 0x3fcde66e558d148f),
        (0x3c41acf6837637f5, 0x3fa4154b581a555a),
        (0xbc000c1483343db8, 0x3f70580604407f51),
        (0x3bbc673a5ee2174f, 0x3f2f290c93113075),
        (0x3b7ea2e3314e056e, 0x3edfed2db5039158),
        (0xbaf8d7013fff21f7, 0x3e7d35fdf2716965),
        (0xba739bf5ff69075f, 0x3dfdb070ae4d9275),
    ],
    [
        (0x3c56cb90701fbfab, 0xbfe2788cfc6fb619),
        (0x3c32d7bac835354c, 0x3fb34c4468e4cca5),
        (0x3c83dfd7ab2663ee, 0x3ff92df4d81668e8),
        (0x3c99f27ceb6a6462, 0x3ffaca12fb44387d),
        (0xbc873d42ae8bf0ee, 0x3fea39db2aa6755c),
        (0x3c6f0709b773a576, 0x3fccee948fe7d9a3),
        (0xbc319c78a343f2ba, 0x3fa320f0d611b675),
        (0xbba5feb85ab5acbc, 0x3f6e855445cb1dae),
        (0x3bba9b56af9b20c5, 0x3f2c5ecc73cbcdd8),
        (0xbb79a32258828ace, 0x3edc257fc6ffd42d),
        (0x3b1efad7acd7d7f1, 0x3e78b3710a7cc9b4),
        (0xba9b941fb3ff6b0c, 0x3df7c1965f2d7194),
    ],
    [
        (0x3c56cb90701fbfab, 0xbfe2788cfc6fb619),
        (0xbc0f99d5dc5f8b06, 0x3fb4e913ab76f9ab),
        (0x3c3191e63046dc0e, 0x3ff9281aa8398c82),
        (0xbc8dca8def824c6e, 0x3ffa845a59ce1115),
        (0xbc84dbb9117bcea5, 0x3fe9b18e7fe7829a),
        (0xbc5e38525daac915, 0x3fcbfcbc6601a1fe),
        (0x3c4df8dcb4769fe3, 0x3fa2364b5ab136dd),
        (0x3c03a220e74edede, 0x3f6c7b78bf3817d2),
        (0xbb831144f7e90d73, 0x3f29d06f3db25586),
        (0x3b7b4bf8f2ec756c, 0x3ed8cd0278bee187),
        (0x3b0c2bf25bbf39b5, 0x3e74e2f6a044ec34),
        (0x3a942086a4928038, 0x3df3096d2199706d),
    ],
];

static DIGAMMA_Q: [[(u64, u64); 12]; 5] = [
    [
        (0x0000000000000000, 0x3ff0000000000000),
        (0xbca6adb0ceca2bb2, 0x4006042dff16f40e),
        (0xbc85b67e7a730640, 0x4008379d3bb3e8e0),
        (0xbc9f004eb95ee438, 0x3ffc7e68e904512d),
        (0xbc86ee1922dc4c80, 0x3fe407ea97357260),
        (0xbc4a51002b1b8870, 0x3fc1a70e08dee4dd),
        (0x3c3c858930786915, 0x3f93c1e4eef4b20b),
        (0x3be9a73c7d428d88, 0x3f5baa2f04f498b1),
        (0x3bbad1b8117f7756, 0x3f171c5667c8dffc),
        (0x3b173ce742b4d895, 0x3ec4e29a45581499),
        (0xbadceb3243cdae8f, 0x3e60a2d5d1c5f8e4),
        (0xba72b2562de7b14b, 0x3ddb303e7518d74b),
    ],
    [
        (0x0000000000000000, 0x3ff0000000000000),
        (0x3c91779e812448f8, 0x4005ed8db27be60e),
        (0x3ca08f641dca5f12, 0x4007fb6b0dcbcebe),
        (0xbc9b5b059968427b, 0x3ffc005c8adea86d),
        (0x3c7a6c4119840b60, 0x3fe37d5ff5b97171),
        (0xbc530ac1661d05de, 0x3fc0f56578da17ee),
        (0xbc0418ed2bc636c5, 0x3f92ad4e7387f851),
        (0xbbc1f659b88045fd, 0x3f599f72920bc12d),
        (0x3bba548ec376cb95, 0x3f14dae97a930698),
        (0x3b440fec06d88466, 0x3ec239fbc98052c7),
        (0xbae8bd7909c83aaf, 0x3e5bcab555b96b0c),
        (0x3a7da912eade2739, 0x3dd5684408d505b7),
    ],
    [
        (0x0000000000000000, 0x3ff0000000000000),
        (0xbca91be54288a0e0, 0x4005d735308bb4dd),
        (0xbcad30b7bbf2b783, 0x4007c03f8e4c4fe6),
        (0xbc9eeb30365815fb, 0x3ffb854f0db08baa),
        (0x3c782d2becaf6172, 0x3fe2f7622383d14d),
        (0xbc5e010ee89a3662, 0x3fc04badf9bdbed9),
        (0x3c2af3eb8a7d2bde, 0x3f91a94aee0050b0),
        (0xbbd2450502e41cf9, 0x3f57be27fec5860c),
        (0xbb4d401f4b8a5f96, 0x3f12d5723c46d383),
        (0xbb56834644f0201f, 0x3ebfda938f459b86),
        (0x3ada0645f44a4bcd, 0x3e574563cf974833),
        (0xba7147ae6edcbc3d, 0x3dd0f0bfcad5f047),
    ],
    [
        (0x0000000000000000, 0x3ff0000000000000),
        (0xbc9045b43a44ffbd, 0x4005c0dfe199018b),
        (0x3cadc1c9c8ce4d30, 0x40078563dd87aa1b),
        (0xbc7ee1018a57916d, 0x3ffb0bbe462659fc),
        (0x3c84b1e3cb3c10f4, 0x3fe2743f18457656),
        (0x3c5ce0b8927bff8a, 0x3fbf4f3f5d1d2c88),
        (0x3c227cc874c2781a, 0x3f90b1f8cc772eb7),
        (0x3bea9648da0df3bd, 0x3f55fdcf409bac26),
        (0x3bbc08910e142370, 0x3f11003e3494b94d),
        (0x3b5a6e0762b57f19, 0x3ebbd3ea01684109),
        (0xbae425ecbc8c7993, 0x3e537e51126d11b7),
        (0xba656f95223fc44f, 0x3dcadec5fd0f622c),
    ],
    [
        (0x0000000000000000, 0x3ff0000000000000),
        (0x3caa0a86221a0755, 0x4005aa867d56dca4),
        (0xbcac770719607c75, 0x40074ac4f6ca5a47),
        (0x3c90d03a36128d7f, 0x3ffa93815f190569),
        (0x3c4d48dbdc6c9a7c, 0x3fe1f3c3b4044619),
        (0x3c55f2ca2ba72f08, 0x3fbe11c76c6789f6),
        (0x3c07ca0eacaa9a9c, 0x3f8f8d146c709524),
        (0xbbd3ae3ef7bb9792, 0x3f545be615cbebf2),
        (0xbb86e82ec7f13b7f, 0x3f0ead31c1a5b093),
        (0xbb3275f4a13e5259, 0x3eb84d337c38d867),
        (0xbaf557dfebeee67e, 0x3e505544e8f1eb85),
        (0xba664f9845fcddf2, 0x3dc559cc156a72a5),
    ],
];

#[inline]
#[allow(clippy::type_complexity)]
pub(crate) fn digamma_coeefs(x: f64) -> (&'static [(u64, u64); 12], &'static [(u64, u64); 12]) {
    if x < 1.25 {
        const FOUR: u64 = 2 << 52;
        let idx = f64::from_bits(x.to_bits().wrapping_add(FOUR)) as usize;
        let p = &DIGAMMA_P[idx];
        let q = &DIGAMMA_Q[idx];
        (p, q)
    } else if x <= 1.461632144 {
        // Polynomial generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx,err1}=MiniMaxApproximation[f[z],{z,{0.25,0.4616321445},11,11},WorkingPrecision->75,MaxIterations->100]
        // num=Numerator[approx];
        // poly=den;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static P: [(u64, u64); 12] = [
            (0x3c56cb90701fbfac, 0xbfe2788cfc6fb619),
            (0x3c519fc5fe329992, 0x3fb66b6f7ca77b95),
            (0x3c934d7e8434b86d, 0x3ff92255627f42b7),
            (0x3c637005ae2e1665, 0x3ffa43326e6dda36),
            (0x3c3cadafce6428f4, 0x3fe9338d9b2539f1),
            (0xbc5934a30d152b54, 0x3fcb1febac693a97),
            (0xbc3d4ec7808fa04f, 0x3fa1636286a89dfe),
            (0xbbf577e9f832bec3, 0x3f6aafbc9ad0d5ad),
            (0xbbcd6cc934d6ba4e, 0x3f279eaf8168ee05),
            (0xbb774ef4099bad0c, 0x3ed605efaec77016),
            (0xbb13d911fcebafac, 0x3e71dae45db651bb),
            (0x3a8f943200c1fd92, 0x3deefe67af5b7c69),
        ];

        // Polynomial generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx,err1}=MiniMaxApproximation[f[z],{z,{0.25,0.4616321445},11,11},WorkingPrecision->75,MaxIterations->100]
        // den=Denominator[approx];
        // poly=den;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static Q: [(u64, u64); 12] = [
            (0x0000000000000000, 0x3ff0000000000000),
            (0x3ca0fb55209ad44d, 0x4005959bb2d38810),
            (0xbc4a7dccb43a2a97, 0x40071428acb23be5),
            (0xbc8b9d316e4d2eac, 0x3ffa24400e09040c),
            (0xbc82963c4efb8cd1, 0x3fe17dfa742912bc),
            (0x3c57e765f5582745, 0x3fbcf24965915c1f),
            (0xbc2bfe1e01bc34f0, 0x3f8de96be6a73955),
            (0x3bff6a6cc1679110, 0x3f52ef034cdc300f),
            (0xbb95e707835497ae, 0x3f0bd94abae4b483),
            (0xbb435b31a77f1eac, 0x3eb5673711efb89a),
            (0xbab7879ef73e896a, 0x3e4bb1a59a808117),
            (0xba50ff72e575b8bd, 0x3dc13fef9cd36cd7),
        ];
        (&P, &Q)
    } else if x <= 1.75 {
        // Polynomial generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx,err1}=MiniMaxApproximation[f[z],{z,{0.4616321445,0.75},11,11},WorkingPrecision->75,MaxIterations->100]
        // num=Numerator[approx];
        // den=Denominator[approx];
        // poly=num;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static P: [(u64, u64); 12] = [
            (0x3c56cb907020a112, 0xbfe2788cfc6fb619),
            (0xbc3cc4d6d2ddccf8, 0x3fb8010bfcb83141),
            (0x3c953d47923ee59b, 0x3ff91bf87a8009ef),
            (0x3c856cf1c819abcd, 0x3ff9fee67ab017a1),
            (0xbc7e5e24a77dd53e, 0x3fe8b0e3f74e8969),
            (0xbc60cf6e08c2898c, 0x3fca3dc2ac288997),
            (0xbc4d6745fb4dd03b, 0x3fa08eb928fceb5d),
            (0x3c0153615a07e895, 0x3f68e95d5669dc9c),
            (0x3bcc104646590a9e, 0x3f25818c7facb551),
            (0xbb6ce130dd405fa2, 0x3ed36f27c8982809),
            (0xbae5a7d3777ea48d, 0x3e6e4b3d604133f4),
            (0xba86b002676d830d, 0x3de9027ae9347c77),
        ];

        // Generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx,err1}=MiniMaxApproximation[f[z],{z,{0.4616321445,0.75},11,11},WorkingPrecision->75,MaxIterations->100]
        // den=Denominator[approx];
        // poly=den;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static Q: [(u64, u64); 12] = [
            (0x0000000000000000, 0x3ff0000000000000),
            (0x3ca13038b3e9b2fe, 0x40057fa6128d0d08),
            (0xbcacf21951ec09a9, 0x4006db1750ade888),
            (0x3c9fd268ab672291, 0x3ff9b0c5a589896f),
            (0xbc4b818d4493b88a, 0x3fe104d8aeed391e),
            (0xbc5531e244a9d256, 0x3fbbce38cef1dcd3),
            (0x3c14e017cdbf9c7b, 0x3f8c45ceb3fbe0a3),
            (0x3bff3d4bd6748d1c, 0x3f51898acc32fa08),
            (0xbb6c0ebdb9714f4b, 0x3f09266424ad60b1),
            (0x3b30abd2021dc5ea, 0x3eb2ba49a890927f),
            (0xba9629f3ce206d76, 0x3e474b9f374cf052),
            (0x39f3beca069e2142, 0x3dbba0c4466d2dcd),
        ];
        (&P, &Q)
    } else if x <= 2. {
        // Polynomial generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{0.75,1},11,11},WorkingPrecision->75,MaxIterations->100]
        // num=Numerator[approx][[1]];
        // poly=num;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static P: [(u64, u64); 12] = [
            (0x3c56cb907bb52809, 0xbfe2788cfc6fb619),
            (0xbc420cb940ff1bad, 0x3fb9b8ce99af6686),
            (0xbc9554bebef623c7, 0x3ff914b779852ffb),
            (0x3c81daceacffe906, 0x3ff9b4fb845f87b4),
            (0xbc589d02262e231e, 0x3fe8251c504ee4e7),
            (0xbc48b053e0ac1f18, 0x3fc94f0c5b394803),
            (0xbc12b44611d0cc3d, 0x3f9f6411022b8fee),
            (0x3bf71626c3a722d6, 0x3f671bef7af39602),
            (0x3bca802abbe6c032, 0x3f236afc88364acd),
            (0xbb73ba0bffc04b11, 0x3ed0f71a4569fdd2),
            (0x3ae6866974b3745b, 0x3e695b19eb1a11d0),
            (0x3a53fcd00ba0f0cd, 0x3de3dbee8f81e7d1),
        ];
        // Generated by Wolfram Mathematica:
        //<<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{0.75,1},11,11},WorkingPrecision->75,MaxIterations->100]
        // den=Denominator[approx][[1]];
        // poly=den;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static Q: [(u64, u64); 12] = [
            (0x0000000000000000, 0x3ff0000000000000),
            (0x3ca79b06fde08bf7, 0x400567d727c212e8),
            (0xbc9b0efe8b146dee, 0x40069d86c47f0240),
            (0xbc8299a0687d2bc4, 0x3ff9351a45c9504d),
            (0xbc7146fcb08e6211, 0x3fe0846a56e8c617),
            (0xbc482c27fd2b0de3, 0x3fba9ca29930f546),
            (0x3beaea4e7ed3058f, 0x3f8a96406d35271a),
            (0x3bf776396bc4adbd, 0x3f5021eb0152afad),
            (0xbb96d179aedf8a7a, 0x3f0682935c1f4e53),
            (0xbb469533a5f262d4, 0x3eb033bbf3243fc1),
            (0x3ad33e062bbd7ac2, 0x3e43535be5d463ef),
            (0x3a5a7d127f1db626, 0x3db5c4427025e959),
        ];
        (&P, &Q)
    } else if x <= 3. {
        // x <= 3
        // Polynomial generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{1,2},11,11},WorkingPrecision->75,MaxIterations->100]
        // num=Numerator[approx][[1]];
        // poly=num;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static P: [(u64, u64); 12] = [
            (0x3c56d1cc8bcd2834, 0xbfe2788cfc6fb619),
            (0x3c50701f78e64a63, 0x3fbd8bb4f2220f15),
            (0xbc6f91b861a7c3f1, 0x3ff903416798b207),
            (0x3c8c53da657116b3, 0x3ff910f28e38ea3f),
            (0xbc8720408446640c, 0x3fe6f4e66e069dae),
            (0xbc34e63b5e59b9bd, 0x3fc75316d7a07060),
            (0xbc22cfa33a314334, 0x3f9bd312bdf809fb),
            (0xbbfd6f2d0a941d18, 0x3f638321ec193179),
            (0xbbbd1cabd7d7d204, 0x3f1edf0eeec2ed8f),
            (0x3b62b3641fcc9055, 0x3ec908c824d24ebd),
            (0x3b09f21b5bfbaac3, 0x3e61119012a56c3c),
            (0xba7ccbf11ca33a44, 0x3dd7e364822d6038),
        ];
        // Generated by Wolfram Mathematica:
        //<<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{1,2},11,11},WorkingPrecision->75,MaxIterations->100]
        // den=Denominator[approx][[1]];
        // poly=den;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static Q: [(u64, u64); 12] = [
            (0x0000000000000000, 0x3ff0000000000000),
            (0x3ca8ae32e83e7971, 0x400532d7f57a1fb8),
            (0x3ca7e06fe036f034, 0x4006159f6c8559a6),
            (0x3c696aeb79408ada, 0x3ff8276de15b9949),
            (0x3c6b4b6c4100ee75, 0x3fdee20a3a9510d1),
            (0xbc5366197ca045c2, 0x3fb81bb8b46e3ac9),
            (0x3c2f178bb9cecf1c, 0x3f8726ac2011b61b),
            (0x3be51ce661a77196, 0x3f4abe5166a64e0d),
            (0xbb84cb13a26b099d, 0x3f018de8d20c9ae7),
            (0xbb4f127e563d4fce, 0x3ea772bc233a7f3d),
            (0xbad8391ca6522a23, 0x3e398817e8c8a3f5),
            (0x3a46dbfc8fc0a31f, 0x3da9c1eb2bb8a09d),
        ];
        (&P, &Q)
    } else if x <= 4. {
        // x <= 4
        // Polynomial generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{2,3},11,11},WorkingPrecision->75,MaxIterations->100]
        // num=Numerator[approx][[1]];
        // poly=num;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static P: [(u64, u64); 12] = [
            (0xbc1138539f5170ce, 0xbfe2788cfc6fb617),
            (0x3c5c28276512f306, 0x3fc1fc2364555134),
            (0xbc6a09be4de1611c, 0x3ff8e1befdb6bc25),
            (0x3c74ca0e29a4c28f, 0x3ff7ff202fdee96b),
            (0xbc81af890e1880a0, 0x3fe50bb10d8d3f83),
            (0xbc63ea18b51ff6b6, 0x3fc444e2fde41a82),
            (0x3c3778d316f51d1e, 0x3f969f9868302eb9),
            (0x3be358e50a96a230, 0x3f5d3a1baf51425c),
            (0x3bbc38a174d2733f, 0x3f14e8fe4a7dddfb),
            (0x3b5f5ed1876db6a7, 0x3ebe07d4dbcc1ad4),
            (0x3af907af35fa32d0, 0x3e51b3f54fae106f),
            (0x3a62af4ca3fd0f35, 0x3dc4d88e2dc21ca8),
        ];
        // Generated by Wolfram Mathematica:
        //<<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{2,3},11,11},WorkingPrecision->75,MaxIterations->100]
        // den=Denominator[approx][[1]];
        // poly=den;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static Q: [(u64, u64); 12] = [
            (0x0000000000000000, 0x3ff0000000000000),
            (0x3c9648605b16dd34, 0x4004d9ced1a59873),
            (0x3c7dc6b76d368927, 0x400534eafce36582),
            (0xbc3ce3eb41ed98ed, 0x3ff673eeb71c9f83),
            (0xbc66f13de6a48d20, 0x3fdb84d7f538c20d),
            (0x3c5035f4fefd130a, 0x3fb45b63620b1db4),
            (0x3c1923173b36285f, 0x3f8246e776c12954),
            (0x3be4c733ad174a92, 0x3f436ec81a059c76),
            (0x3b8012652ce0bc96, 0x3ef70fe1dc481482),
            (0x3b2d0670cd6004bc, 0x3e9b497296f5f46e),
            (0xba85189a8dab3039, 0x3e29bb6d630a41ac),
            (0x3a3cb2bee18cf77a, 0x3d95f146af7f3514),
        ];
        (&P, &Q)
    } else if x <= 5. {
        // x <= 5
        // Polynomial generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{3,4},11,11},WorkingPrecision->75,MaxIterations->100]
        // num=Numerator[approx][[1]];
        // poly=num;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static P: [(u64, u64); 12] = [
            (0x3c82777617606035, 0xbfe2788cfc6fb22a),
            (0x3c6a89168e92a2bd, 0x3fc5286b343ecbe5),
            (0xbc7562c3f1327b91, 0x3ff8bb7dfd30d653),
            (0xbc8167bf7208ef96, 0x3ff6f31495ad9560),
            (0x3c82d2177e694dd6, 0x3fe3431bda031b31),
            (0xbc6be5da70840171, 0x3fc19291a81dd8a2),
            (0x3c3bf690d21eb357, 0x3f9258218b3ef7bf),
            (0xbbffd8d4b0692fbb, 0x3f55d7a5623a2dcf),
            (0x3babb591eb87dd6c, 0x3f0c56ac18bd4d6b),
            (0xbb5c8b806eef4ae7, 0x3eb2209eb7acd22b),
            (0xbad36fc4d27a4168, 0x3e42af070578d594),
            (0xb9dfe0085849fd5a, 0x3db2d9b6d2913a2a),
        ];
        // Generated by Wolfram Mathematica:
        //<<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{3,4},11,11},WorkingPrecision->75,MaxIterations->100]
        // den=Denominator[approx][[1]];
        // poly=den;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static Q: [(u64, u64); 12] = [
            (0x0000000000000000, 0x3ff0000000000000),
            (0xbca9a57355580b15, 0x400481dafdd7bdb2),
            (0x3c8c54a5f9c29d95, 0x40045b68f0902217),
            (0xbc795347ec01fbd2, 0x3ff4daef6b912517),
            (0x3c699ae8f19cf3ec, 0x3fd87d05ac8eb769),
            (0x3c59ea5ce4904986, 0x3fb12877d62826ca),
            (0x3be6bd7425bdd967, 0x3f7cce33e2dd0530),
            (0xbbc460e353f4c8bc, 0x3f3c38bcf76c94b5),
            (0xbb712ad762f9ec9d, 0x3eee601d38ab6d9b),
            (0xbb23ce6868791dc5, 0x3e9006213ba059aa),
            (0xbab95abdc33a7321, 0x3e1a7ac297a0a53d),
            (0xba174f07c41d8cfd, 0x3d836fa7694a651d),
        ];
        (&P, &Q)
    } else if x <= 6. {
        // x <= 6
        // Polynomial generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{4,5},11,11},WorkingPrecision->75,MaxIterations->100]
        // num=Numerator[approx][[1]];
        // poly=num;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static P: [(u64, u64); 12] = [
            (0xbc8032a1d2c0ad13, 0xbfe2788cfc6e9dbe),
            (0x3c38696772d75917, 0x3fc849d273d05d75),
            (0x3c954ca48d09b1aa, 0x3ff890aef7dc27e1),
            (0xbc963b4b7c9a58af, 0x3ff5ed5797be10f8),
            (0x3c87390c0470f3c6, 0x3fe19a75e3a5dbe3),
            (0xbc4a8fca8acf4f1f, 0x3fbe68afab720ce8),
            (0x3c1ed20f546b9104, 0x3f8db0bf34ed5e72),
            (0xbbd6b67d3f6d936e, 0x3f504f9138773832),
            (0x3bae68dee81a333d, 0x3f033ff5436fb4e3),
            (0xbb1fad498922c758, 0x3ea61338f9837209),
            (0x3ad4cdce5341705f, 0x3e341863900b259b),
            (0xba1368ab344aa52f, 0x3da1a36dc66478ff),
        ];
        // Generated by Wolfram Mathematica:
        //<<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{4,5},11,11},WorkingPrecision->75,MaxIterations->100]
        // den=Denominator[approx][[1]];
        // poly=den;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static Q: [(u64, u64); 12] = [
            (0x0000000000000000, 0x3ff0000000000000),
            (0xbc8415c937927b80, 0x40042b14ac1de31a),
            (0xbca02e650cfc91cb, 0x4003893428b832f0),
            (0x3c85d3d357beac28, 0x3ff35bb913ab979e),
            (0x3c763ef88de8bbf4, 0x3fd5c47e7c28af74),
            (0x3c4624d0ce63afac, 0x3face2a04d07592f),
            (0x3befca3faad26d95, 0x3f76adbb8f0c9e71),
            (0x3bd6bb21845ab51f, 0x3f3482259071c038),
            (0xbb888fad8984adb0, 0x3ee4182fa08710cd),
            (0xbb2ec0c609c2589a, 0x3e83082e53b39cd4),
            (0xbaa2954cc8fe5ef2, 0x3e0bd8dfedf43944),
            (0x39e30166e0cac476, 0x3d71dc88357652c3),
        ];
        (&P, &Q)
    } else if x <= 7. {
        // x <= 7
        // Polynomial generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{5,6},11,11},WorkingPrecision->75,MaxIterations->100]
        // num=Numerator[approx][[1]];
        // poly=num;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static P: [(u64, u64); 12] = [
            (0xbc81dfd5a5534592, 0xbfe2788cfc565c6c),
            (0x3c5ee519bbeafb4a, 0x3fcb5e9b249cc2ae),
            (0x3c8df433a8ab926f, 0x3ff86195ebe61e55),
            (0xbc9c26e874cedb94, 0x3ff4eeb2d59b9e46),
            (0xbc8b629f64d30eb4, 0x3fe011547d3b4deb),
            (0x3c4877eafbf1d0bf, 0x3fba461689f2d09b),
            (0xbc02cb677bd729e7, 0x3f8801390c64edcd),
            (0x3be6bb411d2772bf, 0x3f4861722dbdb621),
            (0x3b95c312dfadf0f8, 0x3efa4589f0866235),
            (0x3b2e65576c0b0642, 0x3e9b2d2386424cdf),
            (0x3acbcd0ab8139043, 0x3e260e89319914dd),
            (0x3a2bde33fa1ac8f3, 0x3d9110236b17c522),
        ];
        // Generated by Wolfram Mathematica:
        //<<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{5,6},11,11},WorkingPrecision->75,MaxIterations->100]
        // den=Denominator[approx][[1]];
        // poly=den;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static Q: [(u64, u64); 12] = [
            (0x0000000000000000, 0x3ff0000000000000),
            (0x3c8ff6dd498c91b2, 0x4003d5ac26d888b6),
            (0x3c9acb90be620fba, 0x4002be9b5e78f13f),
            (0x3c80944df7c8a780, 0x3ff1f5e5ec0c3d71),
            (0x3c50f7c451bbdd2e, 0x3fd355e0f87c5218),
            (0xbc4cd8f8f78b44a1, 0x3fa84c8863aae54b),
            (0xbc13df2d02c4d3c4, 0x3f71dc00f5ff8e9c),
            (0x3bc942f7fa51682f, 0x3f2de1134f7b2de6),
            (0xbb5bcf5da39fb215, 0x3edac2e18af5b8d1),
            (0xbb11b483e6820b4e, 0x3e76e765628dffe0),
            (0x3a89bf8734fc1f0e, 0x3dfdf503240dffa2),
            (0xb9fcaf5858297b35, 0x3d610182e7d328a5),
        ];
        (&P, &Q)
    } else if x <= 8. {
        // x <= 8
        // Polynomial generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{6,7},11,11},WorkingPrecision->75,MaxIterations->100]
        // num=Numerator[approx][[1]];
        // poly=num;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static P: [(u64, u64); 12] = [
            (0xbc8162f86e7bad36, 0xbfe2788cfb513f92),
            (0xbc6ddc13d58b297e, 0x3fce652fd3cec73c),
            (0x3c701fd75f0e2153, 0x3ff82e7ee851190e),
            (0x3c86494ccd407812, 0x3ff3f7d8f2cc7221),
            (0x3c71d5a4a3016802, 0x3fdd4df807c7f6a2),
            (0xbc55bd847484e7bd, 0x3fb6aeaa992b7e5e),
            (0x3c1d99552c28a292, 0x3f8368db68048c4f),
            (0xbbecf51acbe60763, 0x3f42437beb7b56ec),
            (0x3b7ec2e94f896a2d, 0x3ef20a7a9ea0accc),
            (0x3a9a0a1b60fffd51, 0x3e90f02feda221d2),
            (0x3ab8a5da0c600246, 0x3e18b9818d3b86a1),
            (0xba1dc40c982b9669, 0x3d810bda1d0bde80),
        ];
        // Generated by Wolfram Mathematica:
        //<<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{6,7},11,11},WorkingPrecision->75,MaxIterations->100]
        // den=Denominator[approx][[1]];
        // poly=den;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static Q: [(u64, u64); 12] = [
            (0x0000000000000000, 0x3ff0000000000000),
            (0x3c95f4382d9f9dbc, 0x400381cd52a27389),
            (0x3ca36a2a63377659, 0x4001fbd9cbbdde5b),
            (0xbc83cb5796f6bba0, 0x3ff0a8db1d651315),
            (0x3c779f72905d08e8, 0x3fd12b8a9f02c16c),
            (0xbc4dacaf20322afe, 0x3fa47160278b25ee),
            (0xbc0262fab0a08a3c, 0x3f6c2a94e4e5cf4f),
            (0x3bc553f072d7f874, 0x3f25d9d7c995af31),
            (0x3b62f92e30dbacb9, 0x3ed1f75f5d275707),
            (0x3b02b661ecc2ba69, 0x3e6bf75615fe0e2f),
            (0x3a571b7290df94bc, 0x3df07d3db0951aef),
            (0xb9f18f3d33a27e32, 0x3d50bfea8f77e0c3),
        ];
        (&P, &Q)
    } else if x <= 9. {
        // x <= 9
        // Polynomial generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{7,8},11,11},WorkingPrecision->75,MaxIterations->100]
        // num=Numerator[approx][[1]];
        // poly=num;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static P: [(u64, u64); 12] = [
            (0x3c8efd7fe9db6e79, 0xbfe2788cf4b3267f),
            (0x3c6a2e8811472031, 0x3fd0ae258336bb2b),
            (0x3c7399e3844d1e5f, 0x3ff7f7b99c5c43be),
            (0x3c822ee42e5fe80d, 0x3ff30958a7e8d80e),
            (0x3c6848d741adf8c4, 0x3fdab4b23acdeb18),
            (0x3c5aa072e08707aa, 0x3fb393aa5016e08a),
            (0xbc18b1985dc2c38c, 0x3f7f6b771fa6e163),
            (0x3bdc86329c365200, 0x3f3b75a61ceee8c8),
            (0x3b8e8f8ff597b2e6, 0x3ee8f7bf1b272a81),
            (0x3b19b561962f8b4e, 0x3e8566c7b18e4a14),
            (0x3aadf125c1ad7adb, 0x3e0c4f51c4444e9a),
            (0x39f1e1df7a4c3a8b, 0x3d7190106017b110),
        ];
        // Generated by Wolfram Mathematica:
        //<<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{7,8},11,11},WorkingPrecision->75,MaxIterations->100]
        // den=Denominator[approx][[1]];
        // poly=den;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static Q: [(u64, u64); 12] = [
            (0x0000000000000000, 0x3ff0000000000000),
            (0x3ca684bdc1b74b83, 0x40032f9b62c83c86),
            (0x3ca5d8d28ceedbb4, 0x4001410ed7fd4814),
            (0x3c7686b33443b989, 0x3feee782a9ab5027),
            (0x3c53620c43b8821f, 0x3fce7f4d1d2ffe3c),
            (0xbc33e2abe9a40984, 0x3fa1362ff4c8e4cb),
            (0xbbef2639d9860de0, 0x3f66430cca12747e),
            (0xbbb8084e63a55a7b, 0x3f200faaa577da43),
            (0x3b6b0dea347bf9f3, 0x3ec859b423a711e8),
            (0x3b04cb8df32e9178, 0x3e61563b86bed1a2),
            (0xba801506c55d0eb4, 0x3de292d18eb8fd2f),
            (0xb9d06300f4b9226a, 0x3d4109bff966bb51),
        ];
        (&P, &Q)
    } else if x <= 10. {
        // x <= 10
        // Polynomial generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{8,9},11,11},WorkingPrecision->75,MaxIterations->100]
        // num=Numerator[approx][[1]];
        // poly=num;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static P: [(u64, u64); 12] = [
            (0xbc8593c1df8ef6c7, 0xbfe2788cd6e083bb),
            (0x3c6415b1331fd1e0, 0x3fd2217df51ae592),
            (0x3c9c29eeb5fe4731, 0x3ff7bd965beb1b87),
            (0xbc85fe7985564d6e, 0x3ff2239b9feaa8a7),
            (0xbc5cc9b9441046be, 0x3fd8541d3ac20fdc),
            (0xbc243a1d71c1b7e2, 0x3fb0e6b0237a2f28),
            (0xbc1534c74442766c, 0x3f797a05781bf99b),
            (0xbbd8ba51ad9881fe, 0x3f34bc78d4108de1),
            (0xbb81efd29f25726f, 0x3ee16d09b7ddc32f),
            (0xba96ec9993a947f2, 0x3e7b6c967312dff2),
            (0xbaa2a0b4acd633ea, 0x3e008d27d632d14c),
            (0xb9ff596e6d378d22, 0x3d62a21a99d77977),
        ];
        // Generated by Wolfram Mathematica:
        //<<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{8,9},11,11},WorkingPrecision->75,MaxIterations->100]
        // den=Denominator[approx][[1]];
        // poly=den;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static Q: [(u64, u64); 12] = [
            (0x0000000000000000, 0x3ff0000000000000),
            (0x3c6f1ef07b1b5dc9, 0x4002df30458e626a),
            (0xbc960127cb9e2eab, 0x40008e3f3217718d),
            (0x3c8412d2aabe8bf8, 0x3fecab1dc0e9bcac),
            (0x3c64edc0f7754efe, 0x3fcb18b1038e24f6),
            (0xbc3abc16284838a3, 0x3f9d05522b423e2f),
            (0xbc0e44998e55a65f, 0x3f61a64c0a74479e),
            (0xbbb190403d2a5a61, 0x3f17c0a8b1842890),
            (0x3b598fe19ed917ac, 0x3ec0ab7613d6b90e),
            (0xbaf7c33460b01c25, 0x3e55d581be9c2732),
            (0xba7278196b41c6f6, 0x3dd565843701949d),
            (0x39912fb123ad4ce0, 0x3d31de54105a6ae4),
        ];
        (&P, &Q)
    } else if x <= 11. {
        // x <= 11
        // Polynomial generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{9,10},11,11},WorkingPrecision->75,MaxIterations->100]
        // num=Numerator[approx][[1]];
        // poly=num;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static P: [(u64, u64); 12] = [
            (0xbc7002de428656f8, 0xbfe2788c706086ea),
            (0xbc7e5dea366e1224, 0x3fd38c4fa29daad3),
            (0x3c95297f4120d6e1, 0x3ff780642930e686),
            (0xbc7cdcec6276d962, 0x3ff146e905a85e7c),
            (0xbc4dde11481dc9d7, 0x3fd629105bd7a86b),
            (0x3c29f43835822d7e, 0x3fad342a0507d56a),
            (0x3c1f98d2402273b2, 0x3f74b55e06fd0100),
            (0x3bc61bf252df59de, 0x3f2f7b95f117d541),
            (0x3b4f7f3c66a58c10, 0x3ed88c81b5a7bffc),
            (0xbb1ba95a86f7955a, 0x3e71d2d46e31e04c),
            (0xb9feae47109db197, 0x3df3c1020acf06fb),
            (0xb9ed26ed15eff6d5, 0x3d5453226c1a47c8),
        ];
        // Generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{9,10},11,11},WorkingPrecision->75,MaxIterations->100]
        // den=Denominator[approx][[1]];
        // poly=den;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static Q: [(u64, u64); 12] = [
            (0x0000000000000000, 0x3ff0000000000000),
            (0xbc8f67d2f15a73c9, 0x4002909d472a6258),
            (0x3c94a8e3edcab93b, 0x3fffc6b132050914),
            (0xbc5b982f33a20a73, 0x3fea9a2aba70b06e),
            (0x3c619d024f5cdba4, 0x3fc817cd43625de8),
            (0xbc3dc40df7f1ecbb, 0x3f9882a708f4756f),
            (0x3b88f0ba3ff4834f, 0x3f5c16f3fb9fe3e9),
            (0xbb8ed7db88e71ba0, 0x3f11ae1da151fd8f),
            (0x3b3437f8aa73a246, 0x3eb711185a5568ff),
            (0x3ab876889e88ff72, 0x3e4bee89a4622cb4),
            (0xba6ac2419dbe0e2f, 0x3dc930ba3560eb64),
            (0xb9b2124aaf27b632, 0x3d234925cfe9d5d8),
        ];
        (&P, &Q)
    } else {
        // x <= 12
        // Polynomial generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{10,11},11,11},WorkingPrecision->75,MaxIterations->100]
        // num=Numerator[approx][[1]];
        // poly=num;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static P: [(u64, u64); 12] = [
            (0xbc88aa87989ef5a1, 0xbfe2788b51df7906),
            (0xbc7af1e1932b591c, 0x3fd4ee69b56504f9),
            (0x3c740a2c04d19bfc, 0x3ff7406f9cf46140),
            (0xbc917572eb9329ef, 0x3ff07369547283f5),
            (0x3c54e46eaf6429e3, 0x3fd4301a4c091a22),
            (0x3c3c3ba8aa15ca20, 0x3fa9426e448ee7db),
            (0xbc1f041777b9f035, 0x3f70e2032a465789),
            (0xbbb783cb47e19f83, 0x3f28099ad5f73d16),
            (0xbb6932c40b32e5e0, 0x3ed175058b1e67e6),
            (0x3b024c00ae6427d0, 0x3e677fd917aaa847),
            (0x3a50b5029ff4eadf, 0x3de80bf02bf0e221),
            (0x39e405e48a7d9326, 0x3d46c1a62ef66417),
        ];
        // Generated by Wolfram Mathematica:
        // <<FunctionApproximations`
        // ClearAll["Global`*"]
        // f[x_]:=PolyGamma[x+1]
        // {err0,approx}=MiniMaxApproximation[f[z],{z,{10,11},11,11},WorkingPrecision->75,MaxIterations->100]
        // den=Denominator[approx][[1]];
        // poly=den;
        // coeffs=CoefficientList[poly,z];
        // TableForm[Table[Row[{"'",NumberForm[coeffs[[i+1]],{50,50}, ExponentFunction->(Null&)],"',"}],{i,0,Length[coeffs]-1}]]
        static Q: [(u64, u64); 12] = [
            (0x0000000000000000, 0x3ff0000000000000),
            (0x3c67856052188c26, 0x400243ec3166f349),
            (0x3c2b34f0f808ddac, 0x3ffe806cb15a71b3),
            (0xbc823b5d5bc54e38, 0x3fe8b21e296d1320),
            (0xbc52f7423180111c, 0x3fc571b5d26833f9),
            (0xbc33e19defee24d6, 0x3f94bf1ad662d38a),
            (0x3beb180e8d55583a, 0x3f56723080074543),
            (0xbb7d192d116623ad, 0x3f0a818d001404be),
            (0x3b194b9889dea9d1, 0x3eb021f1a8b7f978),
            (0x3a8ca7cbf623a001, 0x3e4224e2c71fae88),
            (0xba4d8314f1a91e06, 0x3dbe48212e08d6f2),
            (0xb9a210925f26ebb8, 0x3d15627ecb7e02ac),
        ];
        (&P, &Q)
    }
}
