{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- The @GUri@ type and related functions can be used to parse URIs into
-- their components, and build valid URIs from individual components.
-- 
-- Since @GUri@ only represents absolute URIs, all @GUri@s will have a
-- URI scheme, so 'GI.GLib.Structs.Uri.uriGetScheme' will always return a non-@NULL@
-- answer. Likewise, by definition, all URIs have a path component, so
-- 'GI.GLib.Structs.Uri.uriGetPath' will always return a non-@NULL@ string (which may
-- be empty).
-- 
-- If the URI string has an
-- <https://tools.ietf.org/html/rfc3986#section-3 ‘authority’ component> (that
-- is, if the scheme is followed by @:\/\/@ rather than just @:@), then the
-- @GUri@ will contain a hostname, and possibly a port and ‘userinfo’.
-- Additionally, depending on how the @GUri@ was constructed\/parsed (for example,
-- using the @G_URI_FLAGS_HAS_PASSWORD@ and @G_URI_FLAGS_HAS_AUTH_PARAMS@ flags),
-- the userinfo may be split out into a username, password, and
-- additional authorization-related parameters.
-- 
-- Normally, the components of a @GUri@ will have all @%@-encoded
-- characters decoded. However, if you construct\/parse a @GUri@ with
-- @G_URI_FLAGS_ENCODED@, then the @%@-encoding will be preserved instead in
-- the userinfo, path, and query fields (and in the host field if also
-- created with @G_URI_FLAGS_NON_DNS@). In particular, this is necessary if
-- the URI may contain binary data or non-UTF-8 text, or if decoding
-- the components might change the interpretation of the URI.
-- 
-- For example, with the encoded flag:
-- 
-- 
-- === /c code/
-- >g_autoptr(GUri) uri = g_uri_parse ("http://host/path?query=http%3A%2F%2Fhost%2Fpath%3Fparam%3Dvalue", G_URI_FLAGS_ENCODED, &err);
-- >g_assert_cmpstr (g_uri_get_query (uri), ==, "query=http%3A%2F%2Fhost%2Fpath%3Fparam%3Dvalue");
-- 
-- 
-- While the default @%@-decoding behaviour would give:
-- 
-- 
-- === /c code/
-- >g_autoptr(GUri) uri = g_uri_parse ("http://host/path?query=http%3A%2F%2Fhost%2Fpath%3Fparam%3Dvalue", G_URI_FLAGS_NONE, &err);
-- >g_assert_cmpstr (g_uri_get_query (uri), ==, "query=http://host/path?param=value");
-- 
-- 
-- During decoding, if an invalid UTF-8 string is encountered, parsing will fail
-- with an error indicating the bad string location:
-- 
-- 
-- === /c code/
-- >g_autoptr(GUri) uri = g_uri_parse ("http://host/path?query=http%3A%2F%2Fhost%2Fpath%3Fbad%3D%00alue", G_URI_FLAGS_NONE, &err);
-- >g_assert_error (err, G_URI_ERROR, G_URI_ERROR_BAD_QUERY);
-- 
-- 
-- You should pass @G_URI_FLAGS_ENCODED@ or @G_URI_FLAGS_ENCODED_QUERY@ if you
-- need to handle that case manually. In particular, if the query string
-- contains @=@ characters that are @%@-encoded, you should let
-- 'GI.GLib.Structs.Uri.uriParseParams' do the decoding once of the query.
-- 
-- @GUri@ is immutable once constructed, and can safely be accessed from
-- multiple threads. Its reference counting is atomic.
-- 
-- Note that the scope of @GUri@ is to help manipulate URIs in various applications,
-- following <https://tools.ietf.org/html/rfc3986 RFC 3986>. In particular,
-- it doesn\'t intend to cover web browser needs, and doesn’t implement the
-- <https://url.spec.whatwg.org/ WHATWG URL> standard. No APIs are provided to
-- help prevent
-- <https://en.wikipedia.org/wiki/IDN_homograph_attack homograph attacks>, so
-- @GUri@ is not suitable for formatting URIs for display to the user for making
-- security-sensitive decisions.
-- 
-- == Relative and absolute URIs
-- 
-- As defined in <https://tools.ietf.org/html/rfc3986#section-4 RFC 3986>, the
-- hierarchical nature of URIs means that they can either be ‘relative
-- references’ (sometimes referred to as ‘relative URIs’) or ‘URIs’ (for
-- clarity, ‘URIs’ are referred to in this documentation as
-- ‘absolute URIs’ — although
-- <https://tools.ietf.org/html/rfc3986#section-4.3 in contrast to RFC 3986>,
-- fragment identifiers are always allowed).
-- 
-- Relative references have one or more components of the URI missing. In
-- particular, they have no scheme. Any other component, such as hostname,
-- query, etc. may be missing, apart from a path, which has to be specified (but
-- may be empty). The path may be relative, starting with @.\/@ rather than @\/@.
-- 
-- For example, a valid relative reference is @.\/path?query@,
-- @\/?query#fragment@ or @\/\/example.com@.
-- 
-- Absolute URIs have a scheme specified. Any other components of the URI which
-- are missing are specified as explicitly unset in the URI, rather than being
-- resolved relative to a base URI using 'GI.GLib.Structs.Uri.uriParseRelative'.
-- 
-- For example, a valid absolute URI is @file:\/\/\/home\/bob@ or
-- @https:\/\/search.com?query=string@.
-- 
-- A @GUri@ instance is always an absolute URI. A string may be an absolute URI
-- or a relative reference; see the documentation for individual functions as to
-- what forms they accept.
-- 
-- == Parsing URIs
-- 
-- The most minimalist APIs for parsing URIs are 'GI.GLib.Structs.Uri.uriSplit' and
-- 'GI.GLib.Structs.Uri.uriSplitWithUser'. These split a URI into its component
-- parts, and return the parts; the difference between the two is that
-- 'GI.GLib.Structs.Uri.uriSplit' treats the ‘userinfo’ component of the URI as a
-- single element, while 'GI.GLib.Structs.Uri.uriSplitWithUser' can (depending on the
-- [flags/@gLib@/.UriFlags] you pass) treat it as containing a username, password,
-- and authentication parameters. Alternatively, 'GI.GLib.Structs.Uri.uriSplitNetwork'
-- can be used when you are only interested in the components that are
-- needed to initiate a network connection to the service (scheme,
-- host, and port).
-- 
-- 'GI.GLib.Structs.Uri.uriParse' is similar to 'GI.GLib.Structs.Uri.uriSplit', but instead of
-- returning individual strings, it returns a @GUri@ structure (and it requires
-- that the URI be an absolute URI).
-- 
-- 'GI.GLib.Structs.Uri.uriResolveRelative' and 'GI.GLib.Structs.Uri.uriParseRelative' allow
-- you to resolve a relative URI relative to a base URI.
-- 'GI.GLib.Structs.Uri.uriResolveRelative' takes two strings and returns a string,
-- and 'GI.GLib.Structs.Uri.uriParseRelative' takes a @GUri@ and a string and returns a
-- @GUri@.
-- 
-- All of the parsing functions take a [flags/@gLib@/.UriFlags] argument describing
-- exactly how to parse the URI; see the documentation for that type
-- for more details on the specific flags that you can pass. If you
-- need to choose different flags based on the type of URI, you can
-- use 'GI.GLib.Structs.Uri.uriPeekScheme' on the URI string to check the scheme
-- first, and use that to decide what flags to parse it with.
-- 
-- For example, you might want to use @G_URI_PARAMS_WWW_FORM@ when parsing the
-- params for a web URI, so compare the result of 'GI.GLib.Structs.Uri.uriPeekScheme'
-- against @http@ and @https@.
-- 
-- == Building URIs
-- 
-- 'GI.GLib.Structs.Uri.uriJoin' and 'GI.GLib.Structs.Uri.uriJoinWithUser' can be used to construct
-- valid URI strings from a set of component strings. They are the
-- inverse of 'GI.GLib.Structs.Uri.uriSplit' and 'GI.GLib.Structs.Uri.uriSplitWithUser'.
-- 
-- Similarly, 'GI.GLib.Structs.Uri.uriBuild' and 'GI.GLib.Structs.Uri.uriBuildWithUser' can be
-- used to construct a @GUri@ from a set of component strings.
-- 
-- As with the parsing functions, the building functions take a
-- [flags/@gLib@/.UriFlags] argument. In particular, it is important to keep in mind
-- whether the URI components you are using are already @%@-encoded. If so,
-- you must pass the @G_URI_FLAGS_ENCODED@ flag.
-- 
-- == @file:\/\/@ URIs
-- 
-- Note that Windows and Unix both define special rules for parsing
-- @file:\/\/@ URIs (involving non-UTF-8 character sets on Unix, and the
-- interpretation of path separators on Windows). @GUri@ does not
-- implement these rules. Use 'GI.GLib.Functions.filenameFromUri' and
-- 'GI.GLib.Functions.filenameToUri' if you want to properly convert between
-- @file:\/\/@ URIs and local filenames.
-- 
-- == URI Equality
-- 
-- Note that there is no @g_uri_equal ()@ function, because comparing
-- URIs usefully requires scheme-specific knowledge that @GUri@ does
-- not have. @GUri@ can help with normalization if you use the various
-- encoded [flags/@gLib@/.UriFlags] as well as @G_URI_FLAGS_SCHEME_NORMALIZE@
-- however it is not comprehensive.
-- For example, @data:,foo@ and @data:;base64,Zm9v@ resolve to the same
-- thing according to the @data:@ URI specification which GLib does not
-- handle.
-- 
-- /Since: 2.66/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GLib.Structs.Uri
    ( 

-- * Exported types
    Uri(..)                                 ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [parseRelative]("GI.GLib.Structs.Uri#g:method:parseRelative"), [toString]("GI.GLib.Structs.Uri#g:method:toString"), [toStringPartial]("GI.GLib.Structs.Uri#g:method:toStringPartial").
-- 
-- ==== Getters
-- [getAuthParams]("GI.GLib.Structs.Uri#g:method:getAuthParams"), [getFlags]("GI.GLib.Structs.Uri#g:method:getFlags"), [getFragment]("GI.GLib.Structs.Uri#g:method:getFragment"), [getHost]("GI.GLib.Structs.Uri#g:method:getHost"), [getPassword]("GI.GLib.Structs.Uri#g:method:getPassword"), [getPath]("GI.GLib.Structs.Uri#g:method:getPath"), [getPort]("GI.GLib.Structs.Uri#g:method:getPort"), [getQuery]("GI.GLib.Structs.Uri#g:method:getQuery"), [getScheme]("GI.GLib.Structs.Uri#g:method:getScheme"), [getUser]("GI.GLib.Structs.Uri#g:method:getUser"), [getUserinfo]("GI.GLib.Structs.Uri#g:method:getUserinfo").
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveUriMethod                        ,
#endif

-- ** build #method:build#

    uriBuild                                ,


-- ** buildWithUser #method:buildWithUser#

    uriBuildWithUser                        ,


-- ** errorQuark #method:errorQuark#

    uriErrorQuark                           ,


-- ** escapeBytes #method:escapeBytes#

    uriEscapeBytes                          ,


-- ** escapeString #method:escapeString#

    uriEscapeString                         ,


-- ** getAuthParams #method:getAuthParams#

#if defined(ENABLE_OVERLOADING)
    UriGetAuthParamsMethodInfo              ,
#endif
    uriGetAuthParams                        ,


-- ** getFlags #method:getFlags#

#if defined(ENABLE_OVERLOADING)
    UriGetFlagsMethodInfo                   ,
#endif
    uriGetFlags                             ,


-- ** getFragment #method:getFragment#

#if defined(ENABLE_OVERLOADING)
    UriGetFragmentMethodInfo                ,
#endif
    uriGetFragment                          ,


-- ** getHost #method:getHost#

#if defined(ENABLE_OVERLOADING)
    UriGetHostMethodInfo                    ,
#endif
    uriGetHost                              ,


-- ** getPassword #method:getPassword#

#if defined(ENABLE_OVERLOADING)
    UriGetPasswordMethodInfo                ,
#endif
    uriGetPassword                          ,


-- ** getPath #method:getPath#

#if defined(ENABLE_OVERLOADING)
    UriGetPathMethodInfo                    ,
#endif
    uriGetPath                              ,


-- ** getPort #method:getPort#

#if defined(ENABLE_OVERLOADING)
    UriGetPortMethodInfo                    ,
#endif
    uriGetPort                              ,


-- ** getQuery #method:getQuery#

#if defined(ENABLE_OVERLOADING)
    UriGetQueryMethodInfo                   ,
#endif
    uriGetQuery                             ,


-- ** getScheme #method:getScheme#

#if defined(ENABLE_OVERLOADING)
    UriGetSchemeMethodInfo                  ,
#endif
    uriGetScheme                            ,


-- ** getUser #method:getUser#

#if defined(ENABLE_OVERLOADING)
    UriGetUserMethodInfo                    ,
#endif
    uriGetUser                              ,


-- ** getUserinfo #method:getUserinfo#

#if defined(ENABLE_OVERLOADING)
    UriGetUserinfoMethodInfo                ,
#endif
    uriGetUserinfo                          ,


-- ** isValid #method:isValid#

    uriIsValid                              ,


-- ** join #method:join#

    uriJoin                                 ,


-- ** joinWithUser #method:joinWithUser#

    uriJoinWithUser                         ,


-- ** listExtractUris #method:listExtractUris#

    uriListExtractUris                      ,


-- ** parse #method:parse#

    uriParse                                ,


-- ** parseParams #method:parseParams#

    uriParseParams                          ,


-- ** parseRelative #method:parseRelative#

#if defined(ENABLE_OVERLOADING)
    UriParseRelativeMethodInfo              ,
#endif
    uriParseRelative                        ,


-- ** parseScheme #method:parseScheme#

    uriParseScheme                          ,


-- ** peekScheme #method:peekScheme#

    uriPeekScheme                           ,


-- ** resolveRelative #method:resolveRelative#

    uriResolveRelative                      ,


-- ** split #method:split#

    uriSplit                                ,


-- ** splitNetwork #method:splitNetwork#

    uriSplitNetwork                         ,


-- ** splitWithUser #method:splitWithUser#

    uriSplitWithUser                        ,


-- ** toString #method:toString#

#if defined(ENABLE_OVERLOADING)
    UriToStringMethodInfo                   ,
#endif
    uriToString                             ,


-- ** toStringPartial #method:toStringPartial#

#if defined(ENABLE_OVERLOADING)
    UriToStringPartialMethodInfo            ,
#endif
    uriToStringPartial                      ,


-- ** unescapeBytes #method:unescapeBytes#

    uriUnescapeBytes                        ,


-- ** unescapeSegment #method:unescapeSegment#

    uriUnescapeSegment                      ,


-- ** unescapeString #method:unescapeString#

    uriUnescapeString                       ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import {-# SOURCE #-} qualified GI.GLib.Flags as GLib.Flags
import {-# SOURCE #-} qualified GI.GLib.Structs.Bytes as GLib.Bytes

#else
import {-# SOURCE #-} qualified GI.GLib.Flags as GLib.Flags
import {-# SOURCE #-} qualified GI.GLib.Structs.Bytes as GLib.Bytes

#endif

-- | Memory-managed wrapper type.
newtype Uri = Uri (SP.ManagedPtr Uri)
    deriving (Eq)

instance SP.ManagedPtrNewtype Uri where
    toManagedPtr (Uri p) = p

foreign import ccall "g_uri_get_type" c_g_uri_get_type :: 
    IO GType

type instance O.ParentTypes Uri = '[]
instance O.HasParentTypes Uri

instance B.Types.TypedObject Uri where
    glibType = c_g_uri_get_type

instance B.Types.GBoxed Uri

-- | Convert t'Uri' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Uri) where
    gvalueGType_ = c_g_uri_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr Uri)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr Uri)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed Uri ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Uri
type instance O.AttributeList Uri = UriAttributeList
type UriAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method Uri::get_auth_params
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "uri"
--           , argType = TInterface Name { namespace = "GLib" , name = "Uri" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GUri" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_get_auth_params" g_uri_get_auth_params :: 
    Ptr Uri ->                              -- uri : TInterface (Name {namespace = "GLib", name = "Uri"})
    IO CString

-- | Gets /@uri@/\'s authentication parameters, which may contain
-- @%@-encoding, depending on the flags with which /@uri@/ was created.
-- (If /@uri@/ was not created with 'GI.GLib.Flags.UriFlagsHasAuthParams' then this will
-- be 'P.Nothing'.)
-- 
-- Depending on the URI scheme, 'GI.GLib.Functions.uriParseParams' may be useful for
-- further parsing this information.
-- 
-- /Since: 2.66/
uriGetAuthParams ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Uri
    -- ^ /@uri@/: a t'GI.GLib.Structs.Uri.Uri'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ /@uri@/\'s authentication parameters.
uriGetAuthParams uri = liftIO $ do
    uri' <- unsafeManagedPtrGetPtr uri
    result <- g_uri_get_auth_params uri'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr uri
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data UriGetAuthParamsMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m) => O.OverloadedMethod UriGetAuthParamsMethodInfo Uri signature where
    overloadedMethod = uriGetAuthParams

instance O.OverloadedMethodInfo UriGetAuthParamsMethodInfo Uri where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Uri.uriGetAuthParams",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Uri.html#v:uriGetAuthParams"
        })


#endif

-- method Uri::get_flags
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "uri"
--           , argType = TInterface Name { namespace = "GLib" , name = "Uri" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GUri" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "UriFlags" })
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_get_flags" g_uri_get_flags :: 
    Ptr Uri ->                              -- uri : TInterface (Name {namespace = "GLib", name = "Uri"})
    IO CUInt

-- | Gets /@uri@/\'s flags set upon construction.
-- 
-- /Since: 2.66/
uriGetFlags ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Uri
    -- ^ /@uri@/: a t'GI.GLib.Structs.Uri.Uri'
    -> m [GLib.Flags.UriFlags]
    -- ^ __Returns:__ /@uri@/\'s flags.
uriGetFlags uri = liftIO $ do
    uri' <- unsafeManagedPtrGetPtr uri
    result <- g_uri_get_flags uri'
    let result' = wordToGFlags result
    touchManagedPtr uri
    return result'

#if defined(ENABLE_OVERLOADING)
data UriGetFlagsMethodInfo
instance (signature ~ (m [GLib.Flags.UriFlags]), MonadIO m) => O.OverloadedMethod UriGetFlagsMethodInfo Uri signature where
    overloadedMethod = uriGetFlags

instance O.OverloadedMethodInfo UriGetFlagsMethodInfo Uri where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Uri.uriGetFlags",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Uri.html#v:uriGetFlags"
        })


#endif

-- method Uri::get_fragment
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "uri"
--           , argType = TInterface Name { namespace = "GLib" , name = "Uri" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GUri" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_get_fragment" g_uri_get_fragment :: 
    Ptr Uri ->                              -- uri : TInterface (Name {namespace = "GLib", name = "Uri"})
    IO CString

-- | Gets /@uri@/\'s fragment, which may contain @%@-encoding, depending on
-- the flags with which /@uri@/ was created.
-- 
-- /Since: 2.66/
uriGetFragment ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Uri
    -- ^ /@uri@/: a t'GI.GLib.Structs.Uri.Uri'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ /@uri@/\'s fragment.
uriGetFragment uri = liftIO $ do
    uri' <- unsafeManagedPtrGetPtr uri
    result <- g_uri_get_fragment uri'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr uri
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data UriGetFragmentMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m) => O.OverloadedMethod UriGetFragmentMethodInfo Uri signature where
    overloadedMethod = uriGetFragment

instance O.OverloadedMethodInfo UriGetFragmentMethodInfo Uri where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Uri.uriGetFragment",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Uri.html#v:uriGetFragment"
        })


#endif

-- method Uri::get_host
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "uri"
--           , argType = TInterface Name { namespace = "GLib" , name = "Uri" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GUri" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_get_host" g_uri_get_host :: 
    Ptr Uri ->                              -- uri : TInterface (Name {namespace = "GLib", name = "Uri"})
    IO CString

-- | Gets /@uri@/\'s host. This will never have @%@-encoded characters,
-- unless it is non-UTF-8 (which can only be the case if /@uri@/ was
-- created with 'GI.GLib.Flags.UriFlagsNonDns').
-- 
-- If /@uri@/ contained an IPv6 address literal, this value will be just
-- that address, without the brackets around it that are necessary in
-- the string form of the URI. Note that in this case there may also
-- be a scope ID attached to the address. Eg, @fe80::1234%@@em1@ (or
-- @fe80::1234%@@25em1@ if the string is still encoded).
-- 
-- /Since: 2.66/
uriGetHost ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Uri
    -- ^ /@uri@/: a t'GI.GLib.Structs.Uri.Uri'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ /@uri@/\'s host.
uriGetHost uri = liftIO $ do
    uri' <- unsafeManagedPtrGetPtr uri
    result <- g_uri_get_host uri'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr uri
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data UriGetHostMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m) => O.OverloadedMethod UriGetHostMethodInfo Uri signature where
    overloadedMethod = uriGetHost

instance O.OverloadedMethodInfo UriGetHostMethodInfo Uri where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Uri.uriGetHost",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Uri.html#v:uriGetHost"
        })


#endif

-- method Uri::get_password
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "uri"
--           , argType = TInterface Name { namespace = "GLib" , name = "Uri" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GUri" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_get_password" g_uri_get_password :: 
    Ptr Uri ->                              -- uri : TInterface (Name {namespace = "GLib", name = "Uri"})
    IO CString

-- | Gets /@uri@/\'s password, which may contain @%@-encoding, depending on
-- the flags with which /@uri@/ was created. (If /@uri@/ was not created
-- with 'GI.GLib.Flags.UriFlagsHasPassword' then this will be 'P.Nothing'.)
-- 
-- /Since: 2.66/
uriGetPassword ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Uri
    -- ^ /@uri@/: a t'GI.GLib.Structs.Uri.Uri'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ /@uri@/\'s password.
uriGetPassword uri = liftIO $ do
    uri' <- unsafeManagedPtrGetPtr uri
    result <- g_uri_get_password uri'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr uri
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data UriGetPasswordMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m) => O.OverloadedMethod UriGetPasswordMethodInfo Uri signature where
    overloadedMethod = uriGetPassword

instance O.OverloadedMethodInfo UriGetPasswordMethodInfo Uri where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Uri.uriGetPassword",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Uri.html#v:uriGetPassword"
        })


#endif

-- method Uri::get_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "uri"
--           , argType = TInterface Name { namespace = "GLib" , name = "Uri" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GUri" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_get_path" g_uri_get_path :: 
    Ptr Uri ->                              -- uri : TInterface (Name {namespace = "GLib", name = "Uri"})
    IO CString

-- | Gets /@uri@/\'s path, which may contain @%@-encoding, depending on the
-- flags with which /@uri@/ was created.
-- 
-- /Since: 2.66/
uriGetPath ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Uri
    -- ^ /@uri@/: a t'GI.GLib.Structs.Uri.Uri'
    -> m T.Text
    -- ^ __Returns:__ /@uri@/\'s path.
uriGetPath uri = liftIO $ do
    uri' <- unsafeManagedPtrGetPtr uri
    result <- g_uri_get_path uri'
    checkUnexpectedReturnNULL "uriGetPath" result
    result' <- cstringToText result
    touchManagedPtr uri
    return result'

#if defined(ENABLE_OVERLOADING)
data UriGetPathMethodInfo
instance (signature ~ (m T.Text), MonadIO m) => O.OverloadedMethod UriGetPathMethodInfo Uri signature where
    overloadedMethod = uriGetPath

instance O.OverloadedMethodInfo UriGetPathMethodInfo Uri where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Uri.uriGetPath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Uri.html#v:uriGetPath"
        })


#endif

-- method Uri::get_port
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "uri"
--           , argType = TInterface Name { namespace = "GLib" , name = "Uri" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GUri" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_get_port" g_uri_get_port :: 
    Ptr Uri ->                              -- uri : TInterface (Name {namespace = "GLib", name = "Uri"})
    IO Int32

-- | Gets /@uri@/\'s port.
-- 
-- /Since: 2.66/
uriGetPort ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Uri
    -- ^ /@uri@/: a t'GI.GLib.Structs.Uri.Uri'
    -> m Int32
    -- ^ __Returns:__ /@uri@/\'s port, or @-1@ if no port was specified.
uriGetPort uri = liftIO $ do
    uri' <- unsafeManagedPtrGetPtr uri
    result <- g_uri_get_port uri'
    touchManagedPtr uri
    return result

#if defined(ENABLE_OVERLOADING)
data UriGetPortMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod UriGetPortMethodInfo Uri signature where
    overloadedMethod = uriGetPort

instance O.OverloadedMethodInfo UriGetPortMethodInfo Uri where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Uri.uriGetPort",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Uri.html#v:uriGetPort"
        })


#endif

-- method Uri::get_query
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "uri"
--           , argType = TInterface Name { namespace = "GLib" , name = "Uri" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GUri" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_get_query" g_uri_get_query :: 
    Ptr Uri ->                              -- uri : TInterface (Name {namespace = "GLib", name = "Uri"})
    IO CString

-- | Gets /@uri@/\'s query, which may contain @%@-encoding, depending on the
-- flags with which /@uri@/ was created.
-- 
-- For queries consisting of a series of @name=value@ parameters,
-- t'GI.GLib.Structs.UriParamsIter.UriParamsIter' or 'GI.GLib.Functions.uriParseParams' may be useful.
-- 
-- /Since: 2.66/
uriGetQuery ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Uri
    -- ^ /@uri@/: a t'GI.GLib.Structs.Uri.Uri'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ /@uri@/\'s query.
uriGetQuery uri = liftIO $ do
    uri' <- unsafeManagedPtrGetPtr uri
    result <- g_uri_get_query uri'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr uri
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data UriGetQueryMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m) => O.OverloadedMethod UriGetQueryMethodInfo Uri signature where
    overloadedMethod = uriGetQuery

instance O.OverloadedMethodInfo UriGetQueryMethodInfo Uri where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Uri.uriGetQuery",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Uri.html#v:uriGetQuery"
        })


#endif

-- method Uri::get_scheme
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "uri"
--           , argType = TInterface Name { namespace = "GLib" , name = "Uri" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GUri" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_get_scheme" g_uri_get_scheme :: 
    Ptr Uri ->                              -- uri : TInterface (Name {namespace = "GLib", name = "Uri"})
    IO CString

-- | Gets /@uri@/\'s scheme. Note that this will always be all-lowercase,
-- regardless of the string or strings that /@uri@/ was created from.
-- 
-- /Since: 2.66/
uriGetScheme ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Uri
    -- ^ /@uri@/: a t'GI.GLib.Structs.Uri.Uri'
    -> m T.Text
    -- ^ __Returns:__ /@uri@/\'s scheme.
uriGetScheme uri = liftIO $ do
    uri' <- unsafeManagedPtrGetPtr uri
    result <- g_uri_get_scheme uri'
    checkUnexpectedReturnNULL "uriGetScheme" result
    result' <- cstringToText result
    touchManagedPtr uri
    return result'

#if defined(ENABLE_OVERLOADING)
data UriGetSchemeMethodInfo
instance (signature ~ (m T.Text), MonadIO m) => O.OverloadedMethod UriGetSchemeMethodInfo Uri signature where
    overloadedMethod = uriGetScheme

instance O.OverloadedMethodInfo UriGetSchemeMethodInfo Uri where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Uri.uriGetScheme",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Uri.html#v:uriGetScheme"
        })


#endif

-- method Uri::get_user
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "uri"
--           , argType = TInterface Name { namespace = "GLib" , name = "Uri" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GUri" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_get_user" g_uri_get_user :: 
    Ptr Uri ->                              -- uri : TInterface (Name {namespace = "GLib", name = "Uri"})
    IO CString

-- | Gets the ‘username’ component of /@uri@/\'s userinfo, which may contain
-- @%@-encoding, depending on the flags with which /@uri@/ was created.
-- If /@uri@/ was not created with 'GI.GLib.Flags.UriFlagsHasPassword' or
-- 'GI.GLib.Flags.UriFlagsHasAuthParams', this is the same as 'GI.GLib.Structs.Uri.uriGetUserinfo'.
-- 
-- /Since: 2.66/
uriGetUser ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Uri
    -- ^ /@uri@/: a t'GI.GLib.Structs.Uri.Uri'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ /@uri@/\'s user.
uriGetUser uri = liftIO $ do
    uri' <- unsafeManagedPtrGetPtr uri
    result <- g_uri_get_user uri'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr uri
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data UriGetUserMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m) => O.OverloadedMethod UriGetUserMethodInfo Uri signature where
    overloadedMethod = uriGetUser

instance O.OverloadedMethodInfo UriGetUserMethodInfo Uri where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Uri.uriGetUser",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Uri.html#v:uriGetUser"
        })


#endif

-- method Uri::get_userinfo
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "uri"
--           , argType = TInterface Name { namespace = "GLib" , name = "Uri" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GUri" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_get_userinfo" g_uri_get_userinfo :: 
    Ptr Uri ->                              -- uri : TInterface (Name {namespace = "GLib", name = "Uri"})
    IO CString

-- | Gets /@uri@/\'s userinfo, which may contain @%@-encoding, depending on
-- the flags with which /@uri@/ was created.
-- 
-- /Since: 2.66/
uriGetUserinfo ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Uri
    -- ^ /@uri@/: a t'GI.GLib.Structs.Uri.Uri'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ /@uri@/\'s userinfo.
uriGetUserinfo uri = liftIO $ do
    uri' <- unsafeManagedPtrGetPtr uri
    result <- g_uri_get_userinfo uri'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr uri
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data UriGetUserinfoMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m) => O.OverloadedMethod UriGetUserinfoMethodInfo Uri signature where
    overloadedMethod = uriGetUserinfo

instance O.OverloadedMethodInfo UriGetUserinfoMethodInfo Uri where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Uri.uriGetUserinfo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Uri.html#v:uriGetUserinfo"
        })


#endif

-- method Uri::parse_relative
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "base_uri"
--           , argType = TInterface Name { namespace = "GLib" , name = "Uri" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a base absolute URI"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri_ref"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a string representing a relative or absolute URI"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "UriFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags describing how to parse @uri_ref"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Uri" })
-- throws : True
-- Skip return : False

foreign import ccall "g_uri_parse_relative" g_uri_parse_relative :: 
    Ptr Uri ->                              -- base_uri : TInterface (Name {namespace = "GLib", name = "Uri"})
    CString ->                              -- uri_ref : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "UriFlags"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Uri)

-- | Parses /@uriRef@/ according to /@flags@/ and, if it is a
-- <http://developer.gnome.org/glib/stable/#relative-and-absolute-uris relative URI>, resolves it relative to /@baseUri@/.
-- If the result is not a valid absolute URI, it will be discarded, and an error
-- returned.
-- 
-- /Since: 2.66/
uriParseRelative ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Uri)
    -- ^ /@baseUri@/: a base absolute URI
    -> T.Text
    -- ^ /@uriRef@/: a string representing a relative or absolute URI
    -> [GLib.Flags.UriFlags]
    -- ^ /@flags@/: flags describing how to parse /@uriRef@/
    -> m Uri
    -- ^ __Returns:__ a new t'GI.GLib.Structs.Uri.Uri', or NULL on error. /(Can throw 'Data.GI.Base.GError.GError')/
uriParseRelative baseUri uriRef flags = liftIO $ do
    maybeBaseUri <- case baseUri of
        Nothing -> return FP.nullPtr
        Just jBaseUri -> do
            jBaseUri' <- unsafeManagedPtrGetPtr jBaseUri
            return jBaseUri'
    uriRef' <- textToCString uriRef
    let flags' = gflagsToWord flags
    onException (do
        result <- propagateGError $ g_uri_parse_relative maybeBaseUri uriRef' flags'
        checkUnexpectedReturnNULL "uriParseRelative" result
        result' <- (wrapBoxed Uri) result
        whenJust baseUri touchManagedPtr
        freeMem uriRef'
        return result'
     ) (do
        freeMem uriRef'
     )

#if defined(ENABLE_OVERLOADING)
data UriParseRelativeMethodInfo
instance (signature ~ (T.Text -> [GLib.Flags.UriFlags] -> m Uri), MonadIO m) => O.OverloadedMethod UriParseRelativeMethodInfo Uri signature where
    overloadedMethod i = uriParseRelative (Just i)

instance O.OverloadedMethodInfo UriParseRelativeMethodInfo Uri where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Uri.uriParseRelative",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Uri.html#v:uriParseRelative"
        })


#endif

-- method Uri::to_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "uri"
--           , argType = TInterface Name { namespace = "GLib" , name = "Uri" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GUri" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_to_string" g_uri_to_string :: 
    Ptr Uri ->                              -- uri : TInterface (Name {namespace = "GLib", name = "Uri"})
    IO CString

-- | Returns a string representing /@uri@/.
-- 
-- This is not guaranteed to return a string which is identical to the
-- string that /@uri@/ was parsed from. However, if the source URI was
-- syntactically correct (according to RFC 3986), and it was parsed
-- with 'GI.GLib.Flags.UriFlagsEncoded', then 'GI.GLib.Structs.Uri.uriToString' is guaranteed to return
-- a string which is at least semantically equivalent to the source
-- URI (according to RFC 3986).
-- 
-- If /@uri@/ might contain sensitive details, such as authentication parameters,
-- or private data in its query string, and the returned string is going to be
-- logged, then consider using 'GI.GLib.Structs.Uri.uriToStringPartial' to redact parts.
-- 
-- /Since: 2.66/
uriToString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Uri
    -- ^ /@uri@/: a t'GI.GLib.Structs.Uri.Uri'
    -> m T.Text
    -- ^ __Returns:__ a string representing /@uri@/,
    --     which the caller must free.
uriToString uri = liftIO $ do
    uri' <- unsafeManagedPtrGetPtr uri
    result <- g_uri_to_string uri'
    checkUnexpectedReturnNULL "uriToString" result
    result' <- cstringToText result
    freeMem result
    touchManagedPtr uri
    return result'

#if defined(ENABLE_OVERLOADING)
data UriToStringMethodInfo
instance (signature ~ (m T.Text), MonadIO m) => O.OverloadedMethod UriToStringMethodInfo Uri signature where
    overloadedMethod = uriToString

instance O.OverloadedMethodInfo UriToStringMethodInfo Uri where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Uri.uriToString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Uri.html#v:uriToString"
        })


#endif

-- method Uri::to_string_partial
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "uri"
--           , argType = TInterface Name { namespace = "GLib" , name = "Uri" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GUri" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "UriHideFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags describing what parts of @uri to hide"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_to_string_partial" g_uri_to_string_partial :: 
    Ptr Uri ->                              -- uri : TInterface (Name {namespace = "GLib", name = "Uri"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "UriHideFlags"})
    IO CString

-- | Returns a string representing /@uri@/, subject to the options in
-- /@flags@/. See 'GI.GLib.Structs.Uri.uriToString' and t'GI.GLib.Flags.UriHideFlags' for more details.
-- 
-- /Since: 2.66/
uriToStringPartial ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Uri
    -- ^ /@uri@/: a t'GI.GLib.Structs.Uri.Uri'
    -> [GLib.Flags.UriHideFlags]
    -- ^ /@flags@/: flags describing what parts of /@uri@/ to hide
    -> m T.Text
    -- ^ __Returns:__ a string representing
    --     /@uri@/, which the caller must free.
uriToStringPartial uri flags = liftIO $ do
    uri' <- unsafeManagedPtrGetPtr uri
    let flags' = gflagsToWord flags
    result <- g_uri_to_string_partial uri' flags'
    checkUnexpectedReturnNULL "uriToStringPartial" result
    result' <- cstringToText result
    freeMem result
    touchManagedPtr uri
    return result'

#if defined(ENABLE_OVERLOADING)
data UriToStringPartialMethodInfo
instance (signature ~ ([GLib.Flags.UriHideFlags] -> m T.Text), MonadIO m) => O.OverloadedMethod UriToStringPartialMethodInfo Uri signature where
    overloadedMethod = uriToStringPartial

instance O.OverloadedMethodInfo UriToStringPartialMethodInfo Uri where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Uri.uriToStringPartial",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Uri.html#v:uriToStringPartial"
        })


#endif

-- method Uri::build
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "UriFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags describing how to build the #GUri"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "scheme"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the URI scheme" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "userinfo"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the userinfo component, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "host"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the host component, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "port"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the port, or `-1`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the path component" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "query"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the query component, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "fragment"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the fragment, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Uri" })
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_build" g_uri_build :: 
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "UriFlags"})
    CString ->                              -- scheme : TBasicType TUTF8
    CString ->                              -- userinfo : TBasicType TUTF8
    CString ->                              -- host : TBasicType TUTF8
    Int32 ->                                -- port : TBasicType TInt
    CString ->                              -- path : TBasicType TUTF8
    CString ->                              -- query : TBasicType TUTF8
    CString ->                              -- fragment : TBasicType TUTF8
    IO (Ptr Uri)

-- | Creates a new t'GI.GLib.Structs.Uri.Uri' from the given components according to /@flags@/.
-- 
-- See also 'GI.GLib.Functions.uriBuildWithUser', which allows specifying the
-- components of the \"userinfo\" separately.
-- 
-- /Since: 2.66/
uriBuild ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    [GLib.Flags.UriFlags]
    -- ^ /@flags@/: flags describing how to build the t'GI.GLib.Structs.Uri.Uri'
    -> T.Text
    -- ^ /@scheme@/: the URI scheme
    -> Maybe (T.Text)
    -- ^ /@userinfo@/: the userinfo component, or 'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@host@/: the host component, or 'P.Nothing'
    -> Int32
    -- ^ /@port@/: the port, or @-1@
    -> T.Text
    -- ^ /@path@/: the path component
    -> Maybe (T.Text)
    -- ^ /@query@/: the query component, or 'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@fragment@/: the fragment, or 'P.Nothing'
    -> m Uri
    -- ^ __Returns:__ a new t'GI.GLib.Structs.Uri.Uri'
uriBuild flags scheme userinfo host port path query fragment = liftIO $ do
    let flags' = gflagsToWord flags
    scheme' <- textToCString scheme
    maybeUserinfo <- case userinfo of
        Nothing -> return FP.nullPtr
        Just jUserinfo -> do
            jUserinfo' <- textToCString jUserinfo
            return jUserinfo'
    maybeHost <- case host of
        Nothing -> return FP.nullPtr
        Just jHost -> do
            jHost' <- textToCString jHost
            return jHost'
    path' <- textToCString path
    maybeQuery <- case query of
        Nothing -> return FP.nullPtr
        Just jQuery -> do
            jQuery' <- textToCString jQuery
            return jQuery'
    maybeFragment <- case fragment of
        Nothing -> return FP.nullPtr
        Just jFragment -> do
            jFragment' <- textToCString jFragment
            return jFragment'
    result <- g_uri_build flags' scheme' maybeUserinfo maybeHost port path' maybeQuery maybeFragment
    checkUnexpectedReturnNULL "uriBuild" result
    result' <- (wrapBoxed Uri) result
    freeMem scheme'
    freeMem maybeUserinfo
    freeMem maybeHost
    freeMem path'
    freeMem maybeQuery
    freeMem maybeFragment
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::build_with_user
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "UriFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags describing how to build the #GUri"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "scheme"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the URI scheme" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the user component of the userinfo, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "password"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the password component of the userinfo, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "auth_params"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the auth params of the userinfo, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "host"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the host component, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "port"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the port, or `-1`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the path component" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "query"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the query component, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "fragment"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the fragment, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Uri" })
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_build_with_user" g_uri_build_with_user :: 
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "UriFlags"})
    CString ->                              -- scheme : TBasicType TUTF8
    CString ->                              -- user : TBasicType TUTF8
    CString ->                              -- password : TBasicType TUTF8
    CString ->                              -- auth_params : TBasicType TUTF8
    CString ->                              -- host : TBasicType TUTF8
    Int32 ->                                -- port : TBasicType TInt
    CString ->                              -- path : TBasicType TUTF8
    CString ->                              -- query : TBasicType TUTF8
    CString ->                              -- fragment : TBasicType TUTF8
    IO (Ptr Uri)

-- | Creates a new t'GI.GLib.Structs.Uri.Uri' from the given components according to /@flags@/
-- ('GI.GLib.Flags.UriFlagsHasPassword' is added unconditionally). The /@flags@/ must be
-- coherent with the passed values, in particular use @%@-encoded values with
-- 'GI.GLib.Flags.UriFlagsEncoded'.
-- 
-- In contrast to 'GI.GLib.Functions.uriBuild', this allows specifying the components
-- of the ‘userinfo’ field separately. Note that /@user@/ must be non-'P.Nothing'
-- if either /@password@/ or /@authParams@/ is non-'P.Nothing'.
-- 
-- /Since: 2.66/
uriBuildWithUser ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    [GLib.Flags.UriFlags]
    -- ^ /@flags@/: flags describing how to build the t'GI.GLib.Structs.Uri.Uri'
    -> T.Text
    -- ^ /@scheme@/: the URI scheme
    -> Maybe (T.Text)
    -- ^ /@user@/: the user component of the userinfo, or 'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@password@/: the password component of the userinfo, or 'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@authParams@/: the auth params of the userinfo, or 'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@host@/: the host component, or 'P.Nothing'
    -> Int32
    -- ^ /@port@/: the port, or @-1@
    -> T.Text
    -- ^ /@path@/: the path component
    -> Maybe (T.Text)
    -- ^ /@query@/: the query component, or 'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@fragment@/: the fragment, or 'P.Nothing'
    -> m Uri
    -- ^ __Returns:__ a new t'GI.GLib.Structs.Uri.Uri'
uriBuildWithUser flags scheme user password authParams host port path query fragment = liftIO $ do
    let flags' = gflagsToWord flags
    scheme' <- textToCString scheme
    maybeUser <- case user of
        Nothing -> return FP.nullPtr
        Just jUser -> do
            jUser' <- textToCString jUser
            return jUser'
    maybePassword <- case password of
        Nothing -> return FP.nullPtr
        Just jPassword -> do
            jPassword' <- textToCString jPassword
            return jPassword'
    maybeAuthParams <- case authParams of
        Nothing -> return FP.nullPtr
        Just jAuthParams -> do
            jAuthParams' <- textToCString jAuthParams
            return jAuthParams'
    maybeHost <- case host of
        Nothing -> return FP.nullPtr
        Just jHost -> do
            jHost' <- textToCString jHost
            return jHost'
    path' <- textToCString path
    maybeQuery <- case query of
        Nothing -> return FP.nullPtr
        Just jQuery -> do
            jQuery' <- textToCString jQuery
            return jQuery'
    maybeFragment <- case fragment of
        Nothing -> return FP.nullPtr
        Just jFragment -> do
            jFragment' <- textToCString jFragment
            return jFragment'
    result <- g_uri_build_with_user flags' scheme' maybeUser maybePassword maybeAuthParams maybeHost port path' maybeQuery maybeFragment
    checkUnexpectedReturnNULL "uriBuildWithUser" result
    result' <- (wrapBoxed Uri) result
    freeMem scheme'
    freeMem maybeUser
    freeMem maybePassword
    freeMem maybeAuthParams
    freeMem maybeHost
    freeMem path'
    freeMem maybeQuery
    freeMem maybeFragment
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::error_quark
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just (TBasicType TUInt32)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_error_quark" g_uri_error_quark :: 
    IO Word32

-- | /No description available in the introspection data./
uriErrorQuark ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Word32
uriErrorQuark  = liftIO $ do
    result <- g_uri_error_quark
    return result

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::escape_bytes
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "unescaped"
--           , argType = TCArray False (-1) 1 (TBasicType TUInt8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the unescaped input data."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the length of @unescaped"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "reserved_chars_allowed"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a string of reserved\n  characters that are allowed to be used, or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the length of @unescaped"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_escape_bytes" g_uri_escape_bytes :: 
    Ptr Word8 ->                            -- unescaped : TCArray False (-1) 1 (TBasicType TUInt8)
    FCT.CSize ->                            -- length : TBasicType TSize
    CString ->                              -- reserved_chars_allowed : TBasicType TUTF8
    IO CString

-- | Escapes arbitrary data for use in a URI.
-- 
-- Normally all characters that are not ‘unreserved’ (i.e. ASCII
-- alphanumerical characters plus dash, dot, underscore and tilde) are
-- escaped. But if you specify characters in /@reservedCharsAllowed@/
-- they are not escaped. This is useful for the ‘reserved’ characters
-- in the URI specification, since those are allowed unescaped in some
-- portions of a URI.
-- 
-- Though technically incorrect, this will also allow escaping nul
-- bytes as @%@@00@.
-- 
-- /Since: 2.66/
uriEscapeBytes ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    ByteString
    -- ^ /@unescaped@/: the unescaped input data.
    -> Maybe (T.Text)
    -- ^ /@reservedCharsAllowed@/: a string of reserved
    --   characters that are allowed to be used, or 'P.Nothing'.
    -> m T.Text
    -- ^ __Returns:__ an escaped version of /@unescaped@/.
    --     The returned string should be freed when no longer needed.
uriEscapeBytes unescaped reservedCharsAllowed = liftIO $ do
    let length_ = fromIntegral $ B.length unescaped
    unescaped' <- packByteString unescaped
    maybeReservedCharsAllowed <- case reservedCharsAllowed of
        Nothing -> return FP.nullPtr
        Just jReservedCharsAllowed -> do
            jReservedCharsAllowed' <- textToCString jReservedCharsAllowed
            return jReservedCharsAllowed'
    result <- g_uri_escape_bytes unescaped' length_ maybeReservedCharsAllowed
    checkUnexpectedReturnNULL "uriEscapeBytes" result
    result' <- cstringToText result
    freeMem result
    freeMem unescaped'
    freeMem maybeReservedCharsAllowed
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::escape_string
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "unescaped"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the unescaped input string."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "reserved_chars_allowed"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a string of reserved\n  characters that are allowed to be used, or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "allow_utf8"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "%TRUE if the result can include UTF-8 characters."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_escape_string" g_uri_escape_string :: 
    CString ->                              -- unescaped : TBasicType TUTF8
    CString ->                              -- reserved_chars_allowed : TBasicType TUTF8
    CInt ->                                 -- allow_utf8 : TBasicType TBoolean
    IO CString

-- | Escapes a string for use in a URI.
-- 
-- Normally all characters that are not \"unreserved\" (i.e. ASCII
-- alphanumerical characters plus dash, dot, underscore and tilde) are
-- escaped. But if you specify characters in /@reservedCharsAllowed@/
-- they are not escaped. This is useful for the \"reserved\" characters
-- in the URI specification, since those are allowed unescaped in some
-- portions of a URI.
-- 
-- /Since: 2.16/
uriEscapeString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@unescaped@/: the unescaped input string.
    -> Maybe (T.Text)
    -- ^ /@reservedCharsAllowed@/: a string of reserved
    --   characters that are allowed to be used, or 'P.Nothing'.
    -> Bool
    -- ^ /@allowUtf8@/: 'P.True' if the result can include UTF-8 characters.
    -> m T.Text
    -- ^ __Returns:__ an escaped version of /@unescaped@/. The
    -- returned string should be freed when no longer needed.
uriEscapeString unescaped reservedCharsAllowed allowUtf8 = liftIO $ do
    unescaped' <- textToCString unescaped
    maybeReservedCharsAllowed <- case reservedCharsAllowed of
        Nothing -> return FP.nullPtr
        Just jReservedCharsAllowed -> do
            jReservedCharsAllowed' <- textToCString jReservedCharsAllowed
            return jReservedCharsAllowed'
    let allowUtf8' = (P.fromIntegral . P.fromEnum) allowUtf8
    result <- g_uri_escape_string unescaped' maybeReservedCharsAllowed allowUtf8'
    checkUnexpectedReturnNULL "uriEscapeString" result
    result' <- cstringToText result
    freeMem result
    freeMem unescaped'
    freeMem maybeReservedCharsAllowed
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::is_valid
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "uri_string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string containing an absolute URI"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "UriFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags for parsing @uri_string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_uri_is_valid" g_uri_is_valid :: 
    CString ->                              -- uri_string : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "UriFlags"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Parses /@uriString@/ according to /@flags@/, to determine whether it is a valid
-- <http://developer.gnome.org/glib/stable/#relative-and-absolute-uris absolute URI>, i.e. it does not need to be resolved
-- relative to another URI using 'GI.GLib.Structs.Uri.uriParseRelative'.
-- 
-- If it’s not a valid URI, an error is returned explaining how it’s invalid.
-- 
-- See 'GI.GLib.Functions.uriSplit', and the definition of t'GI.GLib.Flags.UriFlags', for more
-- information on the effect of /@flags@/.
-- 
-- /Since: 2.66/
uriIsValid ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@uriString@/: a string containing an absolute URI
    -> [GLib.Flags.UriFlags]
    -- ^ /@flags@/: flags for parsing /@uriString@/
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
uriIsValid uriString flags = liftIO $ do
    uriString' <- textToCString uriString
    let flags' = gflagsToWord flags
    onException (do
        _ <- propagateGError $ g_uri_is_valid uriString' flags'
        freeMem uriString'
        return ()
     ) (do
        freeMem uriString'
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::join
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "UriFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags describing how to build the URI string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "scheme"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the URI scheme, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "userinfo"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the userinfo component, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "host"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the host component, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "port"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the port, or `-1`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the path component" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "query"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the query component, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "fragment"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the fragment, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_join" g_uri_join :: 
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "UriFlags"})
    CString ->                              -- scheme : TBasicType TUTF8
    CString ->                              -- userinfo : TBasicType TUTF8
    CString ->                              -- host : TBasicType TUTF8
    Int32 ->                                -- port : TBasicType TInt
    CString ->                              -- path : TBasicType TUTF8
    CString ->                              -- query : TBasicType TUTF8
    CString ->                              -- fragment : TBasicType TUTF8
    IO CString

-- | Joins the given components together according to /@flags@/ to create
-- an absolute URI string. /@path@/ may not be 'P.Nothing' (though it may be the empty
-- string).
-- 
-- When /@host@/ is present, /@path@/ must either be empty or begin with a slash (@\/@)
-- character. When /@host@/ is not present, /@path@/ cannot begin with two slash
-- characters (@\/\/@). See
-- <https://tools.ietf.org/html/rfc3986#section-3 RFC 3986, section 3>.
-- 
-- See also 'GI.GLib.Functions.uriJoinWithUser', which allows specifying the
-- components of the ‘userinfo’ separately.
-- 
-- 'GI.GLib.Flags.UriFlagsHasPassword' and 'GI.GLib.Flags.UriFlagsHasAuthParams' are ignored if set
-- in /@flags@/.
-- 
-- /Since: 2.66/
uriJoin ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    [GLib.Flags.UriFlags]
    -- ^ /@flags@/: flags describing how to build the URI string
    -> Maybe (T.Text)
    -- ^ /@scheme@/: the URI scheme, or 'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@userinfo@/: the userinfo component, or 'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@host@/: the host component, or 'P.Nothing'
    -> Int32
    -- ^ /@port@/: the port, or @-1@
    -> T.Text
    -- ^ /@path@/: the path component
    -> Maybe (T.Text)
    -- ^ /@query@/: the query component, or 'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@fragment@/: the fragment, or 'P.Nothing'
    -> m T.Text
    -- ^ __Returns:__ an absolute URI string
uriJoin flags scheme userinfo host port path query fragment = liftIO $ do
    let flags' = gflagsToWord flags
    maybeScheme <- case scheme of
        Nothing -> return FP.nullPtr
        Just jScheme -> do
            jScheme' <- textToCString jScheme
            return jScheme'
    maybeUserinfo <- case userinfo of
        Nothing -> return FP.nullPtr
        Just jUserinfo -> do
            jUserinfo' <- textToCString jUserinfo
            return jUserinfo'
    maybeHost <- case host of
        Nothing -> return FP.nullPtr
        Just jHost -> do
            jHost' <- textToCString jHost
            return jHost'
    path' <- textToCString path
    maybeQuery <- case query of
        Nothing -> return FP.nullPtr
        Just jQuery -> do
            jQuery' <- textToCString jQuery
            return jQuery'
    maybeFragment <- case fragment of
        Nothing -> return FP.nullPtr
        Just jFragment -> do
            jFragment' <- textToCString jFragment
            return jFragment'
    result <- g_uri_join flags' maybeScheme maybeUserinfo maybeHost port path' maybeQuery maybeFragment
    checkUnexpectedReturnNULL "uriJoin" result
    result' <- cstringToText result
    freeMem result
    freeMem maybeScheme
    freeMem maybeUserinfo
    freeMem maybeHost
    freeMem path'
    freeMem maybeQuery
    freeMem maybeFragment
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::join_with_user
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "UriFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags describing how to build the URI string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "scheme"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the URI scheme, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the user component of the userinfo, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "password"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the password component of the userinfo, or\n  %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "auth_params"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the auth params of the userinfo, or\n  %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "host"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the host component, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "port"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the port, or `-1`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the path component" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "query"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the query component, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "fragment"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the fragment, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_join_with_user" g_uri_join_with_user :: 
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "UriFlags"})
    CString ->                              -- scheme : TBasicType TUTF8
    CString ->                              -- user : TBasicType TUTF8
    CString ->                              -- password : TBasicType TUTF8
    CString ->                              -- auth_params : TBasicType TUTF8
    CString ->                              -- host : TBasicType TUTF8
    Int32 ->                                -- port : TBasicType TInt
    CString ->                              -- path : TBasicType TUTF8
    CString ->                              -- query : TBasicType TUTF8
    CString ->                              -- fragment : TBasicType TUTF8
    IO CString

-- | Joins the given components together according to /@flags@/ to create
-- an absolute URI string. /@path@/ may not be 'P.Nothing' (though it may be the empty
-- string).
-- 
-- In contrast to 'GI.GLib.Functions.uriJoin', this allows specifying the components
-- of the ‘userinfo’ separately. It otherwise behaves the same.
-- 
-- 'GI.GLib.Flags.UriFlagsHasPassword' and 'GI.GLib.Flags.UriFlagsHasAuthParams' are ignored if set
-- in /@flags@/.
-- 
-- /Since: 2.66/
uriJoinWithUser ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    [GLib.Flags.UriFlags]
    -- ^ /@flags@/: flags describing how to build the URI string
    -> Maybe (T.Text)
    -- ^ /@scheme@/: the URI scheme, or 'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@user@/: the user component of the userinfo, or 'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@password@/: the password component of the userinfo, or
    --   'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@authParams@/: the auth params of the userinfo, or
    --   'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@host@/: the host component, or 'P.Nothing'
    -> Int32
    -- ^ /@port@/: the port, or @-1@
    -> T.Text
    -- ^ /@path@/: the path component
    -> Maybe (T.Text)
    -- ^ /@query@/: the query component, or 'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@fragment@/: the fragment, or 'P.Nothing'
    -> m T.Text
    -- ^ __Returns:__ an absolute URI string
uriJoinWithUser flags scheme user password authParams host port path query fragment = liftIO $ do
    let flags' = gflagsToWord flags
    maybeScheme <- case scheme of
        Nothing -> return FP.nullPtr
        Just jScheme -> do
            jScheme' <- textToCString jScheme
            return jScheme'
    maybeUser <- case user of
        Nothing -> return FP.nullPtr
        Just jUser -> do
            jUser' <- textToCString jUser
            return jUser'
    maybePassword <- case password of
        Nothing -> return FP.nullPtr
        Just jPassword -> do
            jPassword' <- textToCString jPassword
            return jPassword'
    maybeAuthParams <- case authParams of
        Nothing -> return FP.nullPtr
        Just jAuthParams -> do
            jAuthParams' <- textToCString jAuthParams
            return jAuthParams'
    maybeHost <- case host of
        Nothing -> return FP.nullPtr
        Just jHost -> do
            jHost' <- textToCString jHost
            return jHost'
    path' <- textToCString path
    maybeQuery <- case query of
        Nothing -> return FP.nullPtr
        Just jQuery -> do
            jQuery' <- textToCString jQuery
            return jQuery'
    maybeFragment <- case fragment of
        Nothing -> return FP.nullPtr
        Just jFragment -> do
            jFragment' <- textToCString jFragment
            return jFragment'
    result <- g_uri_join_with_user flags' maybeScheme maybeUser maybePassword maybeAuthParams maybeHost port path' maybeQuery maybeFragment
    checkUnexpectedReturnNULL "uriJoinWithUser" result
    result' <- cstringToText result
    freeMem result
    freeMem maybeScheme
    freeMem maybeUser
    freeMem maybePassword
    freeMem maybeAuthParams
    freeMem maybeHost
    freeMem path'
    freeMem maybeQuery
    freeMem maybeFragment
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::list_extract_uris
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "uri_list"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an URI list" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TCArray True (-1) (-1) (TBasicType TUTF8))
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_list_extract_uris" g_uri_list_extract_uris :: 
    CString ->                              -- uri_list : TBasicType TUTF8
    IO (Ptr CString)

-- | Splits an URI list conforming to the text\/uri-list
-- mime type defined in RFC 2483 into individual URIs,
-- discarding any comments. The URIs are not validated.
-- 
-- /Since: 2.6/
uriListExtractUris ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@uriList@/: an URI list
    -> m [T.Text]
    -- ^ __Returns:__ a newly allocated 'P.Nothing'-terminated list
    --   of strings holding the individual URIs. The array should be freed
    --   with 'GI.GLib.Functions.strfreev'.
uriListExtractUris uriList = liftIO $ do
    uriList' <- textToCString uriList
    result <- g_uri_list_extract_uris uriList'
    checkUnexpectedReturnNULL "uriListExtractUris" result
    result' <- unpackZeroTerminatedUTF8CArray result
    mapZeroTerminatedCArray freeMem result
    freeMem result
    freeMem uriList'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::parse
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "uri_string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string representing an absolute URI"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "UriFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags describing how to parse @uri_string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Uri" })
-- throws : True
-- Skip return : False

foreign import ccall "g_uri_parse" g_uri_parse :: 
    CString ->                              -- uri_string : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "UriFlags"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Uri)

-- | Parses /@uriString@/ according to /@flags@/. If the result is not a
-- valid <http://developer.gnome.org/glib/stable/#relative-and-absolute-uris absolute URI>, it will be discarded, and an
-- error returned.
-- 
-- /Since: 2.66/
uriParse ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@uriString@/: a string representing an absolute URI
    -> [GLib.Flags.UriFlags]
    -- ^ /@flags@/: flags describing how to parse /@uriString@/
    -> m Uri
    -- ^ __Returns:__ a new t'GI.GLib.Structs.Uri.Uri', or NULL on error. /(Can throw 'Data.GI.Base.GError.GError')/
uriParse uriString flags = liftIO $ do
    uriString' <- textToCString uriString
    let flags' = gflagsToWord flags
    onException (do
        result <- propagateGError $ g_uri_parse uriString' flags'
        checkUnexpectedReturnNULL "uriParse" result
        result' <- (wrapBoxed Uri) result
        freeMem uriString'
        return result'
     ) (do
        freeMem uriString'
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::parse_params
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "params"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a `%`-encoded string containing `attribute=value`\n  parameters"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the length of @params, or `-1` if it is nul-terminated"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "separators"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the separator byte character set between parameters. (usually\n  `&`, but sometimes `;` or both `&;`). Note that this function works on\n  bytes not characters, so it can't be used to delimit UTF-8 strings for\n  anything but ASCII characters. You may pass an empty set, in which case\n  no splitting will occur."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "UriParamsFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "flags to modify the way the parameters are handled."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TGHash (TBasicType TUTF8) (TBasicType TUTF8))
-- throws : True
-- Skip return : False

foreign import ccall "g_uri_parse_params" g_uri_parse_params :: 
    CString ->                              -- params : TBasicType TUTF8
    DI.Int64 ->                             -- length : TBasicType TSSize
    CString ->                              -- separators : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "UriParamsFlags"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr (GHashTable CString CString))

-- | Many URI schemes include one or more attribute\/value pairs as part of the URI
-- value. This method can be used to parse them into a hash table. When an
-- attribute has multiple occurrences, the last value is the final returned
-- value. If you need to handle repeated attributes differently, use
-- t'GI.GLib.Structs.UriParamsIter.UriParamsIter'.
-- 
-- The /@params@/ string is assumed to still be @%@-encoded, but the returned
-- values will be fully decoded. (Thus it is possible that the returned values
-- may contain @=@ or /@separators@/, if the value was encoded in the input.)
-- Invalid @%@-encoding is treated as with the 'GI.GLib.Flags.UriFlagsParseRelaxed'
-- rules for 'GI.GLib.Functions.uriParse'. (However, if /@params@/ is the path or query string
-- from a t'GI.GLib.Structs.Uri.Uri' that was parsed without 'GI.GLib.Flags.UriFlagsParseRelaxed' and
-- 'GI.GLib.Flags.UriFlagsEncoded', then you already know that it does not contain any
-- invalid encoding.)
-- 
-- 'GI.GLib.Flags.UriParamsFlagsWwwForm' is handled as documented for 'GI.GLib.Structs.UriParamsIter.uriParamsIterInit'.
-- 
-- If 'GI.GLib.Flags.UriParamsFlagsCaseInsensitive' is passed to /@flags@/, attributes will be
-- compared case-insensitively, so a params string @attr=123&Attr=456@ will only
-- return a single attribute–value pair, @Attr=456@. Case will be preserved in
-- the returned attributes.
-- 
-- If /@params@/ cannot be parsed (for example, it contains two /@separators@/
-- characters in a row), then /@error@/ is set and 'P.Nothing' is returned.
-- 
-- /Since: 2.66/
uriParseParams ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@params@/: a @%@-encoded string containing @attribute=value@
    --   parameters
    -> DI.Int64
    -- ^ /@length@/: the length of /@params@/, or @-1@ if it is nul-terminated
    -> T.Text
    -- ^ /@separators@/: the separator byte character set between parameters. (usually
    --   @&@, but sometimes @;@ or both @&;@). Note that this function works on
    --   bytes not characters, so it can\'t be used to delimit UTF-8 strings for
    --   anything but ASCII characters. You may pass an empty set, in which case
    --   no splitting will occur.
    -> [GLib.Flags.UriParamsFlags]
    -- ^ /@flags@/: flags to modify the way the parameters are handled.
    -> m (Map.Map T.Text T.Text)
    -- ^ __Returns:__ 
    --     A hash table of attribute\/value pairs, with both names and values
    --     fully-decoded; or 'P.Nothing' on error. /(Can throw 'Data.GI.Base.GError.GError')/
uriParseParams params length_ separators flags = liftIO $ do
    params' <- textToCString params
    separators' <- textToCString separators
    let flags' = gflagsToWord flags
    onException (do
        result <- propagateGError $ g_uri_parse_params params' length_ separators' flags'
        checkUnexpectedReturnNULL "uriParseParams" result
        result' <- unpackGHashTable result
        let result'' = mapFirst B.GHT.cstringUnpackPtr result'
        result''' <- mapFirstA cstringToText result''
        let result'''' = mapSecond B.GHT.cstringUnpackPtr result'''
        result''''' <- mapSecondA cstringToText result''''
        let result'''''' = Map.fromList result'''''
        unrefGHashTable result
        freeMem params'
        freeMem separators'
        return result''''''
     ) (do
        freeMem params'
        freeMem separators'
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::parse_scheme
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_parse_scheme" g_uri_parse_scheme :: 
    CString ->                              -- uri : TBasicType TUTF8
    IO CString

-- | Gets the scheme portion of a URI string.
-- <https://tools.ietf.org/html/rfc3986#section-3 RFC 3986> decodes the scheme
-- as:
-- 
-- 	
-- >
-- >URI = scheme ":" hier-part [ "?" query ] [ "#" fragment ]
-- 
-- Common schemes include @file@, @https@, @svn+ssh@, etc.
-- 
-- /Since: 2.16/
uriParseScheme ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@uri@/: a valid URI.
    -> m (Maybe T.Text)
    -- ^ __Returns:__ The ‘scheme’ component of the URI, or
    --     'P.Nothing' on error. The returned string should be freed when no longer needed.
uriParseScheme uri = liftIO $ do
    uri' <- textToCString uri
    result <- g_uri_parse_scheme uri'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        freeMem result'
        return result''
    freeMem uri'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::peek_scheme
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_peek_scheme" g_uri_peek_scheme :: 
    CString ->                              -- uri : TBasicType TUTF8
    IO CString

-- | Gets the scheme portion of a URI string.
-- <https://tools.ietf.org/html/rfc3986#section-3 RFC 3986> decodes the scheme
-- as:
-- 
-- 	
-- >
-- >URI = scheme ":" hier-part [ "?" query ] [ "#" fragment ]
-- 
-- Common schemes include @file@, @https@, @svn+ssh@, etc.
-- 
-- Unlike 'GI.GLib.Functions.uriParseScheme', the returned scheme is normalized to
-- all-lowercase and does not need to be freed.
-- 
-- /Since: 2.66/
uriPeekScheme ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@uri@/: a valid URI.
    -> m (Maybe T.Text)
    -- ^ __Returns:__ The ‘scheme’ component of the URI, or
    --     'P.Nothing' on error. The returned string is normalized to all-lowercase, and
    --     interned via 'GI.GLib.Functions.internString', so it does not need to be freed.
uriPeekScheme uri = liftIO $ do
    uri' <- textToCString uri
    result <- g_uri_peek_scheme uri'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    freeMem uri'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::resolve_relative
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "base_uri_string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string representing a base URI"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri_ref"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a string representing a relative or absolute URI"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "UriFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags describing how to parse @uri_ref"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : True
-- Skip return : False

foreign import ccall "g_uri_resolve_relative" g_uri_resolve_relative :: 
    CString ->                              -- base_uri_string : TBasicType TUTF8
    CString ->                              -- uri_ref : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "UriFlags"})
    Ptr (Ptr GError) ->                     -- error
    IO CString

-- | Parses /@uriRef@/ according to /@flags@/ and, if it is a
-- <http://developer.gnome.org/glib/stable/#relative-and-absolute-uris relative URI>, resolves it relative to
-- /@baseUriString@/. If the result is not a valid absolute URI, it will be
-- discarded, and an error returned.
-- 
-- (If /@baseUriString@/ is 'P.Nothing', this just returns /@uriRef@/, or
-- 'P.Nothing' if /@uriRef@/ is invalid or not absolute.)
-- 
-- /Since: 2.66/
uriResolveRelative ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (T.Text)
    -- ^ /@baseUriString@/: a string representing a base URI
    -> T.Text
    -- ^ /@uriRef@/: a string representing a relative or absolute URI
    -> [GLib.Flags.UriFlags]
    -- ^ /@flags@/: flags describing how to parse /@uriRef@/
    -> m T.Text
    -- ^ __Returns:__ the resolved URI string,
    -- or NULL on error. /(Can throw 'Data.GI.Base.GError.GError')/
uriResolveRelative baseUriString uriRef flags = liftIO $ do
    maybeBaseUriString <- case baseUriString of
        Nothing -> return FP.nullPtr
        Just jBaseUriString -> do
            jBaseUriString' <- textToCString jBaseUriString
            return jBaseUriString'
    uriRef' <- textToCString uriRef
    let flags' = gflagsToWord flags
    onException (do
        result <- propagateGError $ g_uri_resolve_relative maybeBaseUriString uriRef' flags'
        checkUnexpectedReturnNULL "uriResolveRelative" result
        result' <- cstringToText result
        freeMem result
        freeMem maybeBaseUriString
        freeMem uriRef'
        return result'
     ) (do
        freeMem maybeBaseUriString
        freeMem uriRef'
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::split
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "uri_ref"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a string containing a relative or absolute URI"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "UriFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags for parsing @uri_ref"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "scheme"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "on return, contains\n   the scheme (converted to lowercase), or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "userinfo"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "on return, contains\n   the userinfo, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "host"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "on return, contains the\n   host, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "port"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "on return, contains the\n   port, or `-1`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "on return, contains the\n   path"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "query"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "on return, contains the\n   query, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "fragment"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "on return, contains\n   the fragment, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : True

foreign import ccall "g_uri_split" g_uri_split :: 
    CString ->                              -- uri_ref : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "UriFlags"})
    Ptr CString ->                          -- scheme : TBasicType TUTF8
    Ptr CString ->                          -- userinfo : TBasicType TUTF8
    Ptr CString ->                          -- host : TBasicType TUTF8
    Ptr Int32 ->                            -- port : TBasicType TInt
    Ptr CString ->                          -- path : TBasicType TUTF8
    Ptr CString ->                          -- query : TBasicType TUTF8
    Ptr CString ->                          -- fragment : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Parses /@uriRef@/ (which can be an
-- <http://developer.gnome.org/glib/stable/#relative-and-absolute-uris absolute or relative URI>) according to /@flags@/, and
-- returns the pieces. Any component that doesn\'t appear in /@uriRef@/ will be
-- returned as 'P.Nothing' (but note that all URIs always have a path component,
-- though it may be the empty string).
-- 
-- If /@flags@/ contains 'GI.GLib.Flags.UriFlagsEncoded', then @%@-encoded characters in
-- /@uriRef@/ will remain encoded in the output strings. (If not,
-- then all such characters will be decoded.) Note that decoding will
-- only work if the URI components are ASCII or UTF-8, so you will
-- need to use 'GI.GLib.Flags.UriFlagsEncoded' if they are not.
-- 
-- Note that the 'GI.GLib.Flags.UriFlagsHasPassword' and
-- 'GI.GLib.Flags.UriFlagsHasAuthParams' /@flags@/ are ignored by 'GI.GLib.Functions.uriSplit',
-- since it always returns only the full userinfo; use
-- 'GI.GLib.Functions.uriSplitWithUser' if you want it split up.
-- 
-- /Since: 2.66/
uriSplit ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@uriRef@/: a string containing a relative or absolute URI
    -> [GLib.Flags.UriFlags]
    -- ^ /@flags@/: flags for parsing /@uriRef@/
    -> m ((Maybe T.Text, Maybe T.Text, Maybe T.Text, Int32, T.Text, Maybe T.Text, Maybe T.Text))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
uriSplit uriRef flags = liftIO $ do
    uriRef' <- textToCString uriRef
    let flags' = gflagsToWord flags
    scheme <- callocMem :: IO (Ptr CString)
    userinfo <- callocMem :: IO (Ptr CString)
    host <- callocMem :: IO (Ptr CString)
    port <- allocMem :: IO (Ptr Int32)
    path <- callocMem :: IO (Ptr CString)
    query <- callocMem :: IO (Ptr CString)
    fragment <- callocMem :: IO (Ptr CString)
    onException (do
        _ <- propagateGError $ g_uri_split uriRef' flags' scheme userinfo host port path query fragment
        scheme' <- peek scheme
        maybeScheme' <- convertIfNonNull scheme' $ \scheme'' -> do
            scheme''' <- cstringToText scheme''
            return scheme'''
        freeMem scheme'
        userinfo' <- peek userinfo
        maybeUserinfo' <- convertIfNonNull userinfo' $ \userinfo'' -> do
            userinfo''' <- cstringToText userinfo''
            return userinfo'''
        freeMem userinfo'
        host' <- peek host
        maybeHost' <- convertIfNonNull host' $ \host'' -> do
            host''' <- cstringToText host''
            return host'''
        freeMem host'
        port' <- peek port
        path' <- peek path
        path'' <- cstringToText path'
        freeMem path'
        query' <- peek query
        maybeQuery' <- convertIfNonNull query' $ \query'' -> do
            query''' <- cstringToText query''
            return query'''
        freeMem query'
        fragment' <- peek fragment
        maybeFragment' <- convertIfNonNull fragment' $ \fragment'' -> do
            fragment''' <- cstringToText fragment''
            return fragment'''
        freeMem fragment'
        freeMem uriRef'
        freeMem scheme
        freeMem userinfo
        freeMem host
        freeMem port
        freeMem path
        freeMem query
        freeMem fragment
        return (maybeScheme', maybeUserinfo', maybeHost', port', path'', maybeQuery', maybeFragment')
     ) (do
        freeMem uriRef'
        freeMem scheme
        freeMem userinfo
        freeMem host
        freeMem port
        freeMem path
        freeMem query
        freeMem fragment
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::split_network
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "uri_string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string containing an absolute URI"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "UriFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags for parsing @uri_string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "scheme"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "on return, contains\n   the scheme (converted to lowercase), or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "host"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "on return, contains the\n   host, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "port"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "on return, contains the\n   port, or `-1`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : True

foreign import ccall "g_uri_split_network" g_uri_split_network :: 
    CString ->                              -- uri_string : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "UriFlags"})
    Ptr CString ->                          -- scheme : TBasicType TUTF8
    Ptr CString ->                          -- host : TBasicType TUTF8
    Ptr Int32 ->                            -- port : TBasicType TInt
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Parses /@uriString@/ (which must be an <http://developer.gnome.org/glib/stable/#relative-and-absolute-uris absolute URI>)
-- according to /@flags@/, and returns the pieces relevant to connecting to a host.
-- See the documentation for 'GI.GLib.Functions.uriSplit' for more details; this is
-- mostly a wrapper around that function with simpler arguments.
-- However, it will return an error if /@uriString@/ is a relative URI,
-- or does not contain a hostname component.
-- 
-- /Since: 2.66/
uriSplitNetwork ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@uriString@/: a string containing an absolute URI
    -> [GLib.Flags.UriFlags]
    -- ^ /@flags@/: flags for parsing /@uriString@/
    -> m ((Maybe T.Text, Maybe T.Text, Int32))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
uriSplitNetwork uriString flags = liftIO $ do
    uriString' <- textToCString uriString
    let flags' = gflagsToWord flags
    scheme <- callocMem :: IO (Ptr CString)
    host <- callocMem :: IO (Ptr CString)
    port <- allocMem :: IO (Ptr Int32)
    onException (do
        _ <- propagateGError $ g_uri_split_network uriString' flags' scheme host port
        scheme' <- peek scheme
        maybeScheme' <- convertIfNonNull scheme' $ \scheme'' -> do
            scheme''' <- cstringToText scheme''
            return scheme'''
        freeMem scheme'
        host' <- peek host
        maybeHost' <- convertIfNonNull host' $ \host'' -> do
            host''' <- cstringToText host''
            return host'''
        freeMem host'
        port' <- peek port
        freeMem uriString'
        freeMem scheme
        freeMem host
        freeMem port
        return (maybeScheme', maybeHost', port')
     ) (do
        freeMem uriString'
        freeMem scheme
        freeMem host
        freeMem port
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::split_with_user
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "uri_ref"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a string containing a relative or absolute URI"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "UriFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags for parsing @uri_ref"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "scheme"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "on return, contains\n   the scheme (converted to lowercase), or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "user"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "on return, contains\n   the user, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "password"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "on return, contains\n   the password, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "auth_params"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "on return, contains\n   the auth_params, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "host"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "on return, contains the\n   host, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "port"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "on return, contains the\n   port, or `-1`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "on return, contains the\n   path"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "query"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "on return, contains the\n   query, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "fragment"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "on return, contains\n   the fragment, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : True

foreign import ccall "g_uri_split_with_user" g_uri_split_with_user :: 
    CString ->                              -- uri_ref : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "GLib", name = "UriFlags"})
    Ptr CString ->                          -- scheme : TBasicType TUTF8
    Ptr CString ->                          -- user : TBasicType TUTF8
    Ptr CString ->                          -- password : TBasicType TUTF8
    Ptr CString ->                          -- auth_params : TBasicType TUTF8
    Ptr CString ->                          -- host : TBasicType TUTF8
    Ptr Int32 ->                            -- port : TBasicType TInt
    Ptr CString ->                          -- path : TBasicType TUTF8
    Ptr CString ->                          -- query : TBasicType TUTF8
    Ptr CString ->                          -- fragment : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Parses /@uriRef@/ (which can be an
-- <http://developer.gnome.org/glib/stable/#relative-and-absolute-uris absolute or relative URI>) according to /@flags@/, and
-- returns the pieces. Any component that doesn\'t appear in /@uriRef@/ will be
-- returned as 'P.Nothing' (but note that all URIs always have a path component,
-- though it may be the empty string).
-- 
-- See 'GI.GLib.Functions.uriSplit', and the definition of t'GI.GLib.Flags.UriFlags', for more
-- information on the effect of /@flags@/. Note that /@password@/ will only
-- be parsed out if /@flags@/ contains 'GI.GLib.Flags.UriFlagsHasPassword', and
-- /@authParams@/ will only be parsed out if /@flags@/ contains
-- 'GI.GLib.Flags.UriFlagsHasAuthParams'.
-- 
-- /Since: 2.66/
uriSplitWithUser ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@uriRef@/: a string containing a relative or absolute URI
    -> [GLib.Flags.UriFlags]
    -- ^ /@flags@/: flags for parsing /@uriRef@/
    -> m ((Maybe T.Text, Maybe T.Text, Maybe T.Text, Maybe T.Text, Maybe T.Text, Int32, T.Text, Maybe T.Text, Maybe T.Text))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
uriSplitWithUser uriRef flags = liftIO $ do
    uriRef' <- textToCString uriRef
    let flags' = gflagsToWord flags
    scheme <- callocMem :: IO (Ptr CString)
    user <- callocMem :: IO (Ptr CString)
    password <- callocMem :: IO (Ptr CString)
    authParams <- callocMem :: IO (Ptr CString)
    host <- callocMem :: IO (Ptr CString)
    port <- allocMem :: IO (Ptr Int32)
    path <- callocMem :: IO (Ptr CString)
    query <- callocMem :: IO (Ptr CString)
    fragment <- callocMem :: IO (Ptr CString)
    onException (do
        _ <- propagateGError $ g_uri_split_with_user uriRef' flags' scheme user password authParams host port path query fragment
        scheme' <- peek scheme
        maybeScheme' <- convertIfNonNull scheme' $ \scheme'' -> do
            scheme''' <- cstringToText scheme''
            return scheme'''
        freeMem scheme'
        user' <- peek user
        maybeUser' <- convertIfNonNull user' $ \user'' -> do
            user''' <- cstringToText user''
            return user'''
        freeMem user'
        password' <- peek password
        maybePassword' <- convertIfNonNull password' $ \password'' -> do
            password''' <- cstringToText password''
            return password'''
        freeMem password'
        authParams' <- peek authParams
        maybeAuthParams' <- convertIfNonNull authParams' $ \authParams'' -> do
            authParams''' <- cstringToText authParams''
            return authParams'''
        freeMem authParams'
        host' <- peek host
        maybeHost' <- convertIfNonNull host' $ \host'' -> do
            host''' <- cstringToText host''
            return host'''
        freeMem host'
        port' <- peek port
        path' <- peek path
        path'' <- cstringToText path'
        freeMem path'
        query' <- peek query
        maybeQuery' <- convertIfNonNull query' $ \query'' -> do
            query''' <- cstringToText query''
            return query'''
        freeMem query'
        fragment' <- peek fragment
        maybeFragment' <- convertIfNonNull fragment' $ \fragment'' -> do
            fragment''' <- cstringToText fragment''
            return fragment'''
        freeMem fragment'
        freeMem uriRef'
        freeMem scheme
        freeMem user
        freeMem password
        freeMem authParams
        freeMem host
        freeMem port
        freeMem path
        freeMem query
        freeMem fragment
        return (maybeScheme', maybeUser', maybePassword', maybeAuthParams', maybeHost', port', path'', maybeQuery', maybeFragment')
     ) (do
        freeMem uriRef'
        freeMem scheme
        freeMem user
        freeMem password
        freeMem authParams
        freeMem host
        freeMem port
        freeMem path
        freeMem query
        freeMem fragment
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::unescape_bytes
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "escaped_string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A URI-escaped string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length (in bytes) of @escaped_string to escape, or `-1` if it\n  is nul-terminated."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "illegal_characters"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a string of illegal characters\n  not to be allowed, or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Bytes" })
-- throws : True
-- Skip return : False

foreign import ccall "g_uri_unescape_bytes" g_uri_unescape_bytes :: 
    CString ->                              -- escaped_string : TBasicType TUTF8
    DI.Int64 ->                             -- length : TBasicType TSSize
    CString ->                              -- illegal_characters : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr GLib.Bytes.Bytes)

-- | Unescapes a segment of an escaped string as binary data.
-- 
-- Note that in contrast to 'GI.GLib.Functions.uriUnescapeString', this does allow
-- nul bytes to appear in the output.
-- 
-- If any of the characters in /@illegalCharacters@/ appears as an escaped
-- character in /@escapedString@/, then that is an error and 'P.Nothing' will be
-- returned. This is useful if you want to avoid for instance having a slash
-- being expanded in an escaped path element, which might confuse pathname
-- handling.
-- 
-- /Since: 2.66/
uriUnescapeBytes ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@escapedString@/: A URI-escaped string
    -> DI.Int64
    -- ^ /@length@/: the length (in bytes) of /@escapedString@/ to escape, or @-1@ if it
    --   is nul-terminated.
    -> Maybe (T.Text)
    -- ^ /@illegalCharacters@/: a string of illegal characters
    --   not to be allowed, or 'P.Nothing'.
    -> m GLib.Bytes.Bytes
    -- ^ __Returns:__ an unescaped version of /@escapedString@/
    --     or 'P.Nothing' on error (if decoding failed, using 'GI.GLib.Enums.UriErrorFailed' error
    --     code). The returned t'GI.GLib.Structs.Bytes.Bytes' should be unreffed when no longer needed. /(Can throw 'Data.GI.Base.GError.GError')/
uriUnescapeBytes escapedString length_ illegalCharacters = liftIO $ do
    escapedString' <- textToCString escapedString
    maybeIllegalCharacters <- case illegalCharacters of
        Nothing -> return FP.nullPtr
        Just jIllegalCharacters -> do
            jIllegalCharacters' <- textToCString jIllegalCharacters
            return jIllegalCharacters'
    onException (do
        result <- propagateGError $ g_uri_unescape_bytes escapedString' length_ maybeIllegalCharacters
        checkUnexpectedReturnNULL "uriUnescapeBytes" result
        result' <- (wrapBoxed GLib.Bytes.Bytes) result
        freeMem escapedString'
        freeMem maybeIllegalCharacters
        return result'
     ) (do
        freeMem escapedString'
        freeMem maybeIllegalCharacters
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::unescape_segment
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "escaped_string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A string, may be %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "escaped_string_end"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "Pointer to end of @escaped_string,\n  may be %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "illegal_characters"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "An optional string of illegal\n  characters not to be allowed, may be %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_unescape_segment" g_uri_unescape_segment :: 
    CString ->                              -- escaped_string : TBasicType TUTF8
    CString ->                              -- escaped_string_end : TBasicType TUTF8
    CString ->                              -- illegal_characters : TBasicType TUTF8
    IO CString

-- | Unescapes a segment of an escaped string.
-- 
-- If any of the characters in /@illegalCharacters@/ or the NUL
-- character appears as an escaped character in /@escapedString@/, then
-- that is an error and 'P.Nothing' will be returned. This is useful if you
-- want to avoid for instance having a slash being expanded in an
-- escaped path element, which might confuse pathname handling.
-- 
-- Note: @NUL@ byte is not accepted in the output, in contrast to
-- 'GI.GLib.Functions.uriUnescapeBytes'.
-- 
-- /Since: 2.16/
uriUnescapeSegment ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (T.Text)
    -- ^ /@escapedString@/: A string, may be 'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@escapedStringEnd@/: Pointer to end of /@escapedString@/,
    --   may be 'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@illegalCharacters@/: An optional string of illegal
    --   characters not to be allowed, may be 'P.Nothing'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ an unescaped version of /@escapedString@/,
    -- or 'P.Nothing' on error. The returned string should be freed when no longer
    -- needed.  As a special case if 'P.Nothing' is given for /@escapedString@/, this
    -- function will return 'P.Nothing'.
uriUnescapeSegment escapedString escapedStringEnd illegalCharacters = liftIO $ do
    maybeEscapedString <- case escapedString of
        Nothing -> return FP.nullPtr
        Just jEscapedString -> do
            jEscapedString' <- textToCString jEscapedString
            return jEscapedString'
    maybeEscapedStringEnd <- case escapedStringEnd of
        Nothing -> return FP.nullPtr
        Just jEscapedStringEnd -> do
            jEscapedStringEnd' <- textToCString jEscapedStringEnd
            return jEscapedStringEnd'
    maybeIllegalCharacters <- case illegalCharacters of
        Nothing -> return FP.nullPtr
        Just jIllegalCharacters -> do
            jIllegalCharacters' <- textToCString jIllegalCharacters
            return jIllegalCharacters'
    result <- g_uri_unescape_segment maybeEscapedString maybeEscapedStringEnd maybeIllegalCharacters
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        freeMem result'
        return result''
    freeMem maybeEscapedString
    freeMem maybeEscapedStringEnd
    freeMem maybeIllegalCharacters
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method Uri::unescape_string
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "escaped_string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an escaped string to be unescaped."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "illegal_characters"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a string of illegal characters\n  not to be allowed, or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_uri_unescape_string" g_uri_unescape_string :: 
    CString ->                              -- escaped_string : TBasicType TUTF8
    CString ->                              -- illegal_characters : TBasicType TUTF8
    IO CString

-- | Unescapes a whole escaped string.
-- 
-- If any of the characters in /@illegalCharacters@/ or the NUL
-- character appears as an escaped character in /@escapedString@/, then
-- that is an error and 'P.Nothing' will be returned. This is useful if you
-- want to avoid for instance having a slash being expanded in an
-- escaped path element, which might confuse pathname handling.
-- 
-- /Since: 2.16/
uriUnescapeString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@escapedString@/: an escaped string to be unescaped.
    -> Maybe (T.Text)
    -- ^ /@illegalCharacters@/: a string of illegal characters
    --   not to be allowed, or 'P.Nothing'.
    -> m (Maybe T.Text)
    -- ^ __Returns:__ an unescaped version of /@escapedString@/.
    -- The returned string should be freed when no longer needed.
uriUnescapeString escapedString illegalCharacters = liftIO $ do
    escapedString' <- textToCString escapedString
    maybeIllegalCharacters <- case illegalCharacters of
        Nothing -> return FP.nullPtr
        Just jIllegalCharacters -> do
            jIllegalCharacters' <- textToCString jIllegalCharacters
            return jIllegalCharacters'
    result <- g_uri_unescape_string escapedString' maybeIllegalCharacters
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        freeMem result'
        return result''
    freeMem escapedString'
    freeMem maybeIllegalCharacters
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveUriMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveUriMethod "parseRelative" o = UriParseRelativeMethodInfo
    ResolveUriMethod "toString" o = UriToStringMethodInfo
    ResolveUriMethod "toStringPartial" o = UriToStringPartialMethodInfo
    ResolveUriMethod "getAuthParams" o = UriGetAuthParamsMethodInfo
    ResolveUriMethod "getFlags" o = UriGetFlagsMethodInfo
    ResolveUriMethod "getFragment" o = UriGetFragmentMethodInfo
    ResolveUriMethod "getHost" o = UriGetHostMethodInfo
    ResolveUriMethod "getPassword" o = UriGetPasswordMethodInfo
    ResolveUriMethod "getPath" o = UriGetPathMethodInfo
    ResolveUriMethod "getPort" o = UriGetPortMethodInfo
    ResolveUriMethod "getQuery" o = UriGetQueryMethodInfo
    ResolveUriMethod "getScheme" o = UriGetSchemeMethodInfo
    ResolveUriMethod "getUser" o = UriGetUserMethodInfo
    ResolveUriMethod "getUserinfo" o = UriGetUserinfoMethodInfo
    ResolveUriMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveUriMethod t Uri, O.OverloadedMethod info Uri p) => OL.IsLabel t (Uri -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveUriMethod t Uri, O.OverloadedMethod info Uri p, R.HasField t Uri p) => R.HasField t Uri p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveUriMethod t Uri, O.OverloadedMethodInfo info Uri) => OL.IsLabel t (O.MethodProxy info Uri) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


