{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- A @PangoAttrList@ represents a list of attributes that apply to a section
-- of text.
-- 
-- The attributes in a @PangoAttrList@ are, in general, allowed to overlap in
-- an arbitrary fashion. However, if the attributes are manipulated only through
-- 'GI.Pango.Structs.AttrList.attrListChange', the overlap between properties will meet
-- stricter criteria.
-- 
-- Since the @PangoAttrList@ structure is stored as a linear list, it is not
-- suitable for storing attributes for large amounts of text. In general, you
-- should not use a single @PangoAttrList@ for more than one paragraph of text.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Pango.Structs.AttrList
    ( 

-- * Exported types
    AttrList(..)                            ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [change]("GI.Pango.Structs.AttrList#g:method:change"), [copy]("GI.Pango.Structs.AttrList#g:method:copy"), [equal]("GI.Pango.Structs.AttrList#g:method:equal"), [filter]("GI.Pango.Structs.AttrList#g:method:filter"), [insert]("GI.Pango.Structs.AttrList#g:method:insert"), [insertBefore]("GI.Pango.Structs.AttrList#g:method:insertBefore"), [ref]("GI.Pango.Structs.AttrList#g:method:ref"), [splice]("GI.Pango.Structs.AttrList#g:method:splice"), [toString]("GI.Pango.Structs.AttrList#g:method:toString"), [unref]("GI.Pango.Structs.AttrList#g:method:unref"), [update]("GI.Pango.Structs.AttrList#g:method:update").
-- 
-- ==== Getters
-- [getAttributes]("GI.Pango.Structs.AttrList#g:method:getAttributes"), [getIterator]("GI.Pango.Structs.AttrList#g:method:getIterator").
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveAttrListMethod                   ,
#endif

-- ** change #method:change#

#if defined(ENABLE_OVERLOADING)
    AttrListChangeMethodInfo                ,
#endif
    attrListChange                          ,


-- ** copy #method:copy#

#if defined(ENABLE_OVERLOADING)
    AttrListCopyMethodInfo                  ,
#endif
    attrListCopy                            ,


-- ** equal #method:equal#

#if defined(ENABLE_OVERLOADING)
    AttrListEqualMethodInfo                 ,
#endif
    attrListEqual                           ,


-- ** filter #method:filter#

#if defined(ENABLE_OVERLOADING)
    AttrListFilterMethodInfo                ,
#endif
    attrListFilter                          ,


-- ** fromString #method:fromString#

    attrListFromString                      ,


-- ** getAttributes #method:getAttributes#

#if defined(ENABLE_OVERLOADING)
    AttrListGetAttributesMethodInfo         ,
#endif
    attrListGetAttributes                   ,


-- ** getIterator #method:getIterator#

#if defined(ENABLE_OVERLOADING)
    AttrListGetIteratorMethodInfo           ,
#endif
    attrListGetIterator                     ,


-- ** insert #method:insert#

#if defined(ENABLE_OVERLOADING)
    AttrListInsertMethodInfo                ,
#endif
    attrListInsert                          ,


-- ** insertBefore #method:insertBefore#

#if defined(ENABLE_OVERLOADING)
    AttrListInsertBeforeMethodInfo          ,
#endif
    attrListInsertBefore                    ,


-- ** new #method:new#

    attrListNew                             ,


-- ** ref #method:ref#

#if defined(ENABLE_OVERLOADING)
    AttrListRefMethodInfo                   ,
#endif
    attrListRef                             ,


-- ** splice #method:splice#

#if defined(ENABLE_OVERLOADING)
    AttrListSpliceMethodInfo                ,
#endif
    attrListSplice                          ,


-- ** toString #method:toString#

#if defined(ENABLE_OVERLOADING)
    AttrListToStringMethodInfo              ,
#endif
    attrListToString                        ,


-- ** unref #method:unref#

#if defined(ENABLE_OVERLOADING)
    AttrListUnrefMethodInfo                 ,
#endif
    attrListUnref                           ,


-- ** update #method:update#

#if defined(ENABLE_OVERLOADING)
    AttrListUpdateMethodInfo                ,
#endif
    attrListUpdate                          ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.Pango.Callbacks as Pango.Callbacks
import {-# SOURCE #-} qualified GI.Pango.Enums as Pango.Enums
import {-# SOURCE #-} qualified GI.Pango.Flags as Pango.Flags
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrClass as Pango.AttrClass
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrColor as Pango.AttrColor
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrFloat as Pango.AttrFloat
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrFontDesc as Pango.AttrFontDesc
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrFontFeatures as Pango.AttrFontFeatures
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrInt as Pango.AttrInt
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrIterator as Pango.AttrIterator
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrLanguage as Pango.AttrLanguage
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrShape as Pango.AttrShape
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrSize as Pango.AttrSize
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrString as Pango.AttrString
import {-# SOURCE #-} qualified GI.Pango.Structs.Attribute as Pango.Attribute
import {-# SOURCE #-} qualified GI.Pango.Structs.Color as Pango.Color
import {-# SOURCE #-} qualified GI.Pango.Structs.FontDescription as Pango.FontDescription
import {-# SOURCE #-} qualified GI.Pango.Structs.Language as Pango.Language
import {-# SOURCE #-} qualified GI.Pango.Structs.Rectangle as Pango.Rectangle

#else
import qualified GI.Pango.Callbacks as Pango.Callbacks
import {-# SOURCE #-} qualified GI.Pango.Structs.AttrIterator as Pango.AttrIterator
import {-# SOURCE #-} qualified GI.Pango.Structs.Attribute as Pango.Attribute

#endif

-- | Memory-managed wrapper type.
newtype AttrList = AttrList (SP.ManagedPtr AttrList)
    deriving (Eq)

instance SP.ManagedPtrNewtype AttrList where
    toManagedPtr (AttrList p) = p

foreign import ccall "pango_attr_list_get_type" c_pango_attr_list_get_type :: 
    IO GType

type instance O.ParentTypes AttrList = '[]
instance O.HasParentTypes AttrList

instance B.Types.TypedObject AttrList where
    glibType = c_pango_attr_list_get_type

instance B.Types.GBoxed AttrList

-- | Convert t'AttrList' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe AttrList) where
    gvalueGType_ = c_pango_attr_list_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr AttrList)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr AttrList)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed AttrList ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList AttrList
type instance O.AttributeList AttrList = AttrListAttributeList
type AttrListAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method AttrList::new
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "AttrList" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_list_new" pango_attr_list_new :: 
    IO (Ptr AttrList)

-- | Create a new empty attribute list with a reference
-- count of one.
attrListNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m AttrList
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttrList@, which should be freed with
    --   'GI.Pango.Structs.AttrList.attrListUnref'
attrListNew  = liftIO $ do
    result <- pango_attr_list_new
    checkUnexpectedReturnNULL "attrListNew" result
    result' <- (wrapBoxed AttrList) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method AttrList::change
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoAttrList`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attr"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Attribute" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the attribute to insert"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_list_change" pango_attr_list_change :: 
    Ptr AttrList ->                         -- list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    Ptr Pango.Attribute.Attribute ->        -- attr : TInterface (Name {namespace = "Pango", name = "Attribute"})
    IO ()

-- | Insert the given attribute into the @PangoAttrList@.
-- 
-- It will replace any attributes of the same type
-- on that segment and be merged with any adjoining
-- attributes that are identical.
-- 
-- This function is slower than 'GI.Pango.Structs.AttrList.attrListInsert'
-- for creating an attribute list in order (potentially
-- much slower for large lists). However,
-- 'GI.Pango.Structs.AttrList.attrListInsert' is not suitable for
-- continually changing a set of attributes since it
-- never removes or combines existing attributes.
attrListChange ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    AttrList
    -- ^ /@list@/: a @PangoAttrList@
    -> Pango.Attribute.Attribute
    -- ^ /@attr@/: the attribute to insert
    -> m ()
attrListChange list attr = liftIO $ do
    list' <- unsafeManagedPtrGetPtr list
    attr' <- B.ManagedPtr.disownBoxed attr
    pango_attr_list_change list' attr'
    touchManagedPtr list
    touchManagedPtr attr
    return ()

#if defined(ENABLE_OVERLOADING)
data AttrListChangeMethodInfo
instance (signature ~ (Pango.Attribute.Attribute -> m ()), MonadIO m) => O.OverloadedMethod AttrListChangeMethodInfo AttrList signature where
    overloadedMethod = attrListChange

instance O.OverloadedMethodInfo AttrListChangeMethodInfo AttrList where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.AttrList.attrListChange",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-AttrList.html#v:attrListChange"
        })


#endif

-- method AttrList::copy
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoAttrList`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "AttrList" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_list_copy" pango_attr_list_copy :: 
    Ptr AttrList ->                         -- list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    IO (Ptr AttrList)

-- | Copy /@list@/ and return an identical new list.
attrListCopy ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (AttrList)
    -- ^ /@list@/: a @PangoAttrList@
    -> m (Maybe AttrList)
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttrList@, with a reference count of one,
    --   which should be freed with 'GI.Pango.Structs.AttrList.attrListUnref'.
    --   Returns 'P.Nothing' if /@list@/ was 'P.Nothing'.
attrListCopy list = liftIO $ do
    maybeList <- case list of
        Nothing -> return FP.nullPtr
        Just jList -> do
            jList' <- unsafeManagedPtrGetPtr jList
            return jList'
    result <- pango_attr_list_copy maybeList
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed AttrList) result'
        return result''
    whenJust list touchManagedPtr
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data AttrListCopyMethodInfo
instance (signature ~ (m (Maybe AttrList)), MonadIO m) => O.OverloadedMethod AttrListCopyMethodInfo AttrList signature where
    overloadedMethod i = attrListCopy (Just i)

instance O.OverloadedMethodInfo AttrListCopyMethodInfo AttrList where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.AttrList.attrListCopy",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-AttrList.html#v:attrListCopy"
        })


#endif

-- method AttrList::equal
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoAttrList`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "other_list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the other `PangoAttrList`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_list_equal" pango_attr_list_equal :: 
    Ptr AttrList ->                         -- list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    Ptr AttrList ->                         -- other_list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    IO CInt

-- | Checks whether /@list@/ and /@otherList@/ contain the same
-- attributes and whether those attributes apply to the
-- same ranges.
-- 
-- Beware that this will return wrong values if any list
-- contains duplicates.
-- 
-- /Since: 1.46/
attrListEqual ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    AttrList
    -- ^ /@list@/: a @PangoAttrList@
    -> AttrList
    -- ^ /@otherList@/: the other @PangoAttrList@
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the lists are equal, 'P.False' if
    --   they aren\'t
attrListEqual list otherList = liftIO $ do
    list' <- unsafeManagedPtrGetPtr list
    otherList' <- unsafeManagedPtrGetPtr otherList
    result <- pango_attr_list_equal list' otherList'
    let result' = (/= 0) result
    touchManagedPtr list
    touchManagedPtr otherList
    return result'

#if defined(ENABLE_OVERLOADING)
data AttrListEqualMethodInfo
instance (signature ~ (AttrList -> m Bool), MonadIO m) => O.OverloadedMethod AttrListEqualMethodInfo AttrList signature where
    overloadedMethod = attrListEqual

instance O.OverloadedMethodInfo AttrListEqualMethodInfo AttrList where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.AttrList.attrListEqual",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-AttrList.html#v:attrListEqual"
        })


#endif

-- method AttrList::filter
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoAttrList`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "func"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrFilterFunc" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "callback function;\n  returns %TRUE if an attribute should be filtered out"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeCall
--           , argClosure = 2
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Data to be passed to @func"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "AttrList" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_list_filter" pango_attr_list_filter :: 
    Ptr AttrList ->                         -- list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    FunPtr Pango.Callbacks.C_AttrFilterFunc -> -- func : TInterface (Name {namespace = "Pango", name = "AttrFilterFunc"})
    Ptr () ->                               -- data : TBasicType TPtr
    IO (Ptr AttrList)

-- | Given a @PangoAttrList@ and callback function, removes
-- any elements of /@list@/ for which /@func@/ returns 'P.True' and
-- inserts them into a new list.
-- 
-- /Since: 1.2/
attrListFilter ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    AttrList
    -- ^ /@list@/: a @PangoAttrList@
    -> Pango.Callbacks.AttrFilterFunc
    -- ^ /@func@/: callback function;
    --   returns 'P.True' if an attribute should be filtered out
    -> m (Maybe AttrList)
    -- ^ __Returns:__ the new
    --   @PangoAttrList@ or 'P.Nothing' if no attributes of the
    --   given types were found
attrListFilter list func = liftIO $ do
    list' <- unsafeManagedPtrGetPtr list
    func' <- Pango.Callbacks.mk_AttrFilterFunc (Pango.Callbacks.wrap_AttrFilterFunc Nothing (Pango.Callbacks.drop_closures_AttrFilterFunc func))
    let data_ = nullPtr
    result <- pango_attr_list_filter list' func' data_
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed AttrList) result'
        return result''
    safeFreeFunPtr $ castFunPtrToPtr func'
    touchManagedPtr list
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data AttrListFilterMethodInfo
instance (signature ~ (Pango.Callbacks.AttrFilterFunc -> m (Maybe AttrList)), MonadIO m) => O.OverloadedMethod AttrListFilterMethodInfo AttrList signature where
    overloadedMethod = attrListFilter

instance O.OverloadedMethodInfo AttrListFilterMethodInfo AttrList where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.AttrList.attrListFilter",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-AttrList.html#v:attrListFilter"
        })


#endif

-- method AttrList::get_attributes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoAttrList`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TGSList
--                  (TInterface Name { namespace = "Pango" , name = "Attribute" }))
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_list_get_attributes" pango_attr_list_get_attributes :: 
    Ptr AttrList ->                         -- list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    IO (Ptr (GSList (Ptr Pango.Attribute.Attribute)))

-- | Gets a list of all attributes in /@list@/.
-- 
-- /Since: 1.44/
attrListGetAttributes ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    AttrList
    -- ^ /@list@/: a @PangoAttrList@
    -> m [Pango.Attribute.Attribute]
    -- ^ __Returns:__ 
    --   a list of all attributes in /@list@/. To free this value,
    --   call 'GI.Pango.Structs.Attribute.attributeDestroy' on each value and
    --   @/g_slist_free()/@ on the list.
attrListGetAttributes list = liftIO $ do
    list' <- unsafeManagedPtrGetPtr list
    result <- pango_attr_list_get_attributes list'
    result' <- unpackGSList result
    result'' <- mapM (wrapBoxed Pango.Attribute.Attribute) result'
    g_slist_free result
    touchManagedPtr list
    return result''

#if defined(ENABLE_OVERLOADING)
data AttrListGetAttributesMethodInfo
instance (signature ~ (m [Pango.Attribute.Attribute]), MonadIO m) => O.OverloadedMethod AttrListGetAttributesMethodInfo AttrList signature where
    overloadedMethod = attrListGetAttributes

instance O.OverloadedMethodInfo AttrListGetAttributesMethodInfo AttrList where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.AttrList.attrListGetAttributes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-AttrList.html#v:attrListGetAttributes"
        })


#endif

-- method AttrList::get_iterator
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoAttrList`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Pango" , name = "AttrIterator" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_list_get_iterator" pango_attr_list_get_iterator :: 
    Ptr AttrList ->                         -- list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    IO (Ptr Pango.AttrIterator.AttrIterator)

-- | Create a iterator initialized to the beginning of the list.
-- 
-- /@list@/ must not be modified until this iterator is freed.
attrListGetIterator ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    AttrList
    -- ^ /@list@/: a @PangoAttrList@
    -> m Pango.AttrIterator.AttrIterator
    -- ^ __Returns:__ the newly allocated
    --   @PangoAttrIterator@, which should be freed with
    --   'GI.Pango.Structs.AttrIterator.attrIteratorDestroy'
attrListGetIterator list = liftIO $ do
    list' <- unsafeManagedPtrGetPtr list
    result <- pango_attr_list_get_iterator list'
    checkUnexpectedReturnNULL "attrListGetIterator" result
    result' <- (wrapBoxed Pango.AttrIterator.AttrIterator) result
    touchManagedPtr list
    return result'

#if defined(ENABLE_OVERLOADING)
data AttrListGetIteratorMethodInfo
instance (signature ~ (m Pango.AttrIterator.AttrIterator), MonadIO m) => O.OverloadedMethod AttrListGetIteratorMethodInfo AttrList signature where
    overloadedMethod = attrListGetIterator

instance O.OverloadedMethodInfo AttrListGetIteratorMethodInfo AttrList where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.AttrList.attrListGetIterator",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-AttrList.html#v:attrListGetIterator"
        })


#endif

-- method AttrList::insert
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoAttrList`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attr"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Attribute" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the attribute to insert"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_list_insert" pango_attr_list_insert :: 
    Ptr AttrList ->                         -- list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    Ptr Pango.Attribute.Attribute ->        -- attr : TInterface (Name {namespace = "Pango", name = "Attribute"})
    IO ()

-- | Insert the given attribute into the @PangoAttrList@.
-- 
-- It will be inserted after all other attributes with a
-- matching /@startIndex@/.
attrListInsert ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    AttrList
    -- ^ /@list@/: a @PangoAttrList@
    -> Pango.Attribute.Attribute
    -- ^ /@attr@/: the attribute to insert
    -> m ()
attrListInsert list attr = liftIO $ do
    list' <- unsafeManagedPtrGetPtr list
    attr' <- B.ManagedPtr.disownBoxed attr
    pango_attr_list_insert list' attr'
    touchManagedPtr list
    touchManagedPtr attr
    return ()

#if defined(ENABLE_OVERLOADING)
data AttrListInsertMethodInfo
instance (signature ~ (Pango.Attribute.Attribute -> m ()), MonadIO m) => O.OverloadedMethod AttrListInsertMethodInfo AttrList signature where
    overloadedMethod = attrListInsert

instance O.OverloadedMethodInfo AttrListInsertMethodInfo AttrList where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.AttrList.attrListInsert",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-AttrList.html#v:attrListInsert"
        })


#endif

-- method AttrList::insert_before
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoAttrList`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attr"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "Attribute" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the attribute to insert"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_list_insert_before" pango_attr_list_insert_before :: 
    Ptr AttrList ->                         -- list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    Ptr Pango.Attribute.Attribute ->        -- attr : TInterface (Name {namespace = "Pango", name = "Attribute"})
    IO ()

-- | Insert the given attribute into the @PangoAttrList@.
-- 
-- It will be inserted before all other attributes with a
-- matching /@startIndex@/.
attrListInsertBefore ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    AttrList
    -- ^ /@list@/: a @PangoAttrList@
    -> Pango.Attribute.Attribute
    -- ^ /@attr@/: the attribute to insert
    -> m ()
attrListInsertBefore list attr = liftIO $ do
    list' <- unsafeManagedPtrGetPtr list
    attr' <- B.ManagedPtr.disownBoxed attr
    pango_attr_list_insert_before list' attr'
    touchManagedPtr list
    touchManagedPtr attr
    return ()

#if defined(ENABLE_OVERLOADING)
data AttrListInsertBeforeMethodInfo
instance (signature ~ (Pango.Attribute.Attribute -> m ()), MonadIO m) => O.OverloadedMethod AttrListInsertBeforeMethodInfo AttrList signature where
    overloadedMethod = attrListInsertBefore

instance O.OverloadedMethodInfo AttrListInsertBeforeMethodInfo AttrList where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.AttrList.attrListInsertBefore",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-AttrList.html#v:attrListInsertBefore"
        })


#endif

-- method AttrList::ref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoAttrList`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "AttrList" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_list_ref" pango_attr_list_ref :: 
    Ptr AttrList ->                         -- list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    IO (Ptr AttrList)

-- | Increase the reference count of the given attribute
-- list by one.
-- 
-- /Since: 1.10/
attrListRef ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (AttrList)
    -- ^ /@list@/: a @PangoAttrList@
    -> m AttrList
    -- ^ __Returns:__ The attribute list passed in
attrListRef list = liftIO $ do
    maybeList <- case list of
        Nothing -> return FP.nullPtr
        Just jList -> do
            jList' <- unsafeManagedPtrGetPtr jList
            return jList'
    result <- pango_attr_list_ref maybeList
    checkUnexpectedReturnNULL "attrListRef" result
    result' <- (wrapBoxed AttrList) result
    whenJust list touchManagedPtr
    return result'

#if defined(ENABLE_OVERLOADING)
data AttrListRefMethodInfo
instance (signature ~ (m AttrList), MonadIO m) => O.OverloadedMethod AttrListRefMethodInfo AttrList signature where
    overloadedMethod i = attrListRef (Just i)

instance O.OverloadedMethodInfo AttrListRefMethodInfo AttrList where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.AttrList.attrListRef",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-AttrList.html#v:attrListRef"
        })


#endif

-- method AttrList::splice
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoAttrList`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "other"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "another `PangoAttrList`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "pos"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the position in @list at which to insert @other"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "len"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length of the spliced segment. (Note that this\n  must be specified since the attributes in @other may only\n  be present at some subsection of this range)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_list_splice" pango_attr_list_splice :: 
    Ptr AttrList ->                         -- list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    Ptr AttrList ->                         -- other : TInterface (Name {namespace = "Pango", name = "AttrList"})
    Int32 ->                                -- pos : TBasicType TInt
    Int32 ->                                -- len : TBasicType TInt
    IO ()

-- | This function opens up a hole in /@list@/, fills it
-- in with attributes from the left, and then merges
-- /@other@/ on top of the hole.
-- 
-- This operation is equivalent to stretching every attribute
-- that applies at position /@pos@/ in /@list@/ by an amount /@len@/,
-- and then calling 'GI.Pango.Structs.AttrList.attrListChange' with a copy
-- of each attribute in /@other@/ in sequence (offset in position
-- by /@pos@/, and limited in length to /@len@/).
-- 
-- This operation proves useful for, for instance, inserting
-- a pre-edit string in the middle of an edit buffer.
-- 
-- For backwards compatibility, the function behaves differently
-- when /@len@/ is 0. In this case, the attributes from /@other@/ are
-- not imited to /@len@/, and are just overlayed on top of /@list@/.
-- 
-- This mode is useful for merging two lists of attributes together.
attrListSplice ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    AttrList
    -- ^ /@list@/: a @PangoAttrList@
    -> AttrList
    -- ^ /@other@/: another @PangoAttrList@
    -> Int32
    -- ^ /@pos@/: the position in /@list@/ at which to insert /@other@/
    -> Int32
    -- ^ /@len@/: the length of the spliced segment. (Note that this
    --   must be specified since the attributes in /@other@/ may only
    --   be present at some subsection of this range)
    -> m ()
attrListSplice list other pos len = liftIO $ do
    list' <- unsafeManagedPtrGetPtr list
    other' <- unsafeManagedPtrGetPtr other
    pango_attr_list_splice list' other' pos len
    touchManagedPtr list
    touchManagedPtr other
    return ()

#if defined(ENABLE_OVERLOADING)
data AttrListSpliceMethodInfo
instance (signature ~ (AttrList -> Int32 -> Int32 -> m ()), MonadIO m) => O.OverloadedMethod AttrListSpliceMethodInfo AttrList signature where
    overloadedMethod = attrListSplice

instance O.OverloadedMethodInfo AttrListSpliceMethodInfo AttrList where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.AttrList.attrListSplice",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-AttrList.html#v:attrListSplice"
        })


#endif

-- method AttrList::to_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoAttrList`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_list_to_string" pango_attr_list_to_string :: 
    Ptr AttrList ->                         -- list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    IO CString

-- | Serializes a @PangoAttrList@ to a string.
-- 
-- In the resulting string, serialized attributes are separated by newlines or commas.
-- Individual attributes are serialized to a string of the form
-- 
--     [START END] TYPE VALUE
-- 
-- Where START and END are the indices (with -1 being accepted in place
-- of MAXUINT), TYPE is the nickname of the attribute value type, e.g.
-- _weight_ or _stretch_, and the value is serialized according to its type:
-- 
-- Optionally, START and END can be omitted to indicate unlimited extent.
-- 
-- * enum values as nick or numeric value
-- * boolean values as _true_ or _false_
-- * integers and floats as numbers
-- * strings as string, optionally quoted
-- * font features as quoted string
-- * PangoLanguage as string
-- * PangoFontDescription as serialized by 'GI.Pango.Structs.FontDescription.fontDescriptionToString', quoted
-- * PangoColor as serialized by 'GI.Pango.Structs.Color.colorToString'
-- 
-- 
-- Examples:
-- 
--     0 10 foreground red, 5 15 weight bold, 0 200 font-desc \"Sans 10\"
-- 
--     0 -1 weight 700
--     0 100 family Times
-- 
--     weight bold
-- 
-- To parse the returned value, use 'GI.Pango.Structs.AttrList.attrListFromString'.
-- 
-- Note that shape attributes can not be serialized.
-- 
-- /Since: 1.50/
attrListToString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    AttrList
    -- ^ /@list@/: a @PangoAttrList@
    -> m T.Text
    -- ^ __Returns:__ a newly allocated string
attrListToString list = liftIO $ do
    list' <- unsafeManagedPtrGetPtr list
    result <- pango_attr_list_to_string list'
    checkUnexpectedReturnNULL "attrListToString" result
    result' <- cstringToText result
    freeMem result
    touchManagedPtr list
    return result'

#if defined(ENABLE_OVERLOADING)
data AttrListToStringMethodInfo
instance (signature ~ (m T.Text), MonadIO m) => O.OverloadedMethod AttrListToStringMethodInfo AttrList signature where
    overloadedMethod = attrListToString

instance O.OverloadedMethodInfo AttrListToStringMethodInfo AttrList where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.AttrList.attrListToString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-AttrList.html#v:attrListToString"
        })


#endif

-- method AttrList::unref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoAttrList`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_list_unref" pango_attr_list_unref :: 
    Ptr AttrList ->                         -- list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    IO ()

-- | Decrease the reference count of the given attribute
-- list by one.
-- 
-- If the result is zero, free the attribute list
-- and the attributes it contains.
attrListUnref ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (AttrList)
    -- ^ /@list@/: a @PangoAttrList@
    -> m ()
attrListUnref list = liftIO $ do
    maybeList <- case list of
        Nothing -> return FP.nullPtr
        Just jList -> do
            jList' <- unsafeManagedPtrGetPtr jList
            return jList'
    pango_attr_list_unref maybeList
    whenJust list touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data AttrListUnrefMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod AttrListUnrefMethodInfo AttrList signature where
    overloadedMethod i = attrListUnref (Just i)

instance O.OverloadedMethodInfo AttrListUnrefMethodInfo AttrList where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.AttrList.attrListUnref",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-AttrList.html#v:attrListUnref"
        })


#endif

-- method AttrList::update
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "list"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `PangoAttrList`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "pos"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the position of the change"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "remove"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of removed bytes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "add"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of added bytes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_list_update" pango_attr_list_update :: 
    Ptr AttrList ->                         -- list : TInterface (Name {namespace = "Pango", name = "AttrList"})
    Int32 ->                                -- pos : TBasicType TInt
    Int32 ->                                -- remove : TBasicType TInt
    Int32 ->                                -- add : TBasicType TInt
    IO ()

-- | Update indices of attributes in /@list@/ for a change in the
-- text they refer to.
-- 
-- The change that this function applies is removing /@remove@/
-- bytes at position /@pos@/ and inserting /@add@/ bytes instead.
-- 
-- Attributes that fall entirely in the (/@pos@/, /@pos@/ + /@remove@/)
-- range are removed.
-- 
-- Attributes that start or end inside the (/@pos@/, /@pos@/ + /@remove@/)
-- range are shortened to reflect the removal.
-- 
-- Attributes start and end positions are updated if they are
-- behind /@pos@/ + /@remove@/.
-- 
-- /Since: 1.44/
attrListUpdate ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    AttrList
    -- ^ /@list@/: a @PangoAttrList@
    -> Int32
    -- ^ /@pos@/: the position of the change
    -> Int32
    -- ^ /@remove@/: the number of removed bytes
    -> Int32
    -- ^ /@add@/: the number of added bytes
    -> m ()
attrListUpdate list pos remove add = liftIO $ do
    list' <- unsafeManagedPtrGetPtr list
    pango_attr_list_update list' pos remove add
    touchManagedPtr list
    return ()

#if defined(ENABLE_OVERLOADING)
data AttrListUpdateMethodInfo
instance (signature ~ (Int32 -> Int32 -> Int32 -> m ()), MonadIO m) => O.OverloadedMethod AttrListUpdateMethodInfo AttrList signature where
    overloadedMethod = attrListUpdate

instance O.OverloadedMethodInfo AttrListUpdateMethodInfo AttrList where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Pango.Structs.AttrList.attrListUpdate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-pango-1.0.30/docs/GI-Pango-Structs-AttrList.html#v:attrListUpdate"
        })


#endif

-- method AttrList::from_string
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "AttrList" })
-- throws : False
-- Skip return : False

foreign import ccall "pango_attr_list_from_string" pango_attr_list_from_string :: 
    CString ->                              -- text : TBasicType TUTF8
    IO (Ptr AttrList)

-- | Deserializes a @PangoAttrList@ from a string.
-- 
-- This is the counterpart to 'GI.Pango.Structs.AttrList.attrListToString'.
-- See that functions for details about the format.
-- 
-- /Since: 1.50/
attrListFromString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@text@/: a string
    -> m (Maybe AttrList)
    -- ^ __Returns:__ a new @PangoAttrList@
attrListFromString text = liftIO $ do
    text' <- textToCString text
    result <- pango_attr_list_from_string text'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed AttrList) result'
        return result''
    freeMem text'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveAttrListMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveAttrListMethod "change" o = AttrListChangeMethodInfo
    ResolveAttrListMethod "copy" o = AttrListCopyMethodInfo
    ResolveAttrListMethod "equal" o = AttrListEqualMethodInfo
    ResolveAttrListMethod "filter" o = AttrListFilterMethodInfo
    ResolveAttrListMethod "insert" o = AttrListInsertMethodInfo
    ResolveAttrListMethod "insertBefore" o = AttrListInsertBeforeMethodInfo
    ResolveAttrListMethod "ref" o = AttrListRefMethodInfo
    ResolveAttrListMethod "splice" o = AttrListSpliceMethodInfo
    ResolveAttrListMethod "toString" o = AttrListToStringMethodInfo
    ResolveAttrListMethod "unref" o = AttrListUnrefMethodInfo
    ResolveAttrListMethod "update" o = AttrListUpdateMethodInfo
    ResolveAttrListMethod "getAttributes" o = AttrListGetAttributesMethodInfo
    ResolveAttrListMethod "getIterator" o = AttrListGetIteratorMethodInfo
    ResolveAttrListMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveAttrListMethod t AttrList, O.OverloadedMethod info AttrList p) => OL.IsLabel t (AttrList -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveAttrListMethod t AttrList, O.OverloadedMethod info AttrList p, R.HasField t AttrList p) => R.HasField t AttrList p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveAttrListMethod t AttrList, O.OverloadedMethodInfo info AttrList) => OL.IsLabel t (O.MethodProxy info AttrList) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


