// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef UI_BASE_MODELS_COMBOBOX_MODEL_H_
#define UI_BASE_MODELS_COMBOBOX_MODEL_H_

#include "base/component_export.h"
#include "base/strings/string16.h"

namespace ui {

class ComboboxModelObserver;
class ImageModel;

// A data model for a combo box.
class COMPONENT_EXPORT(UI_BASE) ComboboxModel {
 public:
  virtual ~ComboboxModel() {}

  // Returns the number of items in the combo box.
  virtual int GetItemCount() const = 0;

  // Returns the string at the specified index.
  virtual base::string16 GetItemAt(int index) const = 0;

  // Returns the string to be shown in the dropdown for the item at |index|. By
  // default, it returns GetItemAt(index).
  virtual base::string16 GetDropDownTextAt(int index) const;

  // Returns the secondary string at the specified index. Secondary strings are
  // displayed in a second line inside every menu item.
  virtual base::string16 GetDropDownSecondaryTextAt(int index) const;

  // Gets the icon for the item at the specified index. ImageModel is empty if
  // there is no icon.
  virtual ImageModel GetIconAt(int index) const;

  // Gets the icon for the item at |index|. ImageModel is empty if there is no
  // icon. By default, it returns GetIconAt(index).
  virtual ImageModel GetDropDownIconAt(int index) const;

  // Should return true if the item at |index| is a non-selectable separator
  // item.
  virtual bool IsItemSeparatorAt(int index) const;

  // The index of the item that is selected by default (before user
  // interaction).
  virtual int GetDefaultIndex() const;

  // Returns true if the item at |index| is enabled.
  virtual bool IsItemEnabledAt(int index) const;

  // Adds/removes an observer. Override if model supports mutation.
  virtual void AddObserver(ComboboxModelObserver* observer) {}
  virtual void RemoveObserver(ComboboxModelObserver* observer) {}
};

}  // namespace ui

#endif  // UI_BASE_MODELS_COMBOBOX_MODEL_H_
