//
// Syd: rock-solid application kernel
// src/kernel/uname.rs: uname(2) handler
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

// SAFETY: This module has been liberated from unsafe code!
#![forbid(unsafe_code)]

use libseccomp::ScmpNotifResp;
use nix::errno::Errno;

use crate::{
    config::{KERNEL_VERSION_STR, MMAP_MIN_ADDR},
    cookie::safe_uname,
    req::UNotifyEventRequest,
    uts::UTSNAME_LEN,
};

pub(crate) fn sys_uname(request: UNotifyEventRequest) -> ScmpNotifResp {
    syscall_handler!(request, |request: UNotifyEventRequest| {
        let req = request.scmpreq;

        // SAFETY: Check pointer against mmap_min_addr.
        let ptr = if req.data.args[0] < *MMAP_MIN_ADDR {
            return Err(Errno::EFAULT);
        } else {
            req.data.args[0]
        };

        // Read UtsName using uname(2) syscall.
        //
        // SAFETY: Syscall is confined with syscall argument cookies.
        let mut utsname = safe_uname()?;

        // Change the release with Syd kernel version.
        //
        // Step 1: Clear the release field.
        utsname.release.fill(0);

        // Step 2: Copy Syd kernel version string to utsname.release.
        // If the string is longer than 65 characters, it's truncated.
        let release = KERNEL_VERSION_STR.as_bytes();
        let relcopy = release.len().min(UTSNAME_LEN - 1);
        utsname.release[..relcopy].copy_from_slice(&release[..relcopy]);

        // Wipe nodename, domainname, and version fields with zeros.
        utsname.nodename.fill(0);
        utsname.domainname.fill(0);
        utsname.version.fill(0);

        // Write nodename, domainname, and version fields based on sandbox.
        // Truncate to fit into 65 characters ensuring the NUL-byte.
        //
        // Acquire a read-lock to the sandbox.
        let sandbox = request.get_sandbox();

        // Write utsname.nodename.
        let hname = sandbox.hostname.as_bytes();
        let hcopy = hname.len().min(UTSNAME_LEN - 1);
        utsname.nodename[..hcopy].copy_from_slice(&hname[..hcopy]);

        // Write utsname.domainname.
        let dname = sandbox.domainname.as_bytes();
        let dcopy = dname.len().min(UTSNAME_LEN - 1);
        utsname.domainname[..dcopy].copy_from_slice(&dname[..dcopy]);

        // Write utsname.version.
        let vname = sandbox.version.as_bytes();
        let vcopy = vname.len().min(UTSNAME_LEN - 1);
        utsname.version[..vcopy].copy_from_slice(&vname[..vcopy]);

        drop(sandbox); // release the read-lock.

        // Write structure to sandbox process memory.
        request.write_mem(utsname.as_bytes(), ptr)?;

        // All good, return success.
        Ok(request.return_syscall(0))
    })
}
