package wasi

import "fmt"

// ExitCode is the exit code generated by a process when exiting.
type ExitCode uint32

// Signal is a signal condition.
type Signal uint8

const (
	// SIGNONE means no signal. Note that POSIX has special semantics for
	// kill(pid, 0), so this value is reserved.
	SIGNONE Signal = iota

	// SIGHUP means hangup. Action: Terminates the process.
	SIGHUP

	// SIGINT is the terminate interrupt signal. Action: Terminates the
	// process.
	SIGINT

	// SIGQUIT is the terminal quit signal. Action: Terminates the process.
	SIGQUIT

	// SIGILL means illegal instruction. Action: Terminates the process.
	SIGILL

	// SIGTRAP is the trace/breakpoint trap. Action: Terminates the process.
	SIGTRAP

	// SIGABRT is the process abort signal. Action: Terminates the process.
	SIGABRT

	// SIGBUS indicates access to an undefined portion of a memory object.
	// Action: Terminates the process.
	SIGBUS

	// SIGFPE indicates an erroneous arithmetic operation. Action: Terminates
	// the process.
	SIGFPE

	// SIGKILL means kill. Action: Terminates the process.
	SIGKILL

	// SIGUSR1 is the user-defined signal 1. Action: Terminates the process.
	SIGUSR1

	// SIGSEGV indicates an invalid memory reference. Action: Terminates the
	// process.
	SIGSEGV

	// SIGUSR2 is the user-defined signal 1. Action: Terminates the process.
	SIGUSR2

	// SIGPIPE indicates a write on a pipe with no one to read it.
	// Action: Ignored.
	SIGPIPE

	// SIGALRM indicates an alarm clock. Action: Terminates the process.
	SIGALRM

	// SIGTERM is the termination signal. Action: Terminates the process.
	SIGTERM

	// SIGCHLD indicates that a child process terminated, stopped, or
	// continued. Action: Ignored.
	SIGCHLD

	// SIGCONT indicates that execution should continue, if stopped.
	// Action: Continues executing, if stopped.
	SIGCONT

	// SIGSTOP means stop executing. Action: Stops executing.
	SIGSTOP

	// SIGTSTP is the terminal stop signal. Action: Stops executing.
	SIGTSTP

	// SIGTTIN indicates that a background process is attempting read.
	// Action: Stops executing.
	SIGTTIN

	// SIGTTOU indicates that a background process is attempting write.
	// Action: Stops executing.
	SIGTTOU

	// SIGURG indicates that high bandwidth data is available at a socket.
	// Action: Ignored.
	SIGURG

	// SIGXCPU means CPU time limit exceeded. Action: Terminates the process.
	SIGXCPU

	// SIGXFSZ means file size limit exceeded. Action: Terminates the process.
	SIGXFSZ

	// SIGVTALRM means virtual timer expired. Action: Terminates the process.
	SIGVTALRM

	// SIGPROF means profiling timer expired. Action: Terminates the process.
	SIGPROF

	// SIGWINCH means window changed. Action: Ignored.
	SIGWINCH

	// SIGPOLL means I/O is possible. Action: Terminates the process.
	SIGPOLL

	// SIGPWR indicates power failure. Action: Terminates the process.
	SIGPWR

	// SIGSYS indicates a bad system call. Action: Terminates the process.
	SIGSYS
)

func (s Signal) String() string {
	if int(s) < len(signalStrings) {
		return signalStrings[s]
	}
	return fmt.Sprintf("Signal(%d)", s)
}

func (s Signal) Name() string {
	if int(s) < len(signalNames) {
		return signalNames[s]
	}
	return fmt.Sprintf("Signal(%d)", s)
}

var signalStrings = [...]string{
	SIGNONE:   "no signal",
	SIGHUP:    "hangup",
	SIGINT:    "interrupt",
	SIGQUIT:   "quit",
	SIGILL:    "illegal instruction",
	SIGTRAP:   "trace/breakpoint trap",
	SIGABRT:   "abort",
	SIGBUS:    "bus error",
	SIGFPE:    "floating point exception",
	SIGKILL:   "killed",
	SIGUSR1:   "user-defined signal 1",
	SIGSEGV:   "segmentation fault",
	SIGUSR2:   "user-defined signal 2",
	SIGPIPE:   "broken pipe",
	SIGALRM:   "alarm clock",
	SIGTERM:   "terminated",
	SIGCHLD:   "child exited",
	SIGCONT:   "continued",
	SIGSTOP:   "stopped (signal)",
	SIGTSTP:   "stopped",
	SIGTTIN:   "stopped (tty input)",
	SIGTTOU:   "stopped (tty output)",
	SIGURG:    "urgent I/O condition",
	SIGXCPU:   "CPU time limit exceeded",
	SIGXFSZ:   "file size limit exceeded",
	SIGVTALRM: "virtual timer expired",
	SIGPROF:   "profiling timer expired",
	SIGWINCH:  "window changed",
	SIGPOLL:   "I/O possible",
	SIGPWR:    "power failure",
	SIGSYS:    "bad system call",
}

var signalNames = [...]string{
	SIGNONE:   "SIGNONE",
	SIGHUP:    "SIGHUP",
	SIGINT:    "SIGINT",
	SIGQUIT:   "SIGQUIT",
	SIGILL:    "SIGILL",
	SIGTRAP:   "SIGTRAP",
	SIGABRT:   "SIGABRT",
	SIGBUS:    "SIGBUS",
	SIGFPE:    "SIGFPE",
	SIGKILL:   "SIGKILL",
	SIGUSR1:   "SIGUSR1",
	SIGSEGV:   "SIGSEGV",
	SIGUSR2:   "SIGUSR2",
	SIGPIPE:   "SIGPIPE",
	SIGALRM:   "SIGALRM",
	SIGTERM:   "SIGTERM",
	SIGCHLD:   "SIGCHLD",
	SIGCONT:   "SIGCONT",
	SIGSTOP:   "SIGSTOP",
	SIGTSTP:   "SIGTSTP",
	SIGTTIN:   "SIGTTIN",
	SIGTTOU:   "SIGTTOU",
	SIGURG:    "SIGURG",
	SIGXCPU:   "SIGXCPU",
	SIGXFSZ:   "SIGXFSZ",
	SIGVTALRM: "SIGVTALRM",
	SIGPROF:   "SIGPROF",
	SIGWINCH:  "SIGWINCH",
	SIGPOLL:   "SIGPOLL",
	SIGPWR:    "SIGPWR",
	SIGSYS:    "SIGSYS",
}
