// yabridge: a Wine plugin bridge
// Copyright (C) 2020-2024 Robbert van der Helm
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.

#pragma once

#include <pluginterfaces/vst/ivsteditcontroller.h>

#include "../../common.h"
#include "../base.h"

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wnon-virtual-dtor"

/**
 * Wraps around `IMidiMapping` for serialization purposes. This is instantiated
 * as part of `Vst3PluginProxy`.
 */
class YaMidiMapping : public Steinberg::Vst::IMidiMapping {
   public:
    /**
     * These are the arguments for creating a `YaMidiMapping`.
     */
    struct ConstructArgs {
        ConstructArgs() noexcept;

        /**
         * Check whether an existing implementation implements `IMidiMapping`
         * and read arguments from it.
         */
        ConstructArgs(Steinberg::IPtr<Steinberg::FUnknown> object) noexcept;

        /**
         * Whether the object supported this interface.
         */
        bool supported;

        template <typename S>
        void serialize(S& s) {
            s.value1b(supported);
        }
    };

    /**
     * Instantiate this instance with arguments read from another interface
     * implementation.
     */
    YaMidiMapping(ConstructArgs&& args) noexcept;

    virtual ~YaMidiMapping() noexcept = default;

    inline bool supported() const noexcept { return arguments_.supported; }

    /**
     * The response code and returned parameter ID for a call to
     * `IMidiMapping::getMidiControllerAssignment(bus_index, channel,
     * midi_controller_number, &id)`.
     */
    struct GetMidiControllerAssignmentResponse {
        UniversalTResult result;
        Steinberg::Vst::ParamID id;

        template <typename S>
        void serialize(S& s) {
            s.object(result);
            s.value4b(id);
        }
    };

    /**
     * Message to pass through a call to
     * `IMidiMapping::getMidiControllerAssignment(bus_index, channel,
     * midi_controller_number, &id)` to the Wine plugin host.
     */
    struct GetMidiControllerAssignment {
        using Response = GetMidiControllerAssignmentResponse;

        native_size_t instance_id;

        int32 bus_index;
        int16 channel;
        Steinberg::Vst::CtrlNumber midi_controller_number;

        template <typename S>
        void serialize(S& s) {
            s.value8b(instance_id);
            s.value4b(bus_index);
            s.value2b(channel);
            s.value2b(midi_controller_number);
        }
    };

    virtual tresult PLUGIN_API getMidiControllerAssignment(
        int32 busIndex,
        int16 channel,
        Steinberg::Vst::CtrlNumber midiControllerNumber,
        Steinberg::Vst::ParamID& id /*out*/) override = 0;

   protected:
    ConstructArgs arguments_;
};

#pragma GCC diagnostic pop
