﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/dlm/DLM_EXPORTS.h>
#include <aws/dlm/model/GettablePolicyStateValues.h>
#include <aws/dlm/model/PolicyDetails.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace DLM {
namespace Model {

/**
 * <p>Information about a lifecycle policy.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/dlm-2018-01-12/LifecyclePolicy">AWS
 * API Reference</a></p>
 */
class LifecyclePolicy {
 public:
  AWS_DLM_API LifecyclePolicy() = default;
  AWS_DLM_API LifecyclePolicy(Aws::Utils::Json::JsonView jsonValue);
  AWS_DLM_API LifecyclePolicy& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_DLM_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The identifier of the lifecycle policy.</p>
   */
  inline const Aws::String& GetPolicyId() const { return m_policyId; }
  inline bool PolicyIdHasBeenSet() const { return m_policyIdHasBeenSet; }
  template <typename PolicyIdT = Aws::String>
  void SetPolicyId(PolicyIdT&& value) {
    m_policyIdHasBeenSet = true;
    m_policyId = std::forward<PolicyIdT>(value);
  }
  template <typename PolicyIdT = Aws::String>
  LifecyclePolicy& WithPolicyId(PolicyIdT&& value) {
    SetPolicyId(std::forward<PolicyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the lifecycle policy.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  LifecyclePolicy& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The activation state of the lifecycle policy.</p>
   */
  inline GettablePolicyStateValues GetState() const { return m_state; }
  inline bool StateHasBeenSet() const { return m_stateHasBeenSet; }
  inline void SetState(GettablePolicyStateValues value) {
    m_stateHasBeenSet = true;
    m_state = value;
  }
  inline LifecyclePolicy& WithState(GettablePolicyStateValues value) {
    SetState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the status.</p>
   */
  inline const Aws::String& GetStatusMessage() const { return m_statusMessage; }
  inline bool StatusMessageHasBeenSet() const { return m_statusMessageHasBeenSet; }
  template <typename StatusMessageT = Aws::String>
  void SetStatusMessage(StatusMessageT&& value) {
    m_statusMessageHasBeenSet = true;
    m_statusMessage = std::forward<StatusMessageT>(value);
  }
  template <typename StatusMessageT = Aws::String>
  LifecyclePolicy& WithStatusMessage(StatusMessageT&& value) {
    SetStatusMessage(std::forward<StatusMessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the IAM role used to run the operations
   * specified by the lifecycle policy.</p>
   */
  inline const Aws::String& GetExecutionRoleArn() const { return m_executionRoleArn; }
  inline bool ExecutionRoleArnHasBeenSet() const { return m_executionRoleArnHasBeenSet; }
  template <typename ExecutionRoleArnT = Aws::String>
  void SetExecutionRoleArn(ExecutionRoleArnT&& value) {
    m_executionRoleArnHasBeenSet = true;
    m_executionRoleArn = std::forward<ExecutionRoleArnT>(value);
  }
  template <typename ExecutionRoleArnT = Aws::String>
  LifecyclePolicy& WithExecutionRoleArn(ExecutionRoleArnT&& value) {
    SetExecutionRoleArn(std::forward<ExecutionRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The local date and time when the lifecycle policy was created.</p>
   */
  inline const Aws::Utils::DateTime& GetDateCreated() const { return m_dateCreated; }
  inline bool DateCreatedHasBeenSet() const { return m_dateCreatedHasBeenSet; }
  template <typename DateCreatedT = Aws::Utils::DateTime>
  void SetDateCreated(DateCreatedT&& value) {
    m_dateCreatedHasBeenSet = true;
    m_dateCreated = std::forward<DateCreatedT>(value);
  }
  template <typename DateCreatedT = Aws::Utils::DateTime>
  LifecyclePolicy& WithDateCreated(DateCreatedT&& value) {
    SetDateCreated(std::forward<DateCreatedT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The local date and time when the lifecycle policy was last modified.</p>
   */
  inline const Aws::Utils::DateTime& GetDateModified() const { return m_dateModified; }
  inline bool DateModifiedHasBeenSet() const { return m_dateModifiedHasBeenSet; }
  template <typename DateModifiedT = Aws::Utils::DateTime>
  void SetDateModified(DateModifiedT&& value) {
    m_dateModifiedHasBeenSet = true;
    m_dateModified = std::forward<DateModifiedT>(value);
  }
  template <typename DateModifiedT = Aws::Utils::DateTime>
  LifecyclePolicy& WithDateModified(DateModifiedT&& value) {
    SetDateModified(std::forward<DateModifiedT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration of the lifecycle policy</p>
   */
  inline const PolicyDetails& GetPolicyDetails() const { return m_policyDetails; }
  inline bool PolicyDetailsHasBeenSet() const { return m_policyDetailsHasBeenSet; }
  template <typename PolicyDetailsT = PolicyDetails>
  void SetPolicyDetails(PolicyDetailsT&& value) {
    m_policyDetailsHasBeenSet = true;
    m_policyDetails = std::forward<PolicyDetailsT>(value);
  }
  template <typename PolicyDetailsT = PolicyDetails>
  LifecyclePolicy& WithPolicyDetails(PolicyDetailsT&& value) {
    SetPolicyDetails(std::forward<PolicyDetailsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  LifecyclePolicy& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  LifecyclePolicy& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the policy.</p>
   */
  inline const Aws::String& GetPolicyArn() const { return m_policyArn; }
  inline bool PolicyArnHasBeenSet() const { return m_policyArnHasBeenSet; }
  template <typename PolicyArnT = Aws::String>
  void SetPolicyArn(PolicyArnT&& value) {
    m_policyArnHasBeenSet = true;
    m_policyArn = std::forward<PolicyArnT>(value);
  }
  template <typename PolicyArnT = Aws::String>
  LifecyclePolicy& WithPolicyArn(PolicyArnT&& value) {
    SetPolicyArn(std::forward<PolicyArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether the policy is a default lifecycle policy or a custom
   * lifecycle policy.</p> <ul> <li> <p> <code>true</code> - the policy is a default
   * policy.</p> </li> <li> <p> <code>false</code> - the policy is a custom
   * policy.</p> </li> </ul>
   */
  inline bool GetDefaultPolicy() const { return m_defaultPolicy; }
  inline bool DefaultPolicyHasBeenSet() const { return m_defaultPolicyHasBeenSet; }
  inline void SetDefaultPolicy(bool value) {
    m_defaultPolicyHasBeenSet = true;
    m_defaultPolicy = value;
  }
  inline LifecyclePolicy& WithDefaultPolicy(bool value) {
    SetDefaultPolicy(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_policyId;

  Aws::String m_description;

  GettablePolicyStateValues m_state{GettablePolicyStateValues::NOT_SET};

  Aws::String m_statusMessage;

  Aws::String m_executionRoleArn;

  Aws::Utils::DateTime m_dateCreated{};

  Aws::Utils::DateTime m_dateModified{};

  PolicyDetails m_policyDetails;

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::String m_policyArn;

  bool m_defaultPolicy{false};
  bool m_policyIdHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_stateHasBeenSet = false;
  bool m_statusMessageHasBeenSet = false;
  bool m_executionRoleArnHasBeenSet = false;
  bool m_dateCreatedHasBeenSet = false;
  bool m_dateModifiedHasBeenSet = false;
  bool m_policyDetailsHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_policyArnHasBeenSet = false;
  bool m_defaultPolicyHasBeenSet = false;
};

}  // namespace Model
}  // namespace DLM
}  // namespace Aws
