﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/fsx/FSxRequest.h>
#include <aws/fsx/FSx_EXPORTS.h>
#include <aws/fsx/model/S3DataRepositoryConfiguration.h>

#include <utility>

namespace Aws {
namespace FSx {
namespace Model {

/**
 */
class UpdateDataRepositoryAssociationRequest : public FSxRequest {
 public:
  AWS_FSX_API UpdateDataRepositoryAssociationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateDataRepositoryAssociation"; }

  AWS_FSX_API Aws::String SerializePayload() const override;

  AWS_FSX_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the data repository association that you are updating.</p>
   */
  inline const Aws::String& GetAssociationId() const { return m_associationId; }
  inline bool AssociationIdHasBeenSet() const { return m_associationIdHasBeenSet; }
  template <typename AssociationIdT = Aws::String>
  void SetAssociationId(AssociationIdT&& value) {
    m_associationIdHasBeenSet = true;
    m_associationId = std::forward<AssociationIdT>(value);
  }
  template <typename AssociationIdT = Aws::String>
  UpdateDataRepositoryAssociationRequest& WithAssociationId(AssociationIdT&& value) {
    SetAssociationId(std::forward<AssociationIdT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  UpdateDataRepositoryAssociationRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For files imported from a data repository, this value determines the stripe
   * count and maximum amount of data per file (in MiB) stored on a single physical
   * disk. The maximum number of disks that a single file can be striped across is
   * limited by the total number of disks that make up the file system.</p> <p>The
   * default chunk size is 1,024 MiB (1 GiB) and can go as high as 512,000 MiB (500
   * GiB). Amazon S3 objects have a maximum size of 5 TB.</p>
   */
  inline int GetImportedFileChunkSize() const { return m_importedFileChunkSize; }
  inline bool ImportedFileChunkSizeHasBeenSet() const { return m_importedFileChunkSizeHasBeenSet; }
  inline void SetImportedFileChunkSize(int value) {
    m_importedFileChunkSizeHasBeenSet = true;
    m_importedFileChunkSize = value;
  }
  inline UpdateDataRepositoryAssociationRequest& WithImportedFileChunkSize(int value) {
    SetImportedFileChunkSize(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration for an Amazon S3 data repository linked to an Amazon FSx
   * Lustre file system with a data repository association. The configuration defines
   * which file events (new, changed, or deleted files or directories) are
   * automatically imported from the linked data repository to the file system or
   * automatically exported from the file system to the data repository.</p>
   */
  inline const S3DataRepositoryConfiguration& GetS3() const { return m_s3; }
  inline bool S3HasBeenSet() const { return m_s3HasBeenSet; }
  template <typename S3T = S3DataRepositoryConfiguration>
  void SetS3(S3T&& value) {
    m_s3HasBeenSet = true;
    m_s3 = std::forward<S3T>(value);
  }
  template <typename S3T = S3DataRepositoryConfiguration>
  UpdateDataRepositoryAssociationRequest& WithS3(S3T&& value) {
    SetS3(std::forward<S3T>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_associationId;

  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};

  int m_importedFileChunkSize{0};

  S3DataRepositoryConfiguration m_s3;
  bool m_associationIdHasBeenSet = false;
  bool m_clientRequestTokenHasBeenSet = true;
  bool m_importedFileChunkSizeHasBeenSet = false;
  bool m_s3HasBeenSet = false;
};

}  // namespace Model
}  // namespace FSx
}  // namespace Aws
