﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iot-managed-integrations/IoTManagedIntegrationsRequest.h>
#include <aws/iot-managed-integrations/IoTManagedIntegrations_EXPORTS.h>
#include <aws/iot-managed-integrations/model/PushConfig.h>

#include <utility>

namespace Aws {
namespace IoTManagedIntegrations {
namespace Model {

/**
 */
class CreateOtaTaskConfigurationRequest : public IoTManagedIntegrationsRequest {
 public:
  AWS_IOTMANAGEDINTEGRATIONS_API CreateOtaTaskConfigurationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateOtaTaskConfiguration"; }

  AWS_IOTMANAGEDINTEGRATIONS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A description of the over-the-air (OTA) task configuration.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateOtaTaskConfigurationRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the over-the-air (OTA) task.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateOtaTaskConfigurationRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Describes the type of configuration used for the over-the-air (OTA) task.</p>
   */
  inline const PushConfig& GetPushConfig() const { return m_pushConfig; }
  inline bool PushConfigHasBeenSet() const { return m_pushConfigHasBeenSet; }
  template <typename PushConfigT = PushConfig>
  void SetPushConfig(PushConfigT&& value) {
    m_pushConfigHasBeenSet = true;
    m_pushConfig = std::forward<PushConfigT>(value);
  }
  template <typename PushConfigT = PushConfig>
  CreateOtaTaskConfigurationRequest& WithPushConfig(PushConfigT&& value) {
    SetPushConfig(std::forward<PushConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An idempotency token. If you retry a request that completed successfully
   * initially using the same client token and parameters, then the retry attempt
   * will succeed without performing any further actions.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateOtaTaskConfigurationRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_description;

  Aws::String m_name;

  PushConfig m_pushConfig;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_descriptionHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_pushConfigHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace IoTManagedIntegrations
}  // namespace Aws
