﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iot/IoTRequest.h>
#include <aws/iot/IoT_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace IoT {
namespace Model {

/**
 */
class CancelJobExecutionRequest : public IoTRequest {
 public:
  AWS_IOT_API CancelJobExecutionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CancelJobExecution"; }

  AWS_IOT_API Aws::String SerializePayload() const override;

  AWS_IOT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The ID of the job to be canceled.</p>
   */
  inline const Aws::String& GetJobId() const { return m_jobId; }
  inline bool JobIdHasBeenSet() const { return m_jobIdHasBeenSet; }
  template <typename JobIdT = Aws::String>
  void SetJobId(JobIdT&& value) {
    m_jobIdHasBeenSet = true;
    m_jobId = std::forward<JobIdT>(value);
  }
  template <typename JobIdT = Aws::String>
  CancelJobExecutionRequest& WithJobId(JobIdT&& value) {
    SetJobId(std::forward<JobIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the thing whose execution of the job will be canceled.</p>
   */
  inline const Aws::String& GetThingName() const { return m_thingName; }
  inline bool ThingNameHasBeenSet() const { return m_thingNameHasBeenSet; }
  template <typename ThingNameT = Aws::String>
  void SetThingName(ThingNameT&& value) {
    m_thingNameHasBeenSet = true;
    m_thingName = std::forward<ThingNameT>(value);
  }
  template <typename ThingNameT = Aws::String>
  CancelJobExecutionRequest& WithThingName(ThingNameT&& value) {
    SetThingName(std::forward<ThingNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>(Optional) If <code>true</code> the job execution will be canceled if it has
   * status IN_PROGRESS or QUEUED, otherwise the job execution will be canceled only
   * if it has status QUEUED. If you attempt to cancel a job execution that is
   * IN_PROGRESS, and you do not set <code>force</code> to <code>true</code>, then an
   * <code>InvalidStateTransitionException</code> will be thrown. The default is
   * <code>false</code>.</p> <p>Canceling a job execution which is "IN_PROGRESS",
   * will cause the device to be unable to update the job execution status. Use
   * caution and ensure that the device is able to recover to a valid state.</p>
   */
  inline bool GetForce() const { return m_force; }
  inline bool ForceHasBeenSet() const { return m_forceHasBeenSet; }
  inline void SetForce(bool value) {
    m_forceHasBeenSet = true;
    m_force = value;
  }
  inline CancelJobExecutionRequest& WithForce(bool value) {
    SetForce(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>(Optional) The expected current version of the job execution. Each time you
   * update the job execution, its version is incremented. If the version of the job
   * execution stored in Jobs does not match, the update is rejected with a
   * VersionMismatch error, and an ErrorResponse that contains the current job
   * execution status data is returned. (This makes it unnecessary to perform a
   * separate DescribeJobExecution request in order to obtain the job execution
   * status data.)</p>
   */
  inline long long GetExpectedVersion() const { return m_expectedVersion; }
  inline bool ExpectedVersionHasBeenSet() const { return m_expectedVersionHasBeenSet; }
  inline void SetExpectedVersion(long long value) {
    m_expectedVersionHasBeenSet = true;
    m_expectedVersion = value;
  }
  inline CancelJobExecutionRequest& WithExpectedVersion(long long value) {
    SetExpectedVersion(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A collection of name/value pairs that describe the status of the job
   * execution. If not specified, the statusDetails are unchanged. You can specify at
   * most 10 name/value pairs.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetStatusDetails() const { return m_statusDetails; }
  inline bool StatusDetailsHasBeenSet() const { return m_statusDetailsHasBeenSet; }
  template <typename StatusDetailsT = Aws::Map<Aws::String, Aws::String>>
  void SetStatusDetails(StatusDetailsT&& value) {
    m_statusDetailsHasBeenSet = true;
    m_statusDetails = std::forward<StatusDetailsT>(value);
  }
  template <typename StatusDetailsT = Aws::Map<Aws::String, Aws::String>>
  CancelJobExecutionRequest& WithStatusDetails(StatusDetailsT&& value) {
    SetStatusDetails(std::forward<StatusDetailsT>(value));
    return *this;
  }
  template <typename StatusDetailsKeyT = Aws::String, typename StatusDetailsValueT = Aws::String>
  CancelJobExecutionRequest& AddStatusDetails(StatusDetailsKeyT&& key, StatusDetailsValueT&& value) {
    m_statusDetailsHasBeenSet = true;
    m_statusDetails.emplace(std::forward<StatusDetailsKeyT>(key), std::forward<StatusDetailsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_jobId;

  Aws::String m_thingName;

  bool m_force{false};

  long long m_expectedVersion{0};

  Aws::Map<Aws::String, Aws::String> m_statusDetails;
  bool m_jobIdHasBeenSet = false;
  bool m_thingNameHasBeenSet = false;
  bool m_forceHasBeenSet = false;
  bool m_expectedVersionHasBeenSet = false;
  bool m_statusDetailsHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoT
}  // namespace Aws
