﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iot/IoTRequest.h>
#include <aws/iot/IoT_EXPORTS.h>
#include <aws/iot/model/AbortConfig.h>
#include <aws/iot/model/JobExecutionsRetryConfig.h>
#include <aws/iot/model/JobExecutionsRolloutConfig.h>
#include <aws/iot/model/PresignedUrlConfig.h>
#include <aws/iot/model/TimeoutConfig.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace IoT {
namespace Model {

/**
 */
class UpdateJobRequest : public IoTRequest {
 public:
  AWS_IOT_API UpdateJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateJob"; }

  AWS_IOT_API Aws::String SerializePayload() const override;

  AWS_IOT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The ID of the job to be updated.</p>
   */
  inline const Aws::String& GetJobId() const { return m_jobId; }
  inline bool JobIdHasBeenSet() const { return m_jobIdHasBeenSet; }
  template <typename JobIdT = Aws::String>
  void SetJobId(JobIdT&& value) {
    m_jobIdHasBeenSet = true;
    m_jobId = std::forward<JobIdT>(value);
  }
  template <typename JobIdT = Aws::String>
  UpdateJobRequest& WithJobId(JobIdT&& value) {
    SetJobId(std::forward<JobIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A short text description of the job.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateJobRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Configuration information for pre-signed S3 URLs.</p>
   */
  inline const PresignedUrlConfig& GetPresignedUrlConfig() const { return m_presignedUrlConfig; }
  inline bool PresignedUrlConfigHasBeenSet() const { return m_presignedUrlConfigHasBeenSet; }
  template <typename PresignedUrlConfigT = PresignedUrlConfig>
  void SetPresignedUrlConfig(PresignedUrlConfigT&& value) {
    m_presignedUrlConfigHasBeenSet = true;
    m_presignedUrlConfig = std::forward<PresignedUrlConfigT>(value);
  }
  template <typename PresignedUrlConfigT = PresignedUrlConfig>
  UpdateJobRequest& WithPresignedUrlConfig(PresignedUrlConfigT&& value) {
    SetPresignedUrlConfig(std::forward<PresignedUrlConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Allows you to create a staged rollout of the job.</p>
   */
  inline const JobExecutionsRolloutConfig& GetJobExecutionsRolloutConfig() const { return m_jobExecutionsRolloutConfig; }
  inline bool JobExecutionsRolloutConfigHasBeenSet() const { return m_jobExecutionsRolloutConfigHasBeenSet; }
  template <typename JobExecutionsRolloutConfigT = JobExecutionsRolloutConfig>
  void SetJobExecutionsRolloutConfig(JobExecutionsRolloutConfigT&& value) {
    m_jobExecutionsRolloutConfigHasBeenSet = true;
    m_jobExecutionsRolloutConfig = std::forward<JobExecutionsRolloutConfigT>(value);
  }
  template <typename JobExecutionsRolloutConfigT = JobExecutionsRolloutConfig>
  UpdateJobRequest& WithJobExecutionsRolloutConfig(JobExecutionsRolloutConfigT&& value) {
    SetJobExecutionsRolloutConfig(std::forward<JobExecutionsRolloutConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Allows you to create criteria to abort a job.</p>
   */
  inline const AbortConfig& GetAbortConfig() const { return m_abortConfig; }
  inline bool AbortConfigHasBeenSet() const { return m_abortConfigHasBeenSet; }
  template <typename AbortConfigT = AbortConfig>
  void SetAbortConfig(AbortConfigT&& value) {
    m_abortConfigHasBeenSet = true;
    m_abortConfig = std::forward<AbortConfigT>(value);
  }
  template <typename AbortConfigT = AbortConfig>
  UpdateJobRequest& WithAbortConfig(AbortConfigT&& value) {
    SetAbortConfig(std::forward<AbortConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the amount of time each device has to finish its execution of the
   * job. The timer is started when the job execution status is set to
   * <code>IN_PROGRESS</code>. If the job execution status is not set to another
   * terminal state before the time expires, it will be automatically set to
   * <code>TIMED_OUT</code>. </p>
   */
  inline const TimeoutConfig& GetTimeoutConfig() const { return m_timeoutConfig; }
  inline bool TimeoutConfigHasBeenSet() const { return m_timeoutConfigHasBeenSet; }
  template <typename TimeoutConfigT = TimeoutConfig>
  void SetTimeoutConfig(TimeoutConfigT&& value) {
    m_timeoutConfigHasBeenSet = true;
    m_timeoutConfig = std::forward<TimeoutConfigT>(value);
  }
  template <typename TimeoutConfigT = TimeoutConfig>
  UpdateJobRequest& WithTimeoutConfig(TimeoutConfigT&& value) {
    SetTimeoutConfig(std::forward<TimeoutConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The namespace used to indicate that a job is a customer-managed job.</p>
   * <p>When you specify a value for this parameter, Amazon Web Services IoT Core
   * sends jobs notifications to MQTT topics that contain the value in the following
   * format.</p> <p>
   * <code>$aws/things/<i>THING_NAME</i>/jobs/<i>JOB_ID</i>/notify-namespace-<i>NAMESPACE_ID</i>/</code>
   * </p>  <p>The <code>namespaceId</code> feature is only supported by IoT
   * Greengrass at this time. For more information, see <a
   * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/setting-up.html">Setting
   * up IoT Greengrass core devices.</a> </p>
   */
  inline const Aws::String& GetNamespaceId() const { return m_namespaceId; }
  inline bool NamespaceIdHasBeenSet() const { return m_namespaceIdHasBeenSet; }
  template <typename NamespaceIdT = Aws::String>
  void SetNamespaceId(NamespaceIdT&& value) {
    m_namespaceIdHasBeenSet = true;
    m_namespaceId = std::forward<NamespaceIdT>(value);
  }
  template <typename NamespaceIdT = Aws::String>
  UpdateJobRequest& WithNamespaceId(NamespaceIdT&& value) {
    SetNamespaceId(std::forward<NamespaceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Allows you to create the criteria to retry a job.</p>
   */
  inline const JobExecutionsRetryConfig& GetJobExecutionsRetryConfig() const { return m_jobExecutionsRetryConfig; }
  inline bool JobExecutionsRetryConfigHasBeenSet() const { return m_jobExecutionsRetryConfigHasBeenSet; }
  template <typename JobExecutionsRetryConfigT = JobExecutionsRetryConfig>
  void SetJobExecutionsRetryConfig(JobExecutionsRetryConfigT&& value) {
    m_jobExecutionsRetryConfigHasBeenSet = true;
    m_jobExecutionsRetryConfig = std::forward<JobExecutionsRetryConfigT>(value);
  }
  template <typename JobExecutionsRetryConfigT = JobExecutionsRetryConfig>
  UpdateJobRequest& WithJobExecutionsRetryConfig(JobExecutionsRetryConfigT&& value) {
    SetJobExecutionsRetryConfig(std::forward<JobExecutionsRetryConfigT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_jobId;

  Aws::String m_description;

  PresignedUrlConfig m_presignedUrlConfig;

  JobExecutionsRolloutConfig m_jobExecutionsRolloutConfig;

  AbortConfig m_abortConfig;

  TimeoutConfig m_timeoutConfig;

  Aws::String m_namespaceId;

  JobExecutionsRetryConfig m_jobExecutionsRetryConfig;
  bool m_jobIdHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_presignedUrlConfigHasBeenSet = false;
  bool m_jobExecutionsRolloutConfigHasBeenSet = false;
  bool m_abortConfigHasBeenSet = false;
  bool m_timeoutConfigHasBeenSet = false;
  bool m_namespaceIdHasBeenSet = false;
  bool m_jobExecutionsRetryConfigHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoT
}  // namespace Aws
