﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/mediaconvert/MediaConvert_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace MediaConvert {
namespace Model {

/**
 * Use Job engine versions to run jobs for your production workflow on one version,
 * while you test and validate the latest version. Job engine versions are in a
 * YYYY-MM-DD format.<p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/JobEngineVersion">AWS
 * API Reference</a></p>
 */
class JobEngineVersion {
 public:
  AWS_MEDIACONVERT_API JobEngineVersion() = default;
  AWS_MEDIACONVERT_API JobEngineVersion(Aws::Utils::Json::JsonView jsonValue);
  AWS_MEDIACONVERT_API JobEngineVersion& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_MEDIACONVERT_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * The date that this Job engine version expires. Requests to create jobs with an
   * expired version result in a regular job, as if no specific Job engine version
   * was requested.
   */
  inline const Aws::Utils::DateTime& GetExpirationDate() const { return m_expirationDate; }
  inline bool ExpirationDateHasBeenSet() const { return m_expirationDateHasBeenSet; }
  template <typename ExpirationDateT = Aws::Utils::DateTime>
  void SetExpirationDate(ExpirationDateT&& value) {
    m_expirationDateHasBeenSet = true;
    m_expirationDate = std::forward<ExpirationDateT>(value);
  }
  template <typename ExpirationDateT = Aws::Utils::DateTime>
  JobEngineVersion& WithExpirationDate(ExpirationDateT&& value) {
    SetExpirationDate(std::forward<ExpirationDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * Use Job engine versions to run jobs for your production workflow on one version,
   * while you test and validate the latest version. Job engine versions represent
   * periodically grouped MediaConvert releases with new features, updates,
   * improvements, and fixes. Job engine versions are in a YYYY-MM-DD format. Note
   * that the Job engine version feature is not publicly available at this time. To
   * request access, contact AWS support.
   */
  inline const Aws::String& GetVersion() const { return m_version; }
  inline bool VersionHasBeenSet() const { return m_versionHasBeenSet; }
  template <typename VersionT = Aws::String>
  void SetVersion(VersionT&& value) {
    m_versionHasBeenSet = true;
    m_version = std::forward<VersionT>(value);
  }
  template <typename VersionT = Aws::String>
  JobEngineVersion& WithVersion(VersionT&& value) {
    SetVersion(std::forward<VersionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Utils::DateTime m_expirationDate{};

  Aws::String m_version;
  bool m_expirationDateHasBeenSet = false;
  bool m_versionHasBeenSet = false;
};

}  // namespace Model
}  // namespace MediaConvert
}  // namespace Aws
