﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/quicksight/QuickSight_EXPORTS.h>
#include <aws/quicksight/model/ChartAxisLabelOptions.h>
#include <aws/quicksight/model/ContributionAnalysisDefault.h>
#include <aws/quicksight/model/DataLabelOptions.h>
#include <aws/quicksight/model/DonutOptions.h>
#include <aws/quicksight/model/LegendOptions.h>
#include <aws/quicksight/model/PieChartFieldWells.h>
#include <aws/quicksight/model/PieChartSortConfiguration.h>
#include <aws/quicksight/model/SmallMultiplesOptions.h>
#include <aws/quicksight/model/TooltipOptions.h>
#include <aws/quicksight/model/VisualInteractionOptions.h>
#include <aws/quicksight/model/VisualPalette.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace QuickSight {
namespace Model {

/**
 * <p>The configuration of a pie chart.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/quicksight-2018-04-01/PieChartConfiguration">AWS
 * API Reference</a></p>
 */
class PieChartConfiguration {
 public:
  AWS_QUICKSIGHT_API PieChartConfiguration() = default;
  AWS_QUICKSIGHT_API PieChartConfiguration(Aws::Utils::Json::JsonView jsonValue);
  AWS_QUICKSIGHT_API PieChartConfiguration& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_QUICKSIGHT_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The field wells of the visual.</p>
   */
  inline const PieChartFieldWells& GetFieldWells() const { return m_fieldWells; }
  inline bool FieldWellsHasBeenSet() const { return m_fieldWellsHasBeenSet; }
  template <typename FieldWellsT = PieChartFieldWells>
  void SetFieldWells(FieldWellsT&& value) {
    m_fieldWellsHasBeenSet = true;
    m_fieldWells = std::forward<FieldWellsT>(value);
  }
  template <typename FieldWellsT = PieChartFieldWells>
  PieChartConfiguration& WithFieldWells(FieldWellsT&& value) {
    SetFieldWells(std::forward<FieldWellsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The sort configuration of a pie chart.</p>
   */
  inline const PieChartSortConfiguration& GetSortConfiguration() const { return m_sortConfiguration; }
  inline bool SortConfigurationHasBeenSet() const { return m_sortConfigurationHasBeenSet; }
  template <typename SortConfigurationT = PieChartSortConfiguration>
  void SetSortConfiguration(SortConfigurationT&& value) {
    m_sortConfigurationHasBeenSet = true;
    m_sortConfiguration = std::forward<SortConfigurationT>(value);
  }
  template <typename SortConfigurationT = PieChartSortConfiguration>
  PieChartConfiguration& WithSortConfiguration(SortConfigurationT&& value) {
    SetSortConfiguration(std::forward<SortConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The options that determine the shape of the chart. This option determines
   * whether the chart is a pie chart or a donut chart.</p>
   */
  inline const DonutOptions& GetDonutOptions() const { return m_donutOptions; }
  inline bool DonutOptionsHasBeenSet() const { return m_donutOptionsHasBeenSet; }
  template <typename DonutOptionsT = DonutOptions>
  void SetDonutOptions(DonutOptionsT&& value) {
    m_donutOptionsHasBeenSet = true;
    m_donutOptions = std::forward<DonutOptionsT>(value);
  }
  template <typename DonutOptionsT = DonutOptions>
  PieChartConfiguration& WithDonutOptions(DonutOptionsT&& value) {
    SetDonutOptions(std::forward<DonutOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The small multiples setup for the visual.</p>
   */
  inline const SmallMultiplesOptions& GetSmallMultiplesOptions() const { return m_smallMultiplesOptions; }
  inline bool SmallMultiplesOptionsHasBeenSet() const { return m_smallMultiplesOptionsHasBeenSet; }
  template <typename SmallMultiplesOptionsT = SmallMultiplesOptions>
  void SetSmallMultiplesOptions(SmallMultiplesOptionsT&& value) {
    m_smallMultiplesOptionsHasBeenSet = true;
    m_smallMultiplesOptions = std::forward<SmallMultiplesOptionsT>(value);
  }
  template <typename SmallMultiplesOptionsT = SmallMultiplesOptions>
  PieChartConfiguration& WithSmallMultiplesOptions(SmallMultiplesOptionsT&& value) {
    SetSmallMultiplesOptions(std::forward<SmallMultiplesOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The label options of the group/color that is displayed in a pie chart.</p>
   */
  inline const ChartAxisLabelOptions& GetCategoryLabelOptions() const { return m_categoryLabelOptions; }
  inline bool CategoryLabelOptionsHasBeenSet() const { return m_categoryLabelOptionsHasBeenSet; }
  template <typename CategoryLabelOptionsT = ChartAxisLabelOptions>
  void SetCategoryLabelOptions(CategoryLabelOptionsT&& value) {
    m_categoryLabelOptionsHasBeenSet = true;
    m_categoryLabelOptions = std::forward<CategoryLabelOptionsT>(value);
  }
  template <typename CategoryLabelOptionsT = ChartAxisLabelOptions>
  PieChartConfiguration& WithCategoryLabelOptions(CategoryLabelOptionsT&& value) {
    SetCategoryLabelOptions(std::forward<CategoryLabelOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The label options for the value that is displayed in a pie chart.</p>
   */
  inline const ChartAxisLabelOptions& GetValueLabelOptions() const { return m_valueLabelOptions; }
  inline bool ValueLabelOptionsHasBeenSet() const { return m_valueLabelOptionsHasBeenSet; }
  template <typename ValueLabelOptionsT = ChartAxisLabelOptions>
  void SetValueLabelOptions(ValueLabelOptionsT&& value) {
    m_valueLabelOptionsHasBeenSet = true;
    m_valueLabelOptions = std::forward<ValueLabelOptionsT>(value);
  }
  template <typename ValueLabelOptionsT = ChartAxisLabelOptions>
  PieChartConfiguration& WithValueLabelOptions(ValueLabelOptionsT&& value) {
    SetValueLabelOptions(std::forward<ValueLabelOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The legend display setup of the visual.</p>
   */
  inline const LegendOptions& GetLegend() const { return m_legend; }
  inline bool LegendHasBeenSet() const { return m_legendHasBeenSet; }
  template <typename LegendT = LegendOptions>
  void SetLegend(LegendT&& value) {
    m_legendHasBeenSet = true;
    m_legend = std::forward<LegendT>(value);
  }
  template <typename LegendT = LegendOptions>
  PieChartConfiguration& WithLegend(LegendT&& value) {
    SetLegend(std::forward<LegendT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The options that determine if visual data labels are displayed.</p>
   */
  inline const DataLabelOptions& GetDataLabels() const { return m_dataLabels; }
  inline bool DataLabelsHasBeenSet() const { return m_dataLabelsHasBeenSet; }
  template <typename DataLabelsT = DataLabelOptions>
  void SetDataLabels(DataLabelsT&& value) {
    m_dataLabelsHasBeenSet = true;
    m_dataLabels = std::forward<DataLabelsT>(value);
  }
  template <typename DataLabelsT = DataLabelOptions>
  PieChartConfiguration& WithDataLabels(DataLabelsT&& value) {
    SetDataLabels(std::forward<DataLabelsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tooltip display setup of the visual.</p>
   */
  inline const TooltipOptions& GetTooltip() const { return m_tooltip; }
  inline bool TooltipHasBeenSet() const { return m_tooltipHasBeenSet; }
  template <typename TooltipT = TooltipOptions>
  void SetTooltip(TooltipT&& value) {
    m_tooltipHasBeenSet = true;
    m_tooltip = std::forward<TooltipT>(value);
  }
  template <typename TooltipT = TooltipOptions>
  PieChartConfiguration& WithTooltip(TooltipT&& value) {
    SetTooltip(std::forward<TooltipT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The palette (chart color) display setup of the visual.</p>
   */
  inline const VisualPalette& GetVisualPalette() const { return m_visualPalette; }
  inline bool VisualPaletteHasBeenSet() const { return m_visualPaletteHasBeenSet; }
  template <typename VisualPaletteT = VisualPalette>
  void SetVisualPalette(VisualPaletteT&& value) {
    m_visualPaletteHasBeenSet = true;
    m_visualPalette = std::forward<VisualPaletteT>(value);
  }
  template <typename VisualPaletteT = VisualPalette>
  PieChartConfiguration& WithVisualPalette(VisualPaletteT&& value) {
    SetVisualPalette(std::forward<VisualPaletteT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The contribution analysis (anomaly configuration) setup of the visual.</p>
   */
  inline const Aws::Vector<ContributionAnalysisDefault>& GetContributionAnalysisDefaults() const { return m_contributionAnalysisDefaults; }
  inline bool ContributionAnalysisDefaultsHasBeenSet() const { return m_contributionAnalysisDefaultsHasBeenSet; }
  template <typename ContributionAnalysisDefaultsT = Aws::Vector<ContributionAnalysisDefault>>
  void SetContributionAnalysisDefaults(ContributionAnalysisDefaultsT&& value) {
    m_contributionAnalysisDefaultsHasBeenSet = true;
    m_contributionAnalysisDefaults = std::forward<ContributionAnalysisDefaultsT>(value);
  }
  template <typename ContributionAnalysisDefaultsT = Aws::Vector<ContributionAnalysisDefault>>
  PieChartConfiguration& WithContributionAnalysisDefaults(ContributionAnalysisDefaultsT&& value) {
    SetContributionAnalysisDefaults(std::forward<ContributionAnalysisDefaultsT>(value));
    return *this;
  }
  template <typename ContributionAnalysisDefaultsT = ContributionAnalysisDefault>
  PieChartConfiguration& AddContributionAnalysisDefaults(ContributionAnalysisDefaultsT&& value) {
    m_contributionAnalysisDefaultsHasBeenSet = true;
    m_contributionAnalysisDefaults.emplace_back(std::forward<ContributionAnalysisDefaultsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The general visual interactions setup for a visual.</p>
   */
  inline const VisualInteractionOptions& GetInteractions() const { return m_interactions; }
  inline bool InteractionsHasBeenSet() const { return m_interactionsHasBeenSet; }
  template <typename InteractionsT = VisualInteractionOptions>
  void SetInteractions(InteractionsT&& value) {
    m_interactionsHasBeenSet = true;
    m_interactions = std::forward<InteractionsT>(value);
  }
  template <typename InteractionsT = VisualInteractionOptions>
  PieChartConfiguration& WithInteractions(InteractionsT&& value) {
    SetInteractions(std::forward<InteractionsT>(value));
    return *this;
  }
  ///@}
 private:
  PieChartFieldWells m_fieldWells;

  PieChartSortConfiguration m_sortConfiguration;

  DonutOptions m_donutOptions;

  SmallMultiplesOptions m_smallMultiplesOptions;

  ChartAxisLabelOptions m_categoryLabelOptions;

  ChartAxisLabelOptions m_valueLabelOptions;

  LegendOptions m_legend;

  DataLabelOptions m_dataLabels;

  TooltipOptions m_tooltip;

  VisualPalette m_visualPalette;

  Aws::Vector<ContributionAnalysisDefault> m_contributionAnalysisDefaults;

  VisualInteractionOptions m_interactions;
  bool m_fieldWellsHasBeenSet = false;
  bool m_sortConfigurationHasBeenSet = false;
  bool m_donutOptionsHasBeenSet = false;
  bool m_smallMultiplesOptionsHasBeenSet = false;
  bool m_categoryLabelOptionsHasBeenSet = false;
  bool m_valueLabelOptionsHasBeenSet = false;
  bool m_legendHasBeenSet = false;
  bool m_dataLabelsHasBeenSet = false;
  bool m_tooltipHasBeenSet = false;
  bool m_visualPaletteHasBeenSet = false;
  bool m_contributionAnalysisDefaultsHasBeenSet = false;
  bool m_interactionsHasBeenSet = false;
};

}  // namespace Model
}  // namespace QuickSight
}  // namespace Aws
