﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/glacier/GlacierRequest.h>
#include <aws/glacier/Glacier_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Glacier {
namespace Model {

/**
 * <p>The input values for <code>AddTagsToVault</code>.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/glacier-2012-06-01/AddTagsToVaultInput">AWS
 * API Reference</a></p>
 */
class AddTagsToVaultRequest : public GlacierRequest {
 public:
  AWS_GLACIER_API AddTagsToVaultRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AddTagsToVault"; }

  AWS_GLACIER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The <code>AccountId</code> value is the AWS account ID of the account that
   * owns the vault. You can either specify an AWS account ID or optionally a single
   * '<code>-</code>' (hyphen), in which case Amazon Glacier uses the AWS account ID
   * associated with the credentials used to sign the request. If you use an account
   * ID, do not include any hyphens ('-') in the ID.</p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  AddTagsToVaultRequest& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the vault.</p>
   */
  inline const Aws::String& GetVaultName() const { return m_vaultName; }
  inline bool VaultNameHasBeenSet() const { return m_vaultNameHasBeenSet; }
  template <typename VaultNameT = Aws::String>
  void SetVaultName(VaultNameT&& value) {
    m_vaultNameHasBeenSet = true;
    m_vaultName = std::forward<VaultNameT>(value);
  }
  template <typename VaultNameT = Aws::String>
  AddTagsToVaultRequest& WithVaultName(VaultNameT&& value) {
    SetVaultName(std::forward<VaultNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to add to the vault. Each tag is composed of a key and a value. The
   * value can be an empty string.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  AddTagsToVaultRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  AddTagsToVaultRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_accountId;

  Aws::String m_vaultName;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_accountIdHasBeenSet = false;
  bool m_vaultNameHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glacier
}  // namespace Aws
