﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/s3tables/S3TablesRequest.h>
#include <aws/s3tables/S3Tables_EXPORTS.h>
#include <aws/s3tables/model/EncryptionConfiguration.h>
#include <aws/s3tables/model/OpenTableFormat.h>
#include <aws/s3tables/model/StorageClassConfiguration.h>
#include <aws/s3tables/model/TableMetadata.h>

#include <utility>

namespace Aws {
namespace S3Tables {
namespace Model {

/**
 */
class CreateTableRequest : public S3TablesRequest {
 public:
  AWS_S3TABLES_API CreateTableRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateTable"; }

  AWS_S3TABLES_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the table bucket to create the table
   * in.</p>
   */
  inline const Aws::String& GetTableBucketARN() const { return m_tableBucketARN; }
  inline bool TableBucketARNHasBeenSet() const { return m_tableBucketARNHasBeenSet; }
  template <typename TableBucketARNT = Aws::String>
  void SetTableBucketARN(TableBucketARNT&& value) {
    m_tableBucketARNHasBeenSet = true;
    m_tableBucketARN = std::forward<TableBucketARNT>(value);
  }
  template <typename TableBucketARNT = Aws::String>
  CreateTableRequest& WithTableBucketARN(TableBucketARNT&& value) {
    SetTableBucketARN(std::forward<TableBucketARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The namespace to associated with the table.</p>
   */
  inline const Aws::String& GetNamespace() const { return m_namespace; }
  inline bool NamespaceHasBeenSet() const { return m_namespaceHasBeenSet; }
  template <typename NamespaceT = Aws::String>
  void SetNamespace(NamespaceT&& value) {
    m_namespaceHasBeenSet = true;
    m_namespace = std::forward<NamespaceT>(value);
  }
  template <typename NamespaceT = Aws::String>
  CreateTableRequest& WithNamespace(NamespaceT&& value) {
    SetNamespace(std::forward<NamespaceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name for the table.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateTableRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The format for the table.</p>
   */
  inline OpenTableFormat GetFormat() const { return m_format; }
  inline bool FormatHasBeenSet() const { return m_formatHasBeenSet; }
  inline void SetFormat(OpenTableFormat value) {
    m_formatHasBeenSet = true;
    m_format = value;
  }
  inline CreateTableRequest& WithFormat(OpenTableFormat value) {
    SetFormat(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The metadata for the table.</p>
   */
  inline const TableMetadata& GetMetadata() const { return m_metadata; }
  inline bool MetadataHasBeenSet() const { return m_metadataHasBeenSet; }
  template <typename MetadataT = TableMetadata>
  void SetMetadata(MetadataT&& value) {
    m_metadataHasBeenSet = true;
    m_metadata = std::forward<MetadataT>(value);
  }
  template <typename MetadataT = TableMetadata>
  CreateTableRequest& WithMetadata(MetadataT&& value) {
    SetMetadata(std::forward<MetadataT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The encryption configuration to use for the table. This configuration
   * specifies the encryption algorithm and, if using SSE-KMS, the KMS key to use for
   * encrypting the table. </p>  <p>If you choose SSE-KMS encryption you must
   * grant the S3 Tables maintenance principal access to your KMS key. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-kms-permissions.html">Permissions
   * requirements for S3 Tables SSE-KMS encryption</a>.</p>
   */
  inline const EncryptionConfiguration& GetEncryptionConfiguration() const { return m_encryptionConfiguration; }
  inline bool EncryptionConfigurationHasBeenSet() const { return m_encryptionConfigurationHasBeenSet; }
  template <typename EncryptionConfigurationT = EncryptionConfiguration>
  void SetEncryptionConfiguration(EncryptionConfigurationT&& value) {
    m_encryptionConfigurationHasBeenSet = true;
    m_encryptionConfiguration = std::forward<EncryptionConfigurationT>(value);
  }
  template <typename EncryptionConfigurationT = EncryptionConfiguration>
  CreateTableRequest& WithEncryptionConfiguration(EncryptionConfigurationT&& value) {
    SetEncryptionConfiguration(std::forward<EncryptionConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The storage class configuration for the table. If not specified, the table
   * inherits the storage class configuration from its table bucket. Specify this
   * parameter to override the bucket's default storage class for this table.</p>
   */
  inline const StorageClassConfiguration& GetStorageClassConfiguration() const { return m_storageClassConfiguration; }
  inline bool StorageClassConfigurationHasBeenSet() const { return m_storageClassConfigurationHasBeenSet; }
  template <typename StorageClassConfigurationT = StorageClassConfiguration>
  void SetStorageClassConfiguration(StorageClassConfigurationT&& value) {
    m_storageClassConfigurationHasBeenSet = true;
    m_storageClassConfiguration = std::forward<StorageClassConfigurationT>(value);
  }
  template <typename StorageClassConfigurationT = StorageClassConfiguration>
  CreateTableRequest& WithStorageClassConfiguration(StorageClassConfigurationT&& value) {
    SetStorageClassConfiguration(std::forward<StorageClassConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A map of user-defined tags that you would like to apply to the table that you
   * are creating. A tag is a key-value pair that you apply to your resources. Tags
   * can help you organize, track costs for, and control access to resources. For
   * more information, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html">Tagging
   * for cost allocation or attribute-based access control (ABAC)</a>.</p>
   * <p>You must have the <code>s3tables:TagResource</code> permission in addition to
   * <code>s3tables:CreateTable</code> permission to create a table with tags.</p>
   *
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateTableRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateTableRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_tableBucketARN;

  Aws::String m_namespace;

  Aws::String m_name;

  OpenTableFormat m_format{OpenTableFormat::NOT_SET};

  TableMetadata m_metadata;

  EncryptionConfiguration m_encryptionConfiguration;

  StorageClassConfiguration m_storageClassConfiguration;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tableBucketARNHasBeenSet = false;
  bool m_namespaceHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_formatHasBeenSet = false;
  bool m_metadataHasBeenSet = false;
  bool m_encryptionConfigurationHasBeenSet = false;
  bool m_storageClassConfigurationHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3Tables
}  // namespace Aws
